package csbase.logic.algorithms.parameters;

import java.rmi.RemoteException;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import csbase.exception.ParseException;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.CommandLineParameterFormatter;
import csbase.logic.algorithms.parameters.validators.SimpleParameterValidator;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;

/**
 * Parmetro simples.
 * 
 * @param <V> O tipo do valor.
 * 
 * @author lmoreira
 */
public abstract class SimpleParameter<V> extends Parameter<V> {
  /**
   * O padro da linha de comando.
   */
  private String commandLinePattern;

  /**
   * Valor-padro.
   */
  private V defaultValue;

  /**
   * Descrio.
   */
  private String description;

  /**
   * Indica se o parmetro est habilitado ou no.
   */
  private boolean isEnabled;

  /**
   * Indica se o valor do parmetro  opcional ou no.
   */
  private boolean isOptional;

  /**
   * Indica se o parmetro est visvel ou no.
   */
  private boolean isVisible;

  /**
   * Rtulo.
   */
  private String label;

  /**
   * Observadores.
   */
  private List<SimpleParameterListener<V>> listeners;

  /**
   * Valor corrente.
   */
  private V value;

  /**
   * Indica se o parmetro deve ser ignorado (no deve ser adicionado na linha
   * de comando), caso esteja oculto na interface grfica.
   */
  private boolean ignoreIfInvisible;

  /**
   * Indica se o parmetro deve ser ignorado (no deve ser adicionado na linha
   * de comando), caso esteja desabilitado na interface grfica.
   */
  private boolean ignoreIfDisabled;

  /**
   * O validador do parmetro.
   */
  private SimpleParameterValidator<V> parameterValidator;

  /**
   * Cria o parmetro.
   * 
   * @param name O nome (No aceita {@code null}).
   * @param label O rtulo (No aceita {@code null}).
   * @param description A descrio (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se o valor do parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   */
  protected SimpleParameter(String name, String label, String description,
    V defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern) {
    super(name);
    this.listeners = new LinkedList<SimpleParameterListener<V>>();
    this.defaultValue = defaultValue;
    this.isOptional = isOptional;
    this.commandLinePattern = commandLinePattern;
    this.ignoreIfDisabled = false;
    this.ignoreIfInvisible = false;
    setEnabled(true);
    setVisible(isVisible);
    setValue(getDefaultValue());
    setLabel(label);
    setDescription(description);
  }

  /**
   * Adiciona um observador.
   * 
   * @param listener O observador (No aceita {@code null}).
   */
  public final void addSimpleParameterListener(
    SimpleParameterListener<V> listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    this.listeners.add(listener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, Object> exportValue() {
    Map<String, Object> parameterValue = new HashMap<String, Object>();
    parameterValue.put(getName(), getValue());
    return Collections.unmodifiableMap(parameterValue);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String getCommandLine(CommandLineContext context) {
    if (this.commandLinePattern == null) {
      return null;
    }

    if (!this.isVisible && this.ignoreIfInvisible) {
      return null;
    }

    if (!this.isEnabled && this.ignoreIfDisabled) {
      return null;
    }

    String commandValue = getCommandValue(context);
    return CommandLineParameterFormatter.formatCommandLine(getName(),
      commandValue, getCommandLinePattern());
  }

  /**
   * Obtm o padro da linha de comando.
   * 
   * @return .
   */
  public String getCommandLinePattern() {
    return this.commandLinePattern;
  }

  /**
   * Obtm o valor-padro.
   * 
   * @return O valor-padro ou {@code null} se ele no existir.
   */
  public V getDefaultValue() {
    return this.defaultValue;
  }

  /**
   * Obtm a descrio.
   * 
   * @return A descrio.
   */
  public final String getDescription() {
    return this.description;
  }

  /**
   * Obtm o valor corrente convertido para um valor que pode ser utilizado em
   * expresses ({@link csbase.logic.algorithms.parameters.Expression}).
   * 
   * @return O valor corrente ou {@code null} se no tiver valor corrente ou se
   *         este parmetro no pode participar de expresses.
   */
  public abstract Object getExpressionValue();

  /**
   * Obtm o rtulo.
   * 
   * @return O rtulo.
   */
  public final String getLabel() {
    return this.label;
  }

  /**
   * <p>
   * Obtm os observadores.
   * </p>
   * 
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   * 
   * @return A lista de observadores (a lista estar vazia se no houver
   *         observadores).
   */
  public final List<SimpleParameterListener<V>> getSimpleParameterListeners() {
    return Collections.unmodifiableList(this.listeners);
  }

  /**
   * Obtm um texto que representa o tipo deste parmetros.
   * 
   * @return O tipo.
   */
  public abstract String getType();

  /**
   * Obtm o identificador do tipo IDL do parmetro se ele possuium uma
   * interface IDL.
   * 
   * @return O identificador do tipo IDL ou {@code null} se no possuir
   *         interface IDL.
   */
  public String getIDLType() {
    return null;
  }

  /**
   * Obtm o valor corrente.
   * 
   * @return O valor corrente ou {@code null} se ele no existir.
   */
  public final V getValue() {
    return this.value;
  }

  /**
   * <p>
   * Obtm o valor corrente sob a forma de string.
   * </p>
   * 
   * <p>
   * Se mtodo  completar ao mtodo {@link #setValueAsText(String)}.
   * </p>
   * 
   * @return O valor corrente sob a forma de string ou {@code null} se ele no
   *         existir.
   */
  public String getValueAsText() {
    if (getValue() == null) {
      return null;
    }
    return getValue().toString();
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings("unchecked")
  @Override
  public void importValue(Map<String, Object> parameterValues) {
    if (parameterValues == null) {
      throw new IllegalArgumentException(
        "O parmetro parameterValues est nulo.");
    }
    setValue((V) parameterValues.get(getName()));
  }

  /**
   * Indica se este parmetro est habilitado ou no.
   * 
   * @return {@code true} se o parmetro estive habilitado ou {@code false} caso
   *         contrrio.
   */
  public final boolean isEnabled() {
    return this.isEnabled;
  }

  /**
   * <p>
   * Indica se o parmetro  opcional ou obrigatrio.
   * </p>
   * 
   * <p>
   * Um parmetro  obrigatrio se ele precisa ter um valor corrente para estar
   * vlido.
   * </p>
   * 
   * @return {@code true} - opcional ou {@code false} - obrigatrio.
   */
  public final boolean isOptional() {
    return this.isOptional;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isSetDefaultValue() {
    Object val = getValue();
    Object defaultVal = getDefaultValue();
    if (val == null) {
      if (defaultVal == null) {
        return true;
      }
      return false;
    }
    return val.equals(defaultVal);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isVisible() {
    return this.isVisible;
  }

  /**
   * Remove um observador.
   * 
   * @param listener O observador (No aceita {@code null}).
   * 
   * @return {@code true} se puder remover o listener ou {@code false} se o
   *         observador no existia.
   */
  public final boolean removeSimpleParameterListener(
    SimpleParameterListener<V> listener) {
    if (listener == null) {
      throw new IllegalArgumentException("listener == null");
    }
    return this.listeners.remove(listener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void resetValue() {
    setValue(getDefaultValue());
  }

  /**
   * <p>
   * Atribui um valor-padro a este parmetro.
   * </p>
   * 
   * <p>
   * Se o valor-padro for modificado, ele ir gerar o evento
   * {@link SimpleParameterListener#defaultValueWasChanged(SimpleParameter)}.
   * </p>
   * 
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * 
   * @return {@code true} se ele for modificado ou {@code false} se o
   *         valor-padro atual for igual ao valor fornecido.
   */
  public final boolean setDefaultValue(V defaultValue) {
    if (this.defaultValue == null && defaultValue == null) {
      return false;
    }
    if (this.defaultValue != null) {
      if (this.defaultValue.equals(defaultValue)) {
        return false;
      }
    }
    this.defaultValue = defaultValue;
    fireDefaultValueWasChangedEvent();
    return true;
  }

  /**
   * <p>
   * Habilita/desabilita este parmetro.
   * </p>
   * 
   * <p>
   * Se o estado mudar, ele ir gerar o evento
   * {@link SimpleParameterListener#capabilityWasChanged(SimpleParameter)}
   * </p>
   * 
   * @param isEnabled {@code true} para habilitar ou {@code false} para
   *        desabilitar.
   * 
   * @return {@code true} se houver mudana de estado ou {@code false} se o
   *         estado atual foi igual ao estado esperado (exemplo: habilitar um
   *         parmetro j habilitado).
   */
  public final boolean setEnabled(boolean isEnabled) {
    if (this.isEnabled == isEnabled) {
      return false;
    }
    this.isEnabled = isEnabled;
    fireCapabilityWasChangedEvent();
    return true;
  }

  /**
   * <p>
   * Modifica o rtulo de um parmetro
   * </p>
   * .
   * 
   * <p>
   * Se o valor corrente for modificado, ele ir gerar o evento
   * {@link SimpleParameterListener#labelWasChanged(SimpleParameter)}.
   * </p>
   * 
   * @param label O rtulo (No aceita {@code null}).
   * 
   * @return {@code true} se ele for modificado ou {@code false} se o rtulo
   *         corrente for igual ao rtulo fornecido.
   */
  public final boolean setLabel(String label) {
    if (label == null) {
      throw new IllegalArgumentException("label == null");
    }
    if (label.equals(this.label)) {
      return false;
    }
    this.label = label;
    fireLabelWasChangedEvent();
    return true;
  }

  /**
   * <p>
   * Atribui um valor corrente a este parmetro.
   * </p>
   * 
   * <p>
   * Se o valor corrente for modificado, ele ir gerar o evento
   * {@link SimpleParameterListener#valueWasChanged(SimpleParameter)}.
   * </p>
   * 
   * @param value O valor (Aceita {@code null}).
   * 
   * @return {@code true} se ele for modificado ou {@code false} se o valor
   *         corrente for igual ao valor fornecido.
   */
  public boolean setValue(V value) {
    if (this.value == null && value == null) {
      return false;
    }
    if (this.value != null) {
      if (this.value.equals(value)) {
        return false;
      }
    }
    this.value = value;
    fireValueWasChangedEvent();
    return true;
  }

  /**
   * <p>
   * Atribui o valor corrente sob a forma de string.
   * </p>
   * 
   * <p>
   * Se mtodo  completar ao mtodo {@link #getValueAsText()}.
   * </p>
   * 
   * @param parameterValue O valor corrente sob a forma de string (Aceita
   *        {@code null}).
   * 
   * @throws ParseException Se o valor passado no estiver em um formato aceito
   *         por este parmetro.
   */
  public abstract void setValueAsText(String parameterValue)
    throws ParseException;

  /**
   * <p>
   * Exibe/oculta este parmetro.
   * </p>
   * 
   * <p>
   * Se o estado mudar, ele ir gerar o evento
   * {@link SimpleParameterListener#visibilityWasChanged(SimpleParameter)}
   * </p>
   * 
   * @param isVisible {@code true} para exibir ou {@code false} para ocultar.
   * 
   * @return {@code true} se houver mudana de estado ou {@code false} se o
   *         estado atual foi igual ao estado esperado (exemplo: exibir um
   *         parmetro j visvel).
   */
  @Override
  public final boolean setVisible(boolean isVisible) {
    if (this.isVisible == isVisible) {
      return false;
    }
    this.isVisible = isVisible;
    fireVisiblityWasChangedEvent();
    return true;
  }

  /**
   * Dispara o evento
   * {@link SimpleParameterListener#valueWasChanged(SimpleParameter)}.
   */
  protected void fireValueWasChangedEvent() {
    for (SimpleParameterListener<V> listener : this.listeners) {
      listener.valueWasChanged(this);
    }
  }

  /**
   * Dispara o evento
   * {@link SimpleParameterListener#visibilityWasChanged(SimpleParameter)}.
   */
  protected final void fireVisiblityWasChangedEvent() {
    for (SimpleParameterListener<V> listener : this.listeners) {
      listener.visibilityWasChanged(this);
    }
  }

  /**
   * Obtm o valor corrente convertido para um valor que possa ser utilizado na
   * linha de comando.
   * 
   * @param context Contexto para gerao da linha de comando.
   * 
   * @return O valor ou {@code null} se o valor for nulo.
   */
  protected String getCommandValue(CommandLineContext context) {
    V val = getValue();
    if (val == null) {
      return null;
    }
    return val.toString();
  }

  /**
   * Dispara o evento
   * {@link SimpleParameterListener#capabilityWasChanged(SimpleParameter)}.
   */
  private void fireCapabilityWasChangedEvent() {
    for (SimpleParameterListener<V> listener : this.listeners) {
      listener.capabilityWasChanged(this);
    }
  }

  /**
   * Dispara o evento
   * {@link SimpleParameterListener#defaultValueWasChanged(SimpleParameter)}.
   */
  private void fireDefaultValueWasChangedEvent() {
    for (SimpleParameterListener<V> listener : this.listeners) {
      listener.defaultValueWasChanged(this);
    }
  }

  /**
   * Dispara o evento
   * {@link SimpleParameterListener#labelWasChanged(SimpleParameter)}.
   */
  private void fireLabelWasChangedEvent() {
    for (SimpleParameterListener<V> listener : this.listeners) {
      listener.labelWasChanged(this);
    }
  }

  /**
   * Atribui uma descrio.
   * 
   * @param description A descrio (No aceita {@code null}).
   */
  private void setDescription(String description) {
    if (description == null) {
      throw new IllegalArgumentException("description == null");
    }
    this.description = description;
  }

  /**
   * Retorna verdadeiro se o parmetro deve ser ignorado se estiver invisvel ou
   * falso, caso contrrio.
   * 
   * @return ignoreIfInvisible verdadeiro se o parmetro deve ser ignorado se
   *         estiver invisvel ou falso, caso contrrio.
   */
  public boolean ignoreIfInvisible() {
    return ignoreIfInvisible;
  }

  /**
   * Determina se o parmetro deve ser ignorado caso esteja invisvel.
   * 
   * @param ignoreIfInvisible verdadeiro se o parmetro deve ser ignorado se
   *        estiver invisvel ou falso, caso contrrio.
   */
  public void setIgnoreIfInvisible(boolean ignoreIfInvisible) {
    this.ignoreIfInvisible = ignoreIfInvisible;
  }

  /**
   * Retorna verdadeiro se o parmetro deve ser ignorado se estiver desabilitado
   * ou falso, caso contrrio.
   * 
   * @return ignoreIfDisabled verdadeiro se o parmetro deve ser ignorado se
   *         estiver desabilitado ou falso, caso contrrio.
   */
  public boolean ignoreIfDisabled() {
    return ignoreIfDisabled;
  }

  /**
   * Determina se o parmetro deve ser ignorado caso esteja desabilitado.
   * 
   * @param ignoreIfDisabled verdadeiro se o parmetro deve ser ignorado se
   *        estiver desabilitado ou falso, caso contrrio.
   */
  public void setIgnoreIfDisabled(boolean ignoreIfDisabled) {
    this.ignoreIfDisabled = ignoreIfDisabled;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final Validation validate(ValidationContext context)
    throws RemoteException {
    SimpleParameterValidator<V> validator = getParameterValidator();
    validator.setMode(context.getMode());
    Validation result = validator.validateValue(this, getValue(), context);
    return result;
  }

  /**
   * Obtm o validador do parmetro.
   * 
   * @return O validador.
   */
  public SimpleParameterValidator<V> getParameterValidator() {
    if (this.parameterValidator == null) {
      this.parameterValidator = createParameterValidator();
    }
    return this.parameterValidator;
  }

  /**
   * Cria o validador do parmetro.
   * 
   * @return O validador.
   */
  public abstract SimpleParameterValidator<V> createParameterValidator();
}
