/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.logic.algorithms.parameters;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import csbase.exception.ParseException;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.parameters.validators.SimpleParameterValidator;

/**
 * Coluna.
 *
 * @param <V> O tipo do valor armazenado na coluna.
 *
 * @author lmoreira
 */
public abstract class TableColumn<V> implements Serializable {
  /**
   * O valor-padro para as clulas.
   */
  private V defaultValue;

  /**
   * Os valores-padro para as clulas indexados pelo ndice da linha.
   */
  private Map<Integer, V> defaultValuesByRowIndex;

  /**
   * ndices das linhas que possuem valor-padro.
   */
  private Set<Integer> rowIndexesWithDefaultValue;

  /**
   * O identificador.
   */
  private String id;

  /**
   * Indica se a coluna  editvel.
   */
  private boolean isEditable;

  /**
   * Indica se o valor da clula  opcional.
   */
  private boolean isOptional;

  /**
   * O rtulo.
   */
  private String label;

  /**
   * Cria a coluna.
   *
   * @param id O identificador (Aceita {@code null}).
   * @param label O rtulo (No aceita {@code null}).
   * @param defaultValue O valor-padro geral (Aceita {@code null}).
   * @param isOptional Indica se o valor da clula  opcional.
   * @param isEditable Indica se a coluna  editvel.
   */
  protected TableColumn(String id, String label, V defaultValue,
    boolean isOptional, boolean isEditable) {
    setId(id);
    setLabel(label);
    this.defaultValuesByRowIndex = new HashMap<Integer, V>();
    this.rowIndexesWithDefaultValue = new HashSet<Integer>();
    this.defaultValue = defaultValue;
    this.isOptional = isOptional;
    this.isEditable = isEditable;
  }

  /**
   * Adiciona um valor-padro especfico.
   *
   * @param rowIndex O ndice da linha (No pode ser negativo).
   * @param value O valor-padro (Aceita {@code null}).
   *
   * @return {@code true} em caso de sucesso ou {@code false} se j existir um
   *         valor-padro para esta linha.
   */
  public final boolean addDefaultValue(int rowIndex, V value) {
    if (rowIndex < 0) {
      throw new IllegalArgumentException("ndice de linha negativo -> "
        + rowIndex);
    }
    if (!rowIndexesWithDefaultValue.add(rowIndex)) {
      return false;
    }
    this.defaultValuesByRowIndex.put(rowIndex, value);
    return true;
  }

  /**
   * Remove todos os valores-padro especficos.
   */
  public final void clearDefaultValues() {
    rowIndexesWithDefaultValue.clear();
    defaultValuesByRowIndex.clear();
  }

  /**
   * Obtm o validador para os valores desta coluna.
   *
   * @return .
   */
  public abstract SimpleParameterValidator<V> getValidator();

  //TODO Trocar o nome do mtodo.
  /**
   * Decodifica o valor de uma clula desta coluna.
   *
   * @param itemValue O valor codificado (Aceita {@code null}).
   *
   * @return O valor.
   *
   * @throws ParseException Se o valor codificado no representar um valor
   *         vlido.
   */
  public abstract V getItemValueFromText(String itemValue)
    throws ParseException;

  /**
   * Obtm o valor da linha de comando para um valor de clula.
   *
   * @param itemValue O valor da clula (Aceita {@code null}).
   * @param context O contexto (No aceita {@code null}).
   *
   * @return O valor do comando.
   */
  public abstract String getCommandValue(V itemValue,
    CommandLineContext context);

  //TODO Trocar o nome do mtodo.
  /**
   * Codifica o valor de uma clula desta coluna.
   *
   * @param itemValue O valor da clula (Aceita {@code null}).
   *
   * @return O valor codificado.
   */
  public abstract String getItemValueAsText(V itemValue);

  /**
   * Obtm o valor exportvel para o valor armazenado na clula.
   *
   * @param itemValue O valor da clula (Aceita {@code null}).
   *
   * @return .
   */
  public abstract Object getValueToExport(V itemValue);

  /**
   * Obtm o valor armazenvel na clula do valor importado.
   *
   * @param importedValue O valor importado (Aceita {@code null}).
   *
   * @return .
   */
  public abstract V getValueToImport(Object importedValue);

  /**
   * Duas colunas so consideradas iguais se elas estiverem no mesmo rtulo.
   */
  @Override
  public final boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!getClass().equals(obj.getClass())) {
      return false;
    }
    TableColumn<?> column = (TableColumn<?>) obj;
    return getId().equals(column.getId());
  }

  /**
   * Obtm o valor-padro para uma clula.
   *
   * @param rowIndex O ndice da linha (No pode ser negativo).
   *
   * @return O valor-padro da clula.
   */
  public final V getDefaultValue(int rowIndex) {
    if (rowIndex < 0) {
      throw new IllegalArgumentException("ndice da linha negativo -> "
        + rowIndex);
    }
    if (this.rowIndexesWithDefaultValue.contains(rowIndex)) {
      return this.defaultValuesByRowIndex.get(rowIndex);
    }
    return this.defaultValue;
  }

  /**
   * Obtm o valor-padro geral.
   *
   * @return O valor-padro geral ou {@code null} se ele no existir.
   */
  public final V getDefaultValue() {
    return this.defaultValue;
  }

  /**
   * Obtm o identificador.
   *
   * @return .
   */
  public final String getId() {
    return this.id;
  }

  /**
   * Obtm o rtulo.
   *
   * @return O rtulo.
   */
  public final String getLabel() {
    return this.label;
  }

  /**
   * <p>
   * Obtm os ndices das linhas que possuem valor-padro especfico.
   * </p>
   *
   * <p>
   * O conjunto retornado  imutvel (veja
   * {@link Collections#unmodifiableSet(Set)}).
   * </p>
   *
   * @return Os ndices (Se no houver ndices de linhas, o conjunto estar
   *         vazio).
   */
  public final Set<Integer> getRowIndexes() {
    return Collections.unmodifiableSet(this.rowIndexesWithDefaultValue);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final int hashCode() {
    return getId().hashCode();
  }

  /**
   * Indica se a coluna  editvel.
   *
   * @return .
   */
  public final boolean isEditable() {
    return this.isEditable;
  }

  /**
   * Indica se o valor da clula  opcional.
   *
   * @return .
   */
  public final boolean isOptional() {
    return this.isOptional;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String toString() {
    return getLabel();
  }

  /**
   * Atribui o identificador a esta coluna.
   *
   * @param id O identificador (No aceita {@code null}).
   */
  private void setId(String id) {
    if (id == null) {
      throw new IllegalArgumentException("O parmetro id est nulo.");
    }
    this.id = id;
  }

  /**
   * Atribui o rtulo a esta coluna.
   *
   * @param label O rtulo (No aceita {@code null}).
   */
  private void setLabel(String label) {
    if (label == null) {
      throw new IllegalArgumentException("O parmetro label est nulo.");
    }
    this.label = label;
  }
}
