package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import tecgraf.javautils.parsers.FiniteAutomaton;
import tecgraf.javautils.parsers.Parser;
import tecgraf.javautils.parsers.State;
import tecgraf.javautils.parsers.Token;
import tecgraf.javautils.parsers.actions.AppendAction;
import tecgraf.javautils.parsers.actions.CompositeTransitionAction;
import tecgraf.javautils.parsers.actions.DiscardAction;
import tecgraf.javautils.parsers.actions.GenerateTokenAction;
import tecgraf.javautils.parsers.exception.AutomatonException;
import tecgraf.javautils.parsers.iterators.CharSymbolIterator;
import tecgraf.javautils.parsers.symbols.CharSymbol;
import tecgraf.javautils.parsers.symbols.Symbol;

/**
 * Analisador do texto que representa o valor de um parmetro do tipo tabela.
 */
final class TableParameterParser extends Parser {

  /**
   * Caractere que define o incio de uma linha/tabela.
   */
  static final char START_CHARACTER = '{';
  /**
   * Caractere que define o fim de uma linha/tabela.
   */
  static final char END_CHARACTER = '}';

  /**
   * Caractere que separa os elementos/linhas de uma tabela.
   */
  static final char SEPARATOR_CHARACTER = ',';

  /**
   * O analisador de valores de linhas da tabela.
   */
  private RowParameterParser elementGeneratorParser;

  /**
   * Construtor.
   */
  TableParameterParser() {
    super(new TableParameterFiniteAutomaton());
    this.elementGeneratorParser = new RowParameterParser();
  }

  /**
   * Analisa o contedo do leitor e retorna a tabela com os elementos lidos.
   * 
   * @param reader o leitor a ser analisado.
   * @return list a tabela representada como uma lista de listas de valores.
   * 
   * @throws IOException Em caso de erro de I/O.
   * @throws AutomatonException em caso de falha no autmato do analisador.
   */
  List<List<String>> parseText(Reader reader) throws AutomatonException,
    IOException {
    StringBuilder buffer = new StringBuilder();
    int current;
    while ((current = reader.read()) != -1) {
      buffer.append((char) current);
    }
    String text = buffer.toString();
    return parseText(text);
  }

  /**
   * Analisa o contedo da string e retorna a tabela com os elementos lidos.
   * 
   * @param text o texto a ser analisado.
   * @return list a tabela representada como uma lista de listas de valores.
   * 
   * @throws AutomatonException em caso de falha no autmato do analisador.
   */
  List<List<String>> parseText(String text) throws AutomatonException {
    List<List<String>> rows = new LinkedList<List<String>>();
    List<Token> tokens = parse(new CharSymbolIterator(text));
    for (Token token : tokens) {
      StringBuffer buffer = new StringBuffer();
      List<Symbol<?>> symbols = token.getSymbolList();
      for (Symbol<?> symbol : symbols) {
        buffer.append(symbol.getObject());
      }
      List<String> row =
        this.elementGeneratorParser.parseText(buffer.toString());
      rows.add(row);
    }
    return rows;
  }

  /**
   * Analisador de valores de linhas de uma tabela.
   */
  private static final class RowParameterParser extends Parser {

    /**
     * Construtor.
     */
    RowParameterParser() {
      super(new RowFiniteAutomaton());
    }

    /**
     * Analisa o contedo da string e retorna a linha com os elementos lidos.
     * 
     * @param text o texto a ser analisado.
     * @return list a linha representada como uma lista de valores.
     * 
     * @throws AutomatonException em caso de falha no autmato do analisador.
     */
    List<String> parseText(String text) throws AutomatonException {
      List<Token> tokens = parse(new CharSymbolIterator(text));
      List<String> row = new ArrayList<String>(tokens.size());
      for (Token token : tokens) {
        StringBuffer buffer = new StringBuffer();
        List<Symbol<?>> symbols = token.getSymbolList();
        for (Symbol<?> symbol : symbols) {
          Character object = (Character) symbol.getObject();
          buffer.append(object);
        }
        String cellValue;
        if (buffer.length() != 0) {
          cellValue = buffer.toString();
        }
        else {
          cellValue = null;
        }
        row.add(cellValue);
      }
      return row;
    }

    /**
     * O autmato do analisador de linhas da tabela.
     */
    private static final class RowFiniteAutomaton extends FiniteAutomaton {

      /** O smbolo que representa o incio de uma linha */
      private static final CharSymbol START_SYMBOL = new CharSymbol(
        START_CHARACTER);

      /** O smbolo que representa o separador de elementos da linha */
      private static final CharSymbol SEPARATOR_SYMBOL = new CharSymbol(
        SEPARATOR_CHARACTER);

      /** O smbolo que representa o fim de uma linha */
      private static final CharSymbol END_SYMBOL =
        new CharSymbol(END_CHARACTER);

      /**
       * Construtor.
       */
      RowFiniteAutomaton() {
        super(new State(false));
        State startLineState = getInitialState();
        State elementState = new State(false);
        State endLineState = new State(true);
        startLineState.addTransition(START_SYMBOL, DiscardAction.getInstance(),
          elementState);
        elementState.addTransition(SEPARATOR_SYMBOL, GenerateTokenAction
          .getInstance(), elementState);
        elementState.addTransition(END_SYMBOL, new CompositeTransitionAction(
          GenerateTokenAction.getInstance(), DiscardAction.getInstance()),
          endLineState);
        elementState.setDefaultTransition(AppendAction.getInstance(),
          elementState);
      }
    }
  }

  /**
   * O autmato do analisador da tabela.
   */
  private static final class TableParameterFiniteAutomaton extends
    FiniteAutomaton {

    /** O smbolo que representa o incio de uma tabela */
    private static final CharSymbol START_SYMBOL = new CharSymbol(
      START_CHARACTER);

    /** O smbolo que representa o separador de linhas da tabela */
    private static final CharSymbol SEPARATOR_SYMBOL = new CharSymbol(
      SEPARATOR_CHARACTER);

    /** O smbolo que representa o incio de uma tabela */
    private static final CharSymbol END_SYMBOL = new CharSymbol(END_CHARACTER);

    /**
     * Construtor.
     */
    private TableParameterFiniteAutomaton() {
      super(new State(false));
      State startTable = getInitialState();
      State startLine = new State(false);
      State buildingElement = new State(false);
      State endLine = new State(false);
      State endTable = new State(true);
      startTable.addTransition(START_SYMBOL, DiscardAction.getInstance(),
        startLine);
      startLine.addTransition(START_SYMBOL, AppendAction.getInstance(),
        buildingElement);
      startLine
        .addTransition(END_SYMBOL, DiscardAction.getInstance(), endTable);
      buildingElement.setDefaultTransition(AppendAction.getInstance(),
        buildingElement);
      CompositeTransitionAction compositeAction =
        new CompositeTransitionAction();
      compositeAction.addAction(AppendAction.getInstance());
      compositeAction.addAction(GenerateTokenAction.getInstance());
      buildingElement.addTransition(END_SYMBOL, compositeAction, endLine);
      endLine.addTransition(SEPARATOR_SYMBOL, DiscardAction.getInstance(),
        startLine);
      endLine.addTransition(END_SYMBOL, DiscardAction.getInstance(), endTable);
    }
  }
}
