package csbase.logic.algorithms.parameters.validators;

import java.rmi.RemoteException;
import java.util.List;

import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.URLListParameter;
import csbase.logic.algorithms.parameters.URLProtocol;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationError;
import csbase.logic.algorithms.validation.ValidationMode;
import csbase.logic.algorithms.validation.ValidationSuccess;

/**
 * Validador de parmetros do tipo lista de URLs.
 * 
 * @author lmoreira
 */
public final class URLListParameterValidator extends
  ListParameterValidator<FileURLValue> {

  /**
   * Cria um validador de uma lista de arquivos.
   * 
   * @param urlValidator o validor de um arquivo individual.
   */
  public URLListParameterValidator(URLParameterValidator urlValidator) {
    super(urlValidator);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final Validation validateValue(SimpleParameter<?> parameter,
    List<FileURLValue> values, ValidationContext context) throws RemoteException {
    if (!parameter.isVisible() || !parameter.isEnabled()) {
      return new ValidationSuccess();
    }
    if ((values == null) || values.isEmpty()) {
      if (isOptional() || getMode().equals(ValidationMode.ALLOW_EMPY_VALUES)) {
        return new ValidationSuccess();
      }
      LocalizedMessage message =
        new LocalizedMessage(ListParameterValidator.class, "empty_value",
          new Object[] { parameter.getLabel() });
      return new ValidationError(message);
    }
    Validation result;
    URLListParameter urlListParameter = (URLListParameter) parameter;
    URLProtocol protocol = values.iterator().next().getProtocol();
    for (FileURLValue url : values) {
      if (protocol != url.getProtocol()) {
        LocalizedMessage message =
          new LocalizedMessage(URLListParameterValidator.class,
            "multiple_protocol", new Object[] { parameter.getLabel() });
        return new ValidationError(message);
      }
      URLParameterValidator validator = getItemValidator();
      result =
        validator.validateURL(url, urlListParameter.getAllowedProtocols(),
          urlListParameter.mustExist(), urlListParameter.getMode(), context);
      if (!result.isWellSucceded()) {
        return result;
      }
    }
    return new ValidationSuccess();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public URLParameterValidator getItemValidator() {
    return (URLParameterValidator) super.getItemValidator();
  }

}
