package csbase.logic.algorithms.flows;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.FileURLValue;

/**
 * <p>
 * O n de um fluxo de algoritmos.
 * </p>
 *
 * @author lmoreira
 */
public final class FlowNode implements Serializable, Cloneable {
  /**
   * O configurador de algoritmos.
   */
  private transient AlgorithmConfigurator algorithmConfigurator;

  /**
   * O nome do algoritmo.
   */
  private String algorithmName;

  /**
   * O identificador da verso do algoritmo.
   */
  private AlgorithmVersionId algorithmVersionId;

  /**
   * A altura (em pixels).
   */
  private int height;

  /**
   * O identificador do n.
   */
  private int id;

  /**
   * O conjunto de parmetros.
   */
  private Set<NodeParameter> parameters;

  /**
   * A largura (em pixels).
   */
  private int width;

  /**
   * A abscissa esquerda (em pixels).
   */
  private int x;

  /**
   * A ordenada superior (em pixels).
   */
  private int y;

  /**
   * O arquivo de sada (stdout) de um algoritmo sendo executado.
   */
  private FileURLValue standardOutputFile;

  /**
   * O arquivo que armazena o cdigo de sada de um algoritmo executado.
   */
  private FileURLValue exitCodeLogFile;

  /**
   * Indica se o n prov um cdigo de sada durante sua execuo.
   */
  private boolean hasExitCode;

  /**
   * Sinaliza se o n deve ser desviado durante a execuo. O desvio
   * propriamente dito s pode ser feito se um conjunto de condies forem
   * atendidas
   *
   * see {@link FlowAlgorithmConfigurator}.
   */
  private boolean bypassed;

  /**
   * Cria um n.
   *
   * @param id O identificador.
   * @param algorithmConfigurator O configurator de algoritmos. (No aceita
   *        {@code null}).
   * @param parameters O conjunto de parmetros do algoritmo (No aceita
   *        {@code null}).
   * @param x A abscissa esquerda (em pixels).
   * @param y A ordenada superior (em pixels).
   * @param width A largura (em pixels) (Precisa ser positiva).
   * @param height A altura (em pixels) (Precisa ser positiva).
   */
  public FlowNode(int id, AlgorithmConfigurator algorithmConfigurator,
    Set<NodeParameter> parameters, int x, int y, int width, int height) {
    this(id, algorithmConfigurator, parameters, x, y, width, height, false);
  }

  /**
   * Cria um n.
   *
   * @param id O identificador.
   * @param algorithmName O nome do algoritmo (No aceita {@code null}).
   * @param algorithmVersionId O identificador da verso do algoritmo (No
   *        aceita {@code null}).
   * @param parameters O conjunto de parmetros do algoritmo (No aceita
   *        {@code null}).
   * @param x A abscissa esquerda (em pixels).
   * @param y A ordenada superior (em pixels).
   * @param width A largura (em pixels) (Precisa ser positiva).
   * @param height A altura (em pixels) (Precisa ser positiva).
   */
  public FlowNode(int id, String algorithmName,
    AlgorithmVersionId algorithmVersionId, Set<NodeParameter> parameters,
    int x, int y, int width, int height) {
    this(id, algorithmName, algorithmVersionId, parameters, x, y, width,
      height, false, false);
  }

  /**
   * Cria um n.
   *
   * @param id O identificador.
   * @param algorithmConfigurator O configurator de algoritmos. (No aceita
   *        {@code null}).
   * @param parameters O conjunto de parmetros do algoritmo (No aceita
   *        {@code null}).
   * @param x A abscissa esquerda (em pixels).
   * @param y A ordenada superior (em pixels).
   * @param width A largura (em pixels) (Precisa ser positiva).
   * @param height A altura (em pixels) (Precisa ser positiva).
   * @param bypassed O sinal que indica se o n deve ou no ser desviado durante
   *        a execuo do fluxo
   */
  public FlowNode(int id, AlgorithmConfigurator algorithmConfigurator,
    Set<NodeParameter> parameters, int x, int y, int width, int height,
    boolean bypassed) {
    this(id, algorithmConfigurator.getAlgorithmName(), algorithmConfigurator
      .getAlgorithmVersionId(), parameters, x, y, width, height, bypassed,
      algorithmConfigurator.hasExitCode());
    setAlgorithmConfigurator(algorithmConfigurator);
    this.standardOutputFile = algorithmConfigurator.getStandardOutputFile();
    this.exitCodeLogFile = algorithmConfigurator.getExitCodeLogFile();
  }

  /**
   * Cria um n.
   *
   * @param id O identificador.
   * @param algorithmName O nome do algoritmo (No aceita {@code null}).
   * @param algorithmVersionId O identificador da verso do algoritmo (No
   *        aceita {@code null}).
   * @param parameters O conjunto de parmetros do algoritmo (No aceita
   *        {@code null}).
   * @param x A abscissa esquerda (em pixels).
   * @param y A ordenada superior (em pixels).
   * @param width A largura (em pixels) (Precisa ser positiva).
   * @param height A altura (em pixels) (Precisa ser positiva).
   * @param bypassed O sinal que indica se o n deve ou no ser desviado durante
   *        a execuo do fluxo.
   * @param hasExitCode O sinal que indica se o n prov cdigo de sada.
   */
  public FlowNode(int id, String algorithmName,
    AlgorithmVersionId algorithmVersionId, Set<NodeParameter> parameters,
    int x, int y, int width, int height, boolean bypassed, boolean hasExitCode) {
    setId(id);
    setAlgorithmName(algorithmName);
    setAlgorithmVersionId(algorithmVersionId);
    setParameters(parameters);
    setX(x);
    setY(y);
    setHeight(height);
    setWidth(width);
    setBypassed(bypassed);
    setHasExitCode(hasExitCode);
  }

  /**
   * Atribui o valor do arquivo que deve receber a sada padro da execuo de
   * um algoritmo representado por este configurador.
   *
   * @param standardOutputFile
   */
  public void setStandardOutputFile(FileURLValue standardOutputFile) {
    this.standardOutputFile = standardOutputFile;
    // Garante consistncia.
    if (null != algorithmConfigurator) {
      algorithmConfigurator.setStandardOutputFile(standardOutputFile);
    }
  }

  /**
   * Obtm a sada padro que ser utilizada durante a execuo de um algortmo
   * representado por este configurador.
   *
   * @return a sada padro que ser utilizada durante a execuo de um
   *         algortmo representado por este configurador.
   */
  public FileURLValue getStandardOutputFile() {
    return this.standardOutputFile;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FlowNode clone() {
    try {
      FlowNode clone = (FlowNode) super.clone();
      clone.algorithmVersionId = this.algorithmVersionId.clone();
      if (null != standardOutputFile) {
        clone.standardOutputFile = (FileURLValue) standardOutputFile.clone();
      }
      clone.parameters = new HashSet<NodeParameter>();
      for (NodeParameter nodeParameter : this.parameters) {
        clone.parameters.add(nodeParameter.clone());
      }
      return clone;
    }
    catch (CloneNotSupportedException e) {
      throw new IllegalStateException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }
    final FlowNode other = (FlowNode) obj;
    if (this.id != other.id) {
      return false;
    }
    return true;
  }

  /**
   * Obtm o configurador de algoritmos.
   *
   * @return O configurator de algoritmos ou {@code null} se no h um
   *         configurador de algoritmos.
   */
  public AlgorithmConfigurator getAlgorithmConfigurator() {
    return this.algorithmConfigurator;
  }

  /**
   * Obtm o nome do algoritmo.
   *
   * @return O nome do algoritmo.
   */
  public String getAlgorithmName() {
    return this.algorithmName;
  }

  /**
   * Obtm o identificador da verso do algoritmo.
   *
   * @return O identificador da verso do algoritmo.
   */
  public AlgorithmVersionId getAlgorithmVersionId() {
    return this.algorithmVersionId;
  }

  /**
   * Obtm a altura (em pixels) deste n.
   *
   * @return A altura.
   */
  public int getHeight() {
    return this.height;
  }

  /**
   * Obtm o identificador.
   *
   * @return O identificador.
   */
  public int getId() {
    return this.id;
  }

  /**
   * Obtm um parmetro dado o nome do parmetro.
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   *
   * @return O parmetro ou {@code null} se ele no existir.
   */
  public NodeParameter getParameter(String parameterName) {
    if (parameterName == null) {
      throw new IllegalArgumentException("O parmetro parameterName est nulo.");
    }
    for (NodeParameter parameter : parameters) {
      if (parameter.getName().equals(parameterName)) {
        return parameter;
      }
    }
    return null;
  }

  /**
   * <p>
   * Obtm os parmetros deste n.
   * </p>
   *
   * <p>
   * O conjunto retornado  imutvel (veja
   * {@link Collections#unmodifiableSet(Set)}).
   * </p>
   *
   * @return O conjunto de parmetros ou um conjunto vazio se no houver
   *         parmetros.
   */
  public Set<NodeParameter> getParameters() {
    return Collections.unmodifiableSet(this.parameters);
  }

  /**
   * Obtm a largura (em pixels) deste n.
   *
   * @return A largura.
   */
  public int getWidth() {
    return this.width;
  }

  /**
   * Obtm a abscissa esquerda (em pixels) deste n.
   *
   * @return A abscissa esquerda.
   */
  public int getX() {
    return this.x;
  }

  /**
   * Obtm a ordenada superior (em pixels) deste n.
   *
   * @return A ordenada superior.
   */
  public int getY() {
    return this.y;
  }

  /**
   * Obtm o sinal que indica o desvio deste n.
   *
   * @return verdadeiro se o n estiver sendo desviado ou falso caso contrrio.
   */
  public boolean isBypassed() {
    return bypassed;
  }

  /**
   *
   * Obtm o sinal que indica se o cdigo de sada  retornado durante a
   * execuo do algoritmo.
   *
   * @return Verdadeiro se o n prov um cdigo de sada, ou falso, caso
   *         contrrio.
   */
  public boolean hasExitCode() {
    return hasExitCode && !isBypassed();
  }

  /**
   * Obtm o arquivo que armazena o cdigo de sada do n.
   *
   * @return o arquivo que armazena o cdigo de sada.
   */
  public FileURLValue getExitCodeLogFile() {
    if (isBypassed()) {
      return null;
    }
    return exitCodeLogFile;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return this.id;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder();
    builder.append("-> Identificador: " + getId() + "\n");
    builder.append("-> Algoritmo: " + getAlgorithmName() + "\n");
    builder.append("-> Verso:" + getAlgorithmVersionId() + "\n");
    builder.append("-> Configurador: " + getAlgorithmConfigurator() + "\n");
    builder.append("-> Parmetros:\n");
    builder.append("{\n");
    for (NodeParameter parameter : getParameters()) {
      builder.append("--> Nome:" + parameter.getName() + "\n");
      builder.append("--> Rtulo:" + parameter.getLabel() + "\n");
      builder.append("--> Tipo:" + parameter.getType() + "\n");
      builder.append("--> Valor:" + parameter.getValue() + "\n");
    }
    builder.append("}\n");
    builder.append("-> x: " + getX() + "\n");
    builder.append("-> y: " + getY() + "\n");
    builder.append("-> Largura: " + getWidth() + "\n");
    builder.append("-> Altura: " + getHeight() + "\n");
    builder.append("-> Bypassed: " + isBypassed() + "\n");
    return builder.toString();
  }

  /**
   * Atribui um configurador a este n.
   *
   * @param algorithmConfigurator O configurator (No aceita {@code null}).
   */
  private void setAlgorithmConfigurator(
    AlgorithmConfigurator algorithmConfigurator) {
    if (algorithmConfigurator == null) {
      throw new IllegalArgumentException(
        "O parmetro this.algorithmConfigurator est nulo.");
    }

    this.algorithmConfigurator = algorithmConfigurator;
  }

  /**
   * Atribui o nome do algoritmo.
   *
   * @param algorithmName O nome do algoritmo (No aceita {@code null}).
   */
  private void setAlgorithmName(String algorithmName) {
    if (algorithmName == null) {
      throw new IllegalArgumentException("O parmetro algorithmName est nulo.");
    }
    this.algorithmName = algorithmName;
  }

  /**
   * Atribui o identificador da verso do algoritmo.
   *
   * @param algorithmVersionId O identificador da verso do algoritmo (No
   *        aceita {@code null}).
   */
  private void setAlgorithmVersionId(AlgorithmVersionId algorithmVersionId) {
    if (algorithmVersionId == null) {
      throw new IllegalArgumentException(
        "O parmetro algorithmVersionId est nulo.");
    }
    this.algorithmVersionId = algorithmVersionId;
  }

  /**
   * Atribui a altura.
   *
   * @param height A altura (Precisa ser positiva).
   */
  private void setHeight(int height) {
    if (height <= 0) {
      throw new IllegalArgumentException(
        "O parmetro height tem que ser positivo.\nheight = " + height);
    }
    this.height = height;
  }

  /**
   * Atribui o identificador.
   *
   * @param id O identificador.
   */
  private void setId(int id) {
    this.id = id;
  }

  /**
   * <p>
   * Atribui parmetros a este n.
   * </p>
   *
   * <p>
   * Armazena uma <i>shallow copy</i> do conjunto.
   * </p>
   *
   * @param parameters O conjunto de ns (No aceita {@code null}).
   */
  private void setParameters(Set<NodeParameter> parameters) {
    if (parameters == null) {
      throw new IllegalArgumentException("O parmetro parameters est nulo.");
    }
    this.parameters = new HashSet<NodeParameter>(parameters);
  }

  /**
   * Atribui a largura.
   *
   * @param width A largura (Precisa ser positiva).
   */
  private void setWidth(int width) {
    if (width <= 0) {
      throw new IllegalArgumentException(
        "O parmetro width tem que ser positivo.\nwidth = " + width);
    }
    this.width = width;
  }

  /**
   * Atribui a abscissa esquerda.
   *
   * @param x A abscissa esquerda.
   */
  private void setX(int x) {
    this.x = x;
  }

  /**
   * Atribui a ordenada superior.
   *
   * @param y A ordenada superior.
   */
  private void setY(int y) {
    this.y = y;
  }

  /**
   * Atribui o sinal de desvio.
   *
   * @param bypassed O sinal de desvio.
   */
  public void setBypassed(boolean bypassed) {
    this.bypassed = bypassed;
  }

  /**
   * Atribui o sinal que indica que o n prov cdigo de sada durante sua
   * execuo.
   *
   * @param hasExitCode Sinal que indica se o n prov cdigo de sada.
   */
  public void setHasExitCode(boolean hasExitCode) {
    this.hasExitCode = hasExitCode;
  }

  /**
   * Atribui o arquivo que armazena o cdigo de sada do n.
   *
   * @param exitCodeLogFile Arquivo que armazena o cdigo de sada.
   */
  public void setExitCodeLogFile(FileURLValue exitCodeLogFile) {
    this.exitCodeLogFile = exitCodeLogFile;
    // Garante consistncia.
    if (null != algorithmConfigurator) {
      algorithmConfigurator.setExitCodeLogFile(exitCodeLogFile);
    }
  }

}
