/*
 * MailServiceInterface.java $Author: isabella $ $Revision: 175476 $ - $Date:
 * 2010-07-06 16:18:07 +0000 (Tue, 06 Jul 2010) $
 */
package csbase.remote;

import java.io.File;
import java.rmi.RemoteException;

/**
 * Interface remota para acesso ao servio de e-mail.
 * 
 * @author Andr Luiz Clinio.
 */
public interface MailServiceInterface extends ServiceInterface {

  /** Nome do servio para o <code>ServiceManager</code> */
  public static final String SERVICE_NAME = "MailService";

  /**
   * Mtodo para envio de email para um usurio.
   * 
   * @param senderName o remetente do email
   * @param userId o destinatrio do email
   * @param content o contedo textual do email
   * 
   * @return flag indicativo de sucesso.
   * @throws RemoteException em caso de erro de comunicao.
   */
  public boolean mailTo(final String senderName, final Object userId,
    final String content) throws RemoteException;

  /**
   * Mtodo para envio de email com anexo para um usurio.
   * 
   * @param senderName o remetente do email
   * @param userId o destinatrio do email
   * @param content o contedo textual do email
   * @param attachments a lista de arquivos para anexar ao email
   * 
   * @return flag indicativo de sucesso.
   * @throws RemoteException em caso de erro de comunicao.
   */
  public boolean mailTo(final String senderName, final Object userId,
    final String content, final File[] attachments) throws RemoteException;
  
  
  /**
   * Mtodo para envio de email com anexo para um usurio.
   * 
   * @param senderName o remetente do email
   * @param userId o destinatrio do email
   * @param content o contedo textual do email
   * @param attachments a lista de arquivos para anexar ao email
   * @param mimeType Tipo MIME.
   * 
   * @return flag indicativo de sucesso.
   * @throws RemoteException em caso de erro de comunicao.
   */
  public boolean mailTo(final String senderName, final Object userId,
    final String content, final File[] attachments, final String mimeType)
    throws RemoteException;
  

  /**
   * Mtodo para envio de email para o suporte.
   * 
   * @param content o contedo textual do email
   * 
   * @return flag indicativo de sucesso.
   * @throws RemoteException em caso de erro de comunicao.
   */
  public boolean mailSupport(final String content) throws RemoteException;

  /**
   * Mtodo para envio de email com anexo para o suporte.
   * 
   * @param content o contedo textual do email
   * @param attachments a lista de arquivos para anexar ao email
   * 
   * @return flag indicativo de sucesso.
   * @throws RemoteException em caso de erro de comunicao.
   */
  public boolean mailSupport(final String content, final File[] attachments)
    throws RemoteException;

  /**
   * Mtodo de envio de email.
   * 
   * @param senderName o remetente do email
   * @param to a lista de destinatrios do email
   * @param content o contedo textual do email
   * 
   * @return flag indicativo de sucesso.
   * @throws RemoteException em caso de erro de comunicao.
   */
  public boolean sendMail(final String senderName, final String[] to,
    final String content) throws RemoteException;

  /**
   * Mtodo de envio de email com flexibilidade para customizar o assunto, o
   * campo de resposta (reply) e o campo de "copia para" (cc).
   * 
   * @param senderName o remetente do email
   * @param to a lista de destinatrios do email
   * @param content o contedo textual do email
   * @param subject o assunto do email
   * @param replyTo o endereo que ir receber resposta (reply)
   * @param copyTo a lista de endereos que iro receber uma cpia do email
   * 
   * @return flag indicativo de sucesso.
   * @throws RemoteException em caso de erro de comunicao.
   */
  public boolean sendMail(final String senderName, final String[] to,
    final String content, final String subject, final String replyTo,
    final String[] copyTo) throws RemoteException;

  /**
   *  Envia um e-mail.
   * 
   * @param senderName o remetente do email
   * @param to a lista de destinatrios do email
   * @param content o contedo textual do email
   * @param subject o assunto do email
   * @param replyTo o endereo que ir receber resposta (reply)
   * @param copyTo a lista de endereos que iro receber uma cpia do email
   * @param attachments a lista de arquivos para anexar ao email
   * 
   * @return flag indicativo de sucesso.
   * @throws RemoteException em caso de erro de comunicao.
   */
  public boolean sendMail(final String senderName, final String[] to,
    final String content, final String subject, final String replyTo,
    final String[] copyTo, final File[] attachments) throws RemoteException;
  
  /**
   * Envia um e-mail.
   * 
   * @param senderName o remetente do email
   * @param to a lista de destinatrios do email
   * @param content o contedo textual do email
   * @param subject o assunto do email
   * @param replyTo o endereo que ir receber resposta (reply)
   * @param copyTo a lista de endereos que iro receber uma cpia do email
   * @param attachments a lista de arquivos para anexar ao email
   * @param mimeType Tipo MIME do e-mail.
   * 
   * @return flag indicativo de sucesso.
   * @throws RemoteException em caso de erro de comunicao.
   */
  public boolean sendMail(final String senderName, final String[] to,
    final String content, final String subject, final String replyTo,
    final String[] copyTo, final File[] attachments, String mimeType) throws RemoteException;
}
