package csbase.logic.algorithms;


/**
 * Contexto para a criao da linha de comando.
 *
 * @author ururahy e lmoreira
 *
 */
public class CommandLineContext {

  /**
   * Indica se comando deve ser contrudo como um script.
   */
  private final boolean isScript;

  /**
   * Identificador do n para execuo de fluxos.
   */
  private Integer nodeIdForFlow;

  /**
   * O identificador do comando.
   */
  private String commandId;

  /**
   * O separador de arquivo.
   */
  private char fileSeparator;
  /**
   * O identificador da plataforma.
   */
  private String platformId;
  /**
   * O caminho para o diretrio do projeto relativo ao repositrio de projetos.
   */
  private String projectDirectory;

  /**
   * O caminho absoluto para o repositrio de algoritmos (no sga).
   */
  private String algorithmRootDirectory;

  /**
   * O caminho absoluto para o repositrio de projetos (no sga).
   */
  private String projectRootDirectory;

  /**
   * O caminho absoluto para o diretrio base de sandboxes (no sga).
   */
  private String sandboxRootDirectory;

  /**
   * O caminho para a sandbox de execuo do comando relativo ao diretrio base
   * de sandboxes.
   */
  private String sandboxDirectory;

  /**
   * O caminho absoluto para a diretrio onde se faz a persistncia do comando
   * (no sga).
   */
  private String persistenceDirectory;

  /**
   * Nome da mquina que solicitou a execuo do algoritmo.
   */
  private String clientHostName;

  /**
   * Construtor.
   *
   * @param commandId O identificador do comando (No aceita {@code null}).
   * @param platformId O identificador da plataforma (No aceita {@code null}).
   * @param fileSeparator O separador de arquivo (No aceita {@code null}).
   * @param projectDirectory O caminho para o diretrio do projeto relativo ao
   *        diretrio do servidor (No aceita {@code null}).
   * @param algorithmRootDirectory Caminho absoluto para o repositrio de
   *        algoritmos no sga.
   * @param projectRootDirectory Caminho absoluto para o repositrio de projetos
   *        no sga.
   * @param sandboxRootDirectory Caminho absoluto para o diretrio base de
   *        sandboxes no sga.
   * @param sandboxDirectory O Caminho para o sandbox de execuo do comando
   *        relativo ao diretrio base de sandboxes.
   * @param persistenceDirectory Caminho absoluto para o diretrio de
   *        persistncia do comando no sga.
   * @param isScript Indica se comando deve ser contrudo como um script.
   * @param nodeIdForFlow Identificador do n (para o caso de fluxo).
   * @param clientHostName Nome da mquina que solicitou a execuo do
   *        algoritmo.
   */
  public CommandLineContext(String commandId, String platformId,
    char fileSeparator, String projectDirectory, String algorithmRootDirectory,
    String projectRootDirectory, String sandboxRootDirectory,
    String sandboxDirectory, String persistenceDirectory, boolean isScript,
    Integer nodeIdForFlow, String clientHostName) {
    this.commandId = commandId;
    this.platformId = platformId;
    this.fileSeparator = fileSeparator;
    this.projectDirectory = projectDirectory;
    this.algorithmRootDirectory = algorithmRootDirectory;
    this.projectRootDirectory = projectRootDirectory;
    this.sandboxRootDirectory = sandboxRootDirectory;
    this.sandboxDirectory = sandboxDirectory;
    this.persistenceDirectory = persistenceDirectory;
    this.isScript = isScript;
    this.nodeIdForFlow = nodeIdForFlow;
    this.clientHostName = clientHostName;
  }

  /**
   * Constri o contexto a partir dos dados de outro contexto.
   *
   * @param otherContext o outro contexto.
   */
  protected CommandLineContext(CommandLineContext otherContext) {
    this(otherContext.commandId, otherContext.platformId,
      otherContext.fileSeparator, otherContext.projectDirectory,
      otherContext.algorithmRootDirectory, otherContext.projectRootDirectory,
      otherContext.sandboxRootDirectory, otherContext.sandboxDirectory,
      otherContext.persistenceDirectory, otherContext.isScript,
      otherContext.nodeIdForFlow, otherContext.clientHostName);
  }

  /**
   * Obtm o identificador do comando.
   *
   * @return O identificador do comando.
   */
  public String getCommandId() {
    return this.commandId;
  }

  /**
   * Obtm o separador de diretrios.
   *
   * @return o separador de diretrios
   */
  public char getFileSeparator() {
    return this.fileSeparator;
  }

  /**
   * Obtm o identificador da plataforma.
   *
   * @return O identificador da plataforma.
   */
  public String getPlatformId() {
    return this.platformId;
  }

  /**
   * Obtm o caminho para o diretrio do projeto relativo ao repositrio de
   * projetos.
   *
   * @return .
   */
  public String getProjectDirectory() {
    return this.projectDirectory;
  }

  /**
   * Obtm o caminho absoluto para o repositrio de algoritmos no sga.
   *
   * @return O caminho absoluto para o repositrio de algoritmos no sga.
   */
  public String getAlgorithmRootDirectory() {
    return this.algorithmRootDirectory;
  }

  /**
   * Obtm o caminho absoluto para o repositrio de projetos no sga.
   *
   * @return O caminho absoluto para o repositrio de projetos no sga.
   */
  public String getProjectRootDirectory() {
    return this.projectRootDirectory;
  }

  /**
   * Obtm o caminho absoluto para o diretrio base de sandboxes no sga.
   *
   * @return O caminho absoluto para o diretrio de sandboxes no sga.
   */
  public String getSandboxRootDirectory() {
    return this.sandboxRootDirectory;
  }

  /**
   * Obtm o caminho para o sandbox de execuo do comando relativo ao diretrio
   * base de sandboxes.
   *
   * @return O caminho para o sandbox de execuo do comando relativo ao
   *         diretrio base de sandboxes.
   */
  public String getSandboxDirectory() {
    return this.sandboxDirectory;
  }

  /**
   * Obtm o caminho absoluto para o diretrio do arquivo de persistncia do
   * comando no sga.
   *
   * @return O caminho absoluto para o diretrio do arquivo de persistncia do
   *         comando no sga.
   */
  public String getPersistenceDirectory() {
    return this.persistenceDirectory;
  }

  /**
   * Indica se comando deve ser contrudo como um script.
   *
   * @return verdadeiro se o comando deve ser contrudo como um script ou no.
   */
  public boolean isScript() {
    return isScript;
  }

  /**
   * Retorna o identificador do n para execuo do fluxo.
   *
   * @return nodeId O identificador do n do fluxo.
   */
  public Integer getNodeIdForFlow() {
    return nodeIdForFlow;
  }

  /**
   * @return Nome da mquina que solicitou a execuo do algoritmo.
   */
  public String getClientHostName() {
    return clientHostName;
  }

}
