package csbase.logic.algorithms.parsers;

import java.util.Collections;
import java.util.List;

import csbase.exception.ParseException;
import csbase.logic.ProjectFileType;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.logic.algorithms.parameters.OutputFileParameter;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.ParameterGroup;

/**
 * <p>
 * Analisador de {@link OutputFileParameter}.
 * </p>
 *
 * <p>
 * Este parser l os atributos de parmetros do tipo "arquivo de sada". O
 * elemento corrente do {@link XmlParser analisador de XML} precisa ser um
 * elemento {@link OutputFileParameter}.
 * </p>
 *
 */
public class OutputFileParameterParser extends
  AbstractFileParameterParser<OutputFileParameter> {

  /**
   * <p>
   * O elemento {@value #OUTPUT_FILE_PARAMETER_ELEMENT}: descreve as
   * propriedades de um {@link OutputFileParameter parmetro do tipo arquivo de
   * sada}.
   * </p>
   * <p>
   *  filho do elemento {@link ParameterGroup}.
   * </p>
   */
  public static final String OUTPUT_FILE_PARAMETER_ELEMENT = "arquivo_de_saida";

  /**
   * <p>
   * O atributo
   * {@value #OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_ATTRIBUTE} do
   * elemento {@link OutputFileParameter}.
   * </p>
   *
   * <p>
   * Indica se o {@link OutputFileParameter arquivo de sada} deve criar uma
   * extenso se ela no for informada,  opcional, o seu valor-padro 
   * {@link #OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_DEFAULT_VALUE} e
   *  do tipo booleano.
   * </p>
   */
  protected static final String OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_ATTRIBUTE =
    "forcar_extensao";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_ATTRIBUTE} do
   * elemento {@link OutputFileParameter}
   * </p>
   * <p>
   * O seu valor 
   * {@value #OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_DEFAULT_VALUE}.
   * </p>
   */
  protected static final boolean OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_DEFAULT_VALUE =
    false;

  /**
   * {@inheritDoc}
   */
  @Override
  public OutputFileParameter createFileParameter(XmlParser parser, String name,
    String label, String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, String[] types, FileParameterMode mode)
    throws ParseException {

    final boolean sortDefined = parser.hasAttributeValue(
      FILE_PARAMETER_ELEMENT_MUST_SORT_ATTRIBUTE);
    if (sortDefined) {
      boolean mustSort = parser.extractAttributeValueAsBoolean(
        FILE_PARAMETER_ELEMENT_MUST_SORT_ATTRIBUTE,
        FILE_PARAMETER_ELEMENT_MUST_SORT_DEFAULT_VALUE);
      if (mustSort) {
        throw new ParseException("Arquivo de saida no admite atributo: "
          + FILE_PARAMETER_ELEMENT_MUST_SORT_ATTRIBUTE);
      }
    }

    FileParameterPipeAcceptance usesPipe = parser
      .extractAttributeValueAsEnumeration(
        FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_ATTRIBUTE,
        FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_DEFAULT_VALUE,
        new FileParameterPipeAcceptanceConverter());
    boolean mustForceExtension = parser.extractAttributeValueAsBoolean(
      OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_ATTRIBUTE,
      OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_DEFAULT_VALUE);
    boolean useRootDirectoryAsDefault = parser.extractAttributeValueAsBoolean(
      FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_ATTRIBUTE,
      FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_DEFAULT_VALUE);
    FileURLValue defaultValue = null;
    if (useRootDirectoryAsDefault) {
      if (mode == FileParameterMode.REGULAR_FILE) {
        throw new ParseException(
          "O atributo {0}  invlido quando o parmetro de sada  um arquivo.",
          new Object[] {
              FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_ATTRIBUTE });
      }
      defaultValue = new FileURLValue(".", ProjectFileType.DIRECTORY_TYPE);
    }

    boolean useFilter = parser.extractAttributeValueAsBoolean(
      FILE_PARAMETER_ELEMENT_USE_FILTER_ATTRIBUTE,
      FILE_PARAMETER_ELEMENT_USE_FILTER_DEFAULT_VALUE);

    if (types != null && types.length > 1) {
      throw new ParseException(
        "Arquivo de sada no admite mltiplos tipos de arquivo");
    }

    OutputFileParameter parameter = new OutputFileParameter(name, label,
      description, defaultValue, isOptional, isVisible, commandLinePattern,
      types, mode, usesPipe, mustForceExtension, useFilter);
    return parameter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Class<? extends Parameter>> getParameterTypes() {
    return Collections.singletonList(OutputFileParameter.class);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean acceptBothCategory() {
    return false;
  }
}
