/*
 * Platform.java
 *
 * $Author$
 * $Revision$  - $Date$
 */
package csbase.logic;

import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;

import csbase.remote.ClientRemoteLocator;
import csbase.util.restart.RestartListener;
import csbase.util.restart.RestartManager;

/**
 * A classe <code>Platform</code> representa uma Plataforma de Execuo. A
 * classe Platform mantm uma cache dos objetos Platform instanciados
 * localmente.
 *
 * @author $Author$
 * @version $Revision$
 */
public class Platform implements Serializable {
  /** Cache local de plataformas */
  private static Hashtable<Object, Platform> platforms =
    new Hashtable<Object, Platform>(); //cache local

  /** Indica se a cache est completa */
  private static boolean hasAllPlatforms = false;

  /** Objeto que permite a observao local da classe Platform. */
  private static Observable observable =
    new Observable() {
      @Override
      public void notifyObservers(Object arg) {
        setChanged();
        super.notifyObservers(arg);
      }
    };
  private static Comparator<Platform> nameComparator = null;
  private static Comparator descrComparator = null;

  /** Identificao de uma plataforma, gerada pelo servio de administrao */
  private final Object id;

  /** Informaes da plataforma. */
  private final PlatformInfo info;

  /**
   * Verifica se duas plataformas so iguais. Uma plataforma  igual a outra se
   * ambas possuem o mesmo identificador.
   *
   * @param obj plataforma com a qual a comparao deve ser realizada
   *
   * @return .
   */
  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof Platform)) {
      return false;
    }
    Platform platform = (Platform)obj;
    return platform.getId().equals(id);
  }

  /**
   * Calcula o cdigo hash do objeto.
   *
   * @return Inteiro que representa o cdigo hash do objeto.
   */
  @Override
  public int hashCode() {
    return id.hashCode();
  }

  public boolean equalContents(Object obj) {
    if (!(obj instanceof Platform)) {
      return false;
    }
    Platform platform = (Platform)obj;
    return platform.getId().equals(id) &&
    platform.getPlatformInfo().equals(info);
  }

  /**
   * Obtem a plataforma que possui o identificador especificado.
   *
   * @param id identificador da plataforma
   *
   * @return plataforma que possui o identificador ou null caso essa plataforma
   *         no exista
   *
   * @throws RemoteException falha de rmi
   */
  public static Platform getPlatform(Object id)
  throws RemoteException {
    if (id == null) {
      return null;
    }
    Platform platform = platforms.get(id);
    if ((platform == null) && !hasAllPlatforms) {
      platform = ClientRemoteLocator.administrationService.getPlatform(id);
      if (platform != null) {
        platforms.put(id, platform);
      }
    }
    return platform;
  }

  /**
   * Obtem a plataforma que possui o nome especificado.
   *
   * @param name nome da plataforma procurada
   *
   * @return plataforma que possui o nome procurado ou null caso essa
   *         plataforma no exista
   *
   * @throws Exception .
   */
  public static Platform getPlatformByName(String name)
  throws Exception {
    List<Platform> allPlatforms = getAllPlatforms();
    for (Platform platform : allPlatforms) {
      if (platform.getName().equalsIgnoreCase(name)) {
        return platform;
      }
    }
    return null;
  }

  /**
   * Obtem todas as plataformas cadastradas.
   *
   * @return vetor com as plataformas
   *
   * @throws Exception .
   */
  public static List<Platform> getAllPlatforms()
  throws Exception {
    if (hasAllPlatforms) {
      return toVector();
    }
    List<Platform> allPlatforms =
      ClientRemoteLocator.administrationService.getAllPlatforms();
    for (Platform platform : allPlatforms) {
      if (!platforms.containsKey(platform.id)) {
        platforms.put(platform.id, platform);
      }
    }
    hasAllPlatforms = true;
    return allPlatforms;
  }

  /**
   * Obtem todas as plataformas cadastradas
   *
   * @param sortByName <code>true</code> se as plataformas devero estar
   *        ordenadas alfabeticamente pelo nome e <code>false</code> se a
   *        ordenao no  necessria
   *
   * @return vetor com as plataformas
   *
   * @throws Exception .
   */
  public static List<Platform> getAllPlatforms(boolean sortByName)
  throws Exception {
    List<Platform> allPlatforms = getAllPlatforms();
    if (sortByName) {
      Collections.sort(allPlatforms, getNameComparator());
    }
    return allPlatforms;
  }

  /**
   * Criao de uma nova plataforma. A plataforma criada  inserida na cache
   * local.
   *
   * @param info as informaes da nova plataforma
   *
   * @return plataforma criada ou null caso j exista uma plataforma com o
   *         mesmo nome
   *
   * @throws Exception .
   */
  public static Platform createPlatform(PlatformInfo info)
  throws Exception {
    if ((info.name == null) || info.name.trim().equals("")) {
      throw new Exception("nome no pode ser vazio");
    }
    if (getPlatformByName(info.name) != null) {
      return null;
    }
    Platform platform =
      ClientRemoteLocator.administrationService.createPlatform(info);
    platforms.put(platform.id, platform);
    return platform;
  }

  /**
   * Modificao de uma plataforma. No  permitido alterar o nome da
   * plataforma, pois o mesmo  usado na configurao dos SGAs e no
   * Repositrio de Algoritmos.
   *
   * @param id identificador da platforma a ser modificada
   * @param info novas informaes da plataforma
   *
   * @return plataforma modificada ou null se a modificao no pode ser
   *         realizada (modificao de nome) ou se no existe plataforma com o
   *         identificador especificado
   *
   * @throws Exception .
   */
  public static Platform modifyPlatform(Object id, PlatformInfo info)
  throws Exception {
    Platform platform = getPlatform(id);
    if (platform == null) {
      return null;
    }
    if (!platform.getName().equals(info.name)) {
      return null;
    }
    platform =
      ClientRemoteLocator.administrationService.modifyPlatform(id, info);
    platforms.put(platform.id, platform);
    return platform;
  }

  /**
   * Remoo de uma plataforma.
   *
   * @param id identificador da plataforma a ser removida
   *
   * @throws Exception .
   */
  public static void deletePlatform(Object id)
  throws Exception {
    ClientRemoteLocator.administrationService.deletePlatform(id);
    platforms.remove(id);
  }

  /**
   * Obtm o identificador da plataforma
   *
   * @return identificador da plataforma
   */
  public Object getId() {
    return id;
  }

  /**
   * Obtm o nome da plataforma
   *
   * @return nome da plataforma
   */
  public String getName() {
    return info.name;
  }

  /**
   * Obtm a descrio da plataforma
   *
   * @return descrio da plataforma
   */
  public String getDescription() {
    return info.description;
  }

  /**
   * Obtm uma cpia do PlatformInfo desta plataforma.
   *
   * @return a cpia do PlatformInfo
   */
  public PlatformInfo getPlatformInfo() {
    return (PlatformInfo)info.clone();
  }

  /**
   * Obtm um vetor com as plataformas na cache.
   *
   * @return vetor contendo todas as plataformas
   */
  private static Vector<Platform> toVector() {
    Vector<Platform> allPlatforms = new Vector<Platform>();
    for (Enumeration<Platform> elems = platforms.elements();
         elems.hasMoreElements();) {
      allPlatforms.add(elems.nextElement());
    }
    return allPlatforms;
  }

  /**
   * Obtm um comparator de <code>Platform</code> pelo critrio de ordem
   * alfabtica do nome da plataforma
   *
   * @return comparador por nome
   */
  public static Comparator<Platform> getNameComparator() {
    if (nameComparator == null) {
      nameComparator =
        new Comparator<Platform>() {
            public int compare(Platform p1, Platform p2) {
              return (p1.getName().compareTo(p2.getName()));
            }
          };
    }
    return nameComparator;
  }

  /**
   * Obtm um comparator de <code>Platform</code> pelo critrio de ordem
   * alfabtica da descrio da plataforma
   *
   * @return comparador por descrio
   */
  public static Comparator getDescrComparator() {
    if (descrComparator == null) {
      descrComparator =
        new Comparator() {
            public int compare(Object o1, Object o2) {
              return ((Platform)o1).getDescription().compareTo(
                ((Platform)o2).getDescription());
            }
          };
    }
    return descrComparator;
  }

  /**
   * Obtm um getter para o nome de <code>Platform</code>.
   *
   * @return getter para o nome
   */
  public static Getter getNameGetter() {
    Getter nameGetter =
      new Getter() {
        public Object get(Object o) {
          return ((Platform)o).getName();
        }
      };
    return nameGetter;
  }

  /**
   * Obtm um getter para o identificador de <code>Platform</code>.
   *
   * @return getter para o identificador
   */
  public static Getter getIdGetter() {
    Getter idGetter =
      new Getter() {
        public Object get(Object o) {
          return ((Platform)o).getId();
        }
      };
    return idGetter;
  }

  /**
   * Obtm um getter para a descrio de <code>Platform</code>.
   *
   * @return getter para o descrio
   */
  public static Getter getDescriptionGetter() {
    Getter descriptionGetter =
      new Getter() {
        public Object get(Object o) {
          return ((Platform)o).getDescription();
        }
      };
    return descriptionGetter;
  }

  /**
   * Adiciona um observador local da classe. Sempre que a classe  avisada pelo
   * servio de administrao de que alguma mudana ocorreu, os observadores
   * locais tambm so notificados.
   *
   * @param obs um observador local
   */
  public static void addObserver(Observer obs) {
    observable.addObserver(obs);
  }

  /**
   * Remove um observador local da classe Platform.
   *
   * @param obs o observador a ser removido
   */
  public static void deleteObserver(Observer obs) {
    observable.deleteObserver(obs);
  }

  /**
   * Esse mtodo  chamado quando um servio de administrao sofre alguma
   * alterao relativa a plataformas. Atualiza o cache de plataformas e
   * notifica seus observadores locais.
   *
   * @param event a ao que ocorreu no servio de administrao
   */
  public static void update(AdministrationEvent event) {
    Platform platform = (Platform)event.item;
    Object id = platform.id;
    switch (event.type) {
      case AdministrationEvent.CREATE:
      case AdministrationEvent.MODIFY:
        platforms.put(id, platform);
        break;
      case AdministrationEvent.DELETE:
        platforms.remove(id);
        break;
    }
    observable.notifyObservers(event);
  }

  /**
   * Constri uma Platforma de Execuo. Somente o servidor pode construir um
   * objeto Platform, pois  quem  capaz de gerar identificadores.
   *
   * @param id identificador da plataforma
   * @param info informaes da plataforma
   */
  public Platform(Object id, PlatformInfo info) {
    this.id = id;
    this.info = info;
  }

  static {
    RestartManager.getInstance().addListener(
      new RestartListener() {
        public void restart() {
          Platform.platforms = new Hashtable<Object, Platform>();
          Platform.hasAllPlatforms = false;
          Platform.observable.deleteObservers();
        }
      });
  }
}
