package csbase.logic;

/**
 * Dado dinmico de progresso do comando.
 */
public class ProgressData {

  /**
   * Descrio textual do progresso.
   */
  private String description;

  /**
   * Valor percentual do progresso (de 0 a 100).
   */
  private Double value;

  /**
   * Construtor.
   * 
   * @param value Valor percentual do progresso.
   */
  public ProgressData(Double value) {
    this(null, value);
  }

  /**
   * Construtor.
   * 
   * @param description Descrio textual do progresso.
   */
  public ProgressData(String description) {
    this(description, Double.NaN);
  }

  /**
   * Construtor.
   * 
   * @param description Descrio textual do progresso.
   * @param value Valor percentual do progresso.
   */
  public ProgressData(String description, Double value) {
    if (value == null || value.isNaN() || value < 0 || value > 100) {
      this.value = Double.NaN;
      this.description = description;
    }
    else {
      this.value = value;
      if (description == null) {
        this.description = getFormattedValue();
      }
      else {
        this.description = description;
      }
    }
  }

  /**
   * Obtm o valor formatado como uma string e arredondado para um nmero
   * inteiro. Ex.: 20%.
   * 
   * @return o valor do progresso formatado.
   */
  public String getFormattedValue() {
    return String.format("%.0f%%", this.value);
  }

  /**
   * Obtm o valor percentual do progresso (de 0 a 100).
   * 
   * @return o valor do progresso.
   */
  public Double getValue() {
    return value;
  }

  /**
   * Obtm a descrio textual do progresso.
   * 
   * @return descrio.
   */
  public String getDescription() {
    return description;
  }

}
