package csbase.logic;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

/**
 * Informaes sobre o Tipo de Arquivos de Projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ProjectFileTypeInfo implements Serializable {
  private String code;
  private String description;
  private String mimeType;
  private byte[] baseIcon;
  private byte[] ucIcon;
  private byte[] cutIcon;
  private Set<String> extensions;
  private boolean isDirectory;

  /**
   * Cria as informaes sobre o Tipo de Arquivos de Projeto.
   * 
   * @param code O cdigo que identifica o tipo (no aceita null).
   * @param description A descrio (no aceita null).
   * @param mimeType O mime-type (no aceita null).
   * @param baseIcon O cone base (no aceita null).
   * @param ucIcon O cone em construo (no aceita null).
   * @param cutIcon O cone cortar (no aceita null).
   * @param extensions As extenses. Para informar que no h
   * extenses, use um conjunto vazio (no aceita null).
   * @param isDirectory Indica se o tipo  aplicavel a diretrios
   * {@code true} ou a arquivos {@code false}.
   */
  public ProjectFileTypeInfo(String code, String description,
    String mimeType, byte[] baseIcon, byte[] ucIcon, byte[] cutIcon,
    Set<String> extensions, boolean isDirectory) {
    setCode(code);
    setDescription(description);
    setMimeType(mimeType);
    setBaseIcon(baseIcon);
    setUnderConstruction(ucIcon);
    setCutIcon(cutIcon);
    setExtensions(extensions);
    setDirectory(isDirectory);
  }

  /**
   * Obtm o cdigo que identifica o tipo.
   * 
   * @return O cdigo.
   */
  public String getCode() {
    return code;
  }

  /**
   * Obtm a descrio do tipo.
   * 
   * @return A descrio.
   */
  public String getDescription() {
    return description;
  }

  /**
   * Obtm o mime-type do tipo.
   * 
   * @return O mime-type.
   */
  public String getMimeType() {
    return mimeType;
  }

  /**
   * Obtm o cone base.
   * 
   * @return O cone base.
   */
  public byte[] getBaseIcon() {
    return baseIcon;
  }

  /**
   * Obtm o cone em construo.
   * 
   * @return O cone em construo.
   */
  public byte[] getUnderConstructionIcon() {
    return ucIcon;
  }

  /**
   * Obtm o cone cortar.
   * 
   * @return O cone cortar.
   */
  public byte[] getCutIcon() {
    return cutIcon;
  }

  /**
   * Obtm as extenses.
   * 
   * O conjunto retornado  imutvel.
   * 
   * @return As extenses. Se no houver extenses, retornar um conjunto vazio.
   */
  public Set<String> getExtensions() {
    return Collections.unmodifiableSet(extensions);
  }

  /**
   * Indica se o tipo  aplicavel a diretrios ou a arquivos.
   * 
   * @return {@code true} se  aplicavel a diretrios ou {@code false} se 
   * aplicavel a arquivos.
   */
  public boolean isDirectory() {
    return isDirectory;
  }

  private void setCode(String code) {
    if (code == null) {
      throw new IllegalArgumentException("code == null");
    }
    this.code = code;
  }

  private void setDescription(String description) {
    if (description == null) {
      throw new IllegalArgumentException("description == null");
    }
    this.description = description;
  }

  private void setMimeType(String mimeType) {
    if (mimeType == null) {
      throw new IllegalArgumentException("mimeType == null");
    }
    this.mimeType = mimeType;
  }

  private void setBaseIcon(byte[] baseIcon) {
    if (baseIcon == null) {
      throw new IllegalArgumentException("baseIcon == null");
    }
    this.baseIcon = Arrays.copyOf(baseIcon, baseIcon.length);
  }

  private void setUnderConstruction(byte[] ucIcon) {
    if (ucIcon == null) {
      throw new IllegalArgumentException("ucIcon == null");
    }
    this.ucIcon = Arrays.copyOf(ucIcon, ucIcon.length);
  }

  private void setCutIcon(byte[] cutIcon) {
    if (cutIcon == null) {
      throw new IllegalArgumentException("cutIcon == null");
    }
    this.cutIcon = Arrays.copyOf(cutIcon, cutIcon.length);
  }

  private void setExtensions(Set<String> extensions) {
    if (extensions == null) {
      throw new IllegalArgumentException("extensions == null");
    }
    this.extensions = new HashSet<String>();
    this.extensions.addAll(extensions);
  }

  private void setDirectory(boolean isDirectory) {
    this.isDirectory = isDirectory;
  }
}
