/*
 * $Id$
 */
package csbase.logic;

import java.util.Collections;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Set;

/**
 * Representa os dados de cada maquina de um SGA para ser mostrado no cliente.
 * 
 * @author $Author$
 * @version $Revision$
 */
public class SGAInfo implements MonitoringSet, Comparable<SGAInfo> {
  /**
   * Capacidade padro dos SGAs quando o benchmark de rede no deve ser
   * executado pois no h transferncia de arquivos devido ao projeto estar
   * localizado em uma rea compartilhada e no via csfs.
   */
  public static final int ALL_CAPACITY = -2;
  /**
   * Capacidade padro dos SGAs quando no so executados benchmarks para a
   * avaliao dos servidores.
   */
  public static final int NO_CAPACITY = -1;
  /**
   * Define que um SGA deve calcular regularmente a sua taxa de transferncia de
   * dados em rede.
   */
  public static final int CALC_CAPACITY = 0;

  /* Tipos de ordenao de bytes de processador. */
  /**
   * Tipo onde o bit menos significativo vem primeiro.
   */
  public static final int LITTLE_ENDIAN = 0;
  /**
   * Tipo onde bit mais significativo vem primeiro.
   */
  public static final int BIG_ENDIAN = 1;

  /** Ordenao de bytes no processador. */
  private int byteOrder;

  /** Separador de diretrios usado pelo n. */
  private char fileSeparator;

  /** Localizao do diretrio base dos projetos. */
  private String[] projectRootDirectory;

  /** Localizao do diretrio base dos algoritmos. */
  private String[] algorithmRootDirectory;

  /** Localizao do diretrio base de sandboxes. */
  private String[] sandboxRootDirectory;

  /** Nome da mquina hospedeira. */
  private String hostName;

  /** Identificao da plataforma. */
  private String platformId;

  /** Nmero de processadores da mquina hospedeira. */
  private int numProcessors;

  /** Tamanho da memria RAM da mquina hospedeira. */
  private int memoryRamInfoMb;
  /** Tamanho da memria de swap da mquina hospedeira. */
  private int memorySwapInfoMb;

  /** Velocidade do processador da mquina hospedeira. */
  private int clockSpeedMHz;

  /** Percentual de memria RAM livre. */
  private double hostRAMFreeMemory;
  /** Percentual de memria de swap livre. */
  private double hostSwapFreeMemory;

  /** Nmero de jobs em execuo na mquina hospedeira. */
  private int numberOfJobs;

  /**
   * Carga na mquina hospedeira (ltimo(s) 1 min.). Essa informao corresponde
   * ao percentual de ocupao do(s) processador(es) da mquina hospedeira.
   */
  private double hostLoadAvg1;

  /** Tabela com as capacidades do SGA */
  private Hashtable<CapacityType, Long> capacities;

  /**
   * Carga na mquina hospedeira (ltimo(s) 5 min.). Essa informao corresponde
   * ao percentual de ocupao do(s) processador(es) da mquina hospedeira.
   */
  private double hostLoadAvg5;

  /**
   * Carga na mquina hospedeira (ltimo(s) 15 min.). Essa informao
   * corresponde ao percentual de ocupao do(s) processador(es) da mquina
   * hospedeira.
   */
  private double hostLoadAvg15;

  /** Indica se a maquina est? acess?vel. */
  private boolean alive;

  /** Requisitos que um n possui. */
  private Set<String> requirements;

  /**
   * Adiciona um requisito ao n.
   * 
   * @param requirement o requisito a ser adicionado.
   */
  public void addRequirement(String requirement) {
    requirements.add(requirement.trim().toUpperCase());
  }

  /**
   * Verifica se o n possui um determinado requisito.
   * 
   * @param requirement o requisito que se deseja verificar.
   * 
   * @return true se o n possui o requisito.
   */
  public boolean hasRequirement(String requirement) {
    return requirements.contains(requirement.trim().toUpperCase());
  }

  /**
   * Obtm uma cpia no-modificvel do conjunto de requisitos.
   * 
   * @return cpia no-modificvel do conjunto de requisitos
   */
  public Set<String> getRequirements() {
    return Collections.unmodifiableSet(requirements);
  }

  /**
   * @see java.lang.Comparable#compareTo(java.lang.Object)
   */
  @Override
  public int compareTo(SGAInfo sgaInfo) {
    return toString().compareTo(sgaInfo.toString());
  }

  /**
   * Obtm a carga na mquina hospedeira no ltimo minuto.
   * 
   * @return carga (percentual) no ltimo minuto
   */
  public double getCPULoad1() {
    return hostLoadAvg1;
  }

  /**
   * Obtm a carga na mquina hospedeira nos ltimos 5 minutos.
   * 
   * @return carga (percentual) nos ltimos 5 minutos
   */
  public double getCPULoad5() {
    return hostLoadAvg5;
  }

  /**
   * Obtm a carga na mquina hospedeira nos ltimos 15 minutos.
   * 
   * @return carga (percentual) nos ltimos 15 minutos
   */
  public double getCPULoad15() {
    return hostLoadAvg15;
  }

  /**
   * Obtm o percentual de memria RAM livre.
   * 
   * @return percentual de memria RAM livre
   */
  public double getRAMFreeMemory() {
    return this.hostRAMFreeMemory;
  }

  /**
   * Obtm a memria RAM livre em Mb
   * 
   * @return a memria RAM livre em Mb
   */
  public double getRAMFreeMemoryMb() {
    return hostRAMFreeMemory * (memoryRamInfoMb / 100.0);
  }

  /**
   * Obtm o percentual de memria Swap livre.
   * 
   * @return percentual de memria Swap livre
   */
  public double getSwapFreeMemory() {
    return this.hostSwapFreeMemory;
  }

  /**
   * Obtm o nmero de jobs em execuo na mquina hospedeira.
   * 
   * @return nmero de jobs em execuo na mquina hospedeira.
   */
  public int getNumberOfJobs() {
    return this.numberOfJobs;
  }

  /**
   * Obtm o diretrio raiz de projetos do SGA em relao ao servidor SSI (NFS).
   * 
   * @return o diretrio raiz de projetos
   */
  public String[] getProjectRootDirectory() {
    return this.projectRootDirectory;
  }

  /**
   * Obtm o diretrio raiz de algoritmos do SGA em relao ao servidor SSI
   * (NFS).
   * 
   * @return o diretrio raiz de algoritmos
   */
  public String[] getAlgorithmRootDirectory() {
    return this.algorithmRootDirectory;
  }

  /**
   * Obtm o diretrio raiz de sandboxes do SGA.
   * 
   * @return o diretrio raiz de sandboxes
   */
  public String[] getSandboxRootDirectory() {
    return this.sandboxRootDirectory;
  }

  /**
   * Obtm o separador de arquivos.
   * 
   * @return o caracter de separao de diretrios do SGA.
   */
  public char getFileSeparator() {
    return this.fileSeparator;
  }

  /**
   * Converte para uma string.
   * 
   * @return o nome da mquina.
   */
  @Override
  public String toString() {
    return getHostName();
  }

  /**
   * Obtm o nome da mquina hospedeira.
   * 
   * @return nome (hostname) da mquina hospedeira
   */
  public String getHostName() {
    return hostName;
  }

  /**
   * Obtm a plataforma da mquina hospedeira.
   * 
   * @return identificao da plataforma
   */
  public String getPlatformId() {
    return platformId;
  }

  /**
   * Verifica se a maquina est? acess?vel.
   * 
   * @return true se maquina SGA est? acess?vel, false caso contr?rio
   */
  public boolean getAlive() {
    return alive;
  }

  /**
   * Atualiza acessibilidade da maquina. Invalida (zera) as informaes
   * dinmicas se a maquina est inacessvel.
   * 
   * @param alive true se a maquina est acessvel, false caso contrrio
   */
  public void setAlive(boolean alive) {
    this.alive = alive;
    if (!alive) {
      setRAMFreeMemory(0.0);
      setSwapFreeMemory(0.0);
      setCPULoad(0.0, 0.0, 0.0);
    }
  }

  /**
   * Recebe a capacidade resultante de um benchmarks e insere o valor recebido
   * em uma tabela de capacidades.
   * 
   * @param type Tipo do benchmark medido
   * @param value Valor resultante do benchmark
   */
  public void addCapacities(CapacityType type, long value) {
    capacities.put(type, Long.valueOf(value));
  }

  /**
   * Obtm as capacidades dos sgas medidas atravs dos benchmarks.
   * 
   * @param capacity Tipo de capacidade a ser obtida.
   * @return O valor da capacidade.
   */
  public long getCapacity(CapacityType capacity) {
    if (capacities.containsKey(capacity)) {
      return capacities.get(capacity);
    }

    return SGAInfo.NO_CAPACITY;
  }

  /**
   * Obtm o nmero de processadores da mquina hospedeira.
   * 
   * @return nmero de processadores
   */
  public int getNumProcessors() {
    return numProcessors;
  }

  /**
   * Obtm o tamanho da memria RAM da mquina hospedeira.
   * 
   * @return tamanho da memria RAM (em Mb)
   */
  public int getRAMMemoryInfoMb() {
    return memoryRamInfoMb;
  }

  /**
   * Obtm o tamanho da memria Swap da mquina hospedeira.
   * 
   * @return tamanho da memria Swap (em Mb)
   */
  public int getSwapMemoryInfoMb() {
    return memorySwapInfoMb;
  }

  /**
   * Obtm a velocidade do processador da mquina hospedeira.
   * 
   * @return velocidade do processador (em MHz)
   */
  public int getClockSpeedMHz() {
    return clockSpeedMHz;
  }

  /**
   * Obtm a ordenao de bytes no processador.
   * 
   * @return tipo de ordenao: LITTLE_ENDIAN ou BIG_ENDIAN
   */
  public int getByteOrder() {
    return byteOrder;
  }

  /**
   * Atualiza a informao de carga na mquina hospedeira.
   * 
   * @param hostLoadAvg1 carga (percentual) no ltimo minuto
   * @param hostLoadAvg5 carga (percentual) nos ltimos 5 minutos
   * @param hostLoadAvg15 carga (percentual) nos ltimos 15 minutos
   */
  public void setCPULoad(double hostLoadAvg1, double hostLoadAvg5,
    double hostLoadAvg15) {
    this.hostLoadAvg1 = hostLoadAvg1;
    this.hostLoadAvg5 = hostLoadAvg5;
    this.hostLoadAvg15 = hostLoadAvg15;
  }

  /**
   * Atualiza memria RAM livre.
   * 
   * @param hostRAMFreeMemory percentual de memria RAM livre
   */
  public void setRAMFreeMemory(double hostRAMFreeMemory) {
    this.hostRAMFreeMemory = hostRAMFreeMemory;
  }

  /**
   * Atualiza o nmero de jobs em execuo na mquina hospedeira.
   * 
   * @param numberOfJobs nmero de jobs em execuo na mquina hospedeira.
   */
  public void setNumberOfJobs(int numberOfJobs) {
    this.numberOfJobs = numberOfJobs;
  }

  /**
   * Atualiza memria Swap livre.
   * 
   * @param hostSwapFreeMemory percentual de memria Swap livre
   */
  public void setSwapFreeMemory(double hostSwapFreeMemory) {
    this.hostSwapFreeMemory = hostSwapFreeMemory;
  }

  /**
   * Inicializa a taxa de transferncia em rede do SGA. Quando a coleta da taxa
   * de transferncia em rede deixar de ser iniciada pelo servidor, essa
   * informao passar a ser informada atravs dos dados dinmicos dos SGAs e
   * esse mtodo deixar de existir.
   * 
   * @param transferRate Taxa de transferncia do SGA.
   */
  public void setTransferRate(long transferRate) {
    capacities.put(CapacityType.NET, Long.valueOf(transferRate));
  }

  /**
   * Mtodo para comparar um SGAInfo com outro.
   * 
   * @param o .
   * 
   * @return .
   */
  @Override
  public boolean equals(MonitoringSet o) {
    if (!(o instanceof SGAInfo)) {
      return false;
    }
    SGAInfo other = (SGAInfo) o;
    if (!getHostName().equals(other.getHostName())) {
      return false;
    }
    if (getAlive() != other.getAlive()) {
      return false;
    }
    if (getCPULoad1() != other.getCPULoad1()) {
      return false;
    }
    if (getRAMMemoryInfoMb() != other.getRAMMemoryInfoMb()) {
      return false;
    }
    if (getSwapMemoryInfoMb() != other.getSwapMemoryInfoMb()) {
      return false;
    }
    if (!getPlatformId().equals(other.getPlatformId())) {
      return false;
    }
    return true;
  }

  /**
   * Cdigo hash do objeto.
   * 
   * @return inteiro representando o cdigo hash.
   */
  @Override
  public int hashCode() {
    StringBuffer buffer = new StringBuffer();
    buffer.append(getHostName());
    buffer.append(Boolean.toString(getAlive()));
    buffer.append(Double.toString(getCPULoad1()));
    buffer.append(Integer.toString(getRAMMemoryInfoMb()));
    buffer.append(Integer.toString(getSwapMemoryInfoMb()));
    buffer.append(getPlatformId());
    return buffer.toString().hashCode();
  }

  /**
   * Mtodo que retorna um String identificador do MonitoringSet.
   * 
   * @return .
   */
  @Override
  public String getKey() {
    return getHostName();
  }

  /**
   * Construtor de um SGAInfo.
   * 
   * @param hostName nome da mquina hospedeira
   * @param platformId identificao da plataforma
   * @param numProcessors nmero de processadores
   * @param memoryRamInfoMb tamanho da memria RAM (em Mb)
   * @param memorySwapInfoMb tamanho da memria Swap (em Mb)
   * @param clockSpeedMHz velocidade do processador (em MHz)
   * @param fileSeparator separador de diretrios usado pelo n
   * @param projectRootDirectory localizao do diretrio base de projetos
   * @param algorithmRootDirectory localizao do diretrio base de algoritmos
   * @param sandboxRootDirectory localizao do diretrio base de sandboxes
   * @param byteOrder ordenao de bytes no processador
   */
  public SGAInfo(String hostName, String platformId, int numProcessors,
    int memoryRamInfoMb, int memorySwapInfoMb, int clockSpeedMHz,
    char fileSeparator, String[] projectRootDirectory,
    String[] algorithmRootDirectory, String[] sandboxRootDirectory,
    int byteOrder) {
    this.hostName = hostName;
    this.platformId = platformId;
    this.numProcessors = numProcessors;
    this.memoryRamInfoMb = memoryRamInfoMb;
    this.memorySwapInfoMb = memorySwapInfoMb;
    this.clockSpeedMHz = clockSpeedMHz;
    this.fileSeparator = fileSeparator;
    this.projectRootDirectory = projectRootDirectory;
    this.algorithmRootDirectory = algorithmRootDirectory;
    this.sandboxRootDirectory = sandboxRootDirectory;
    this.byteOrder = byteOrder;
    this.hostRAMFreeMemory = -1.0;
    this.hostSwapFreeMemory = -1.0;
    this.hostLoadAvg1 = -1.0;
    this.hostLoadAvg5 = -1.0;
    this.hostLoadAvg15 = -1.0;
    this.numberOfJobs = -1;
    this.alive = false;
    this.requirements = new HashSet<String>();
    this.capacities = new Hashtable<CapacityType, Long>();
  }
}
