package csbase.logic.algorithms.parsers;

import java.io.Serializable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.DefaultEnumerationItem;
import csbase.logic.algorithms.parameters.EnumerationColumn;
import csbase.logic.algorithms.parameters.EnumerationItem;
import csbase.logic.algorithms.parameters.EnumerationListParameter;
import csbase.logic.algorithms.parameters.EnumerationParameter;
import csbase.logic.algorithms.parameters.TableParameter;

/**
 * O analisador de {@link DefaultEnumerationItem}.
 *
 * @author Tecgraf/PUC-Rio
 */
public final class EnumerationItemFactory implements Serializable  {
  /**
   * <p>
   * O elemento {@value #ELEMENT_NAME}: descreve um
   * {@link DefaultEnumerationItem item de enumerao} da
   * {@link EnumerationParameter enumerao}, {@link EnumerationListParameter}
   * ou {@link EnumerationColumn coluna de enumerao} que  descrita pelo
   * elemento que  pai deste elemento.
   * </p>
   * <p>
   *  elemento-filho de: {@link EnumerationListParameter},
   * {@link EnumerationParameter} ou {@link EnumerationColumn} de um
   * {@link TableParameter}.
   * </p>
   */
  public static final String ELEMENT_NAME = "item_de_enumeracao";

  /**
   * O atributo {@value #DESCRIPTION_ATTRIBUTE_NAME} do elemento
   * {@link #ELEMENT_NAME}. Indica a descrio do item,  opcional e  do tipo
   * string.
   */
  private static final String DESCRIPTION_ATTRIBUTE_NAME = "dica";

  /**
   * O atributo {@value #ID_ATTRIBUTE_NAME} do elemento {@link #ELEMENT_NAME}.
   * Indica o identificador do item,  obrigatrio e  do tipo string.
   */
  private static final String ID_ATTRIBUTE_NAME = "id";

  /**
   * O atributo {@value #LABEL_ATTRIBUTE_NAME} do elemento {@link #ELEMENT_NAME}
   * . Indica o rtulo do item,  obrigatrio e  do tipo string.
   */
  private static final String LABEL_ATTRIBUTE_NAME = "rotulo";

  /**
   * O atributo {@value #VALUE_ATTRIBUTE_NAME} do elemento {@link #ELEMENT_NAME}
   * . Indica o valor do item,  obrigatrio e  do tipo string.
   */
  private static final String VALUE_ATTRIBUTE_NAME = "valor";

  /**
   * <p>
   * Carrega os {@link DefaultEnumerationItem}.
   * </p>
   *
   * @param parser O analisador (No aceita {@code null}).
   * @param parameterName O nome da enumerao (No aceita {@code null}).
   * @return Os items (Se no houver itens a lista estar vazia).
   *
   * @throws ParseException Em caso de erro no XML.
   */
  public List<EnumerationItem> loadEnumerationItems(XmlParser parser,
    String parameterName) throws ParseException {
    List<EnumerationItem> items = new LinkedList<EnumerationItem>();
    if (parser.goToFirstChild(ELEMENT_NAME)) {
      do {
        EnumerationItem enumerationItem = loadEnumerationItem(parser,
          parameterName);
        if (items.contains(enumerationItem)) {
          throw new ParseException(
            "O item {0} est duplicado na enumerao {1}.", enumerationItem
              .getId(), parameterName);
        }
        for (EnumerationItem item : items) {
          if (enumerationItem.getLabel().equals(item.getLabel())) {
            throw new ParseException(
              "O item {0} est duplicado na enumerao {1}.", enumerationItem
                .getLabel(), parameterName);
          }
        }
        items.add(enumerationItem);
      } while (parser.goToNextSibling(ELEMENT_NAME));
      parser.goToParent();
    }
    return items;
  }

  /**
   * <p>
   * Carrega um {@link DefaultEnumerationItem}.
   * </p>
   *
   * <p>
   * O elemento corrente tem que ser {@link #ELEMENT_NAME}.
   * </p>
   *
   * @param parser O analisador (No aceita {@code null}).
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   * @return O item de enumerao.
   *
   * @throws ParseException Em caso de erro no XML.
   */
  public EnumerationItem loadEnumerationItem(XmlParser parser,
    String parameterName) throws ParseException {
    try {
      String elementName = parser.getElementName();
      if (!elementName.equals(ELEMENT_NAME)) {
        throw new ParseException(
          "O nome do elemento do XML encontrado {0} no  suportado. "
            + "Suportado: {1}.", elementName, ELEMENT_NAME);
      }
      String id = parser.extractAttributeValue(ID_ATTRIBUTE_NAME);
      String label = parser.extractAttributeValue(LABEL_ATTRIBUTE_NAME);
      String value = parser.extractAttributeValue(VALUE_ATTRIBUTE_NAME);
      String description = parser.extractAttributeValue(
        DESCRIPTION_ATTRIBUTE_NAME, null);
      parser.checkAttributes();
      parser.checkChildElements();
      EnumerationItem enumerationItem = new DefaultEnumerationItem(id, label,
        value, description);
      return enumerationItem;
    }
    catch (ParseException e) {
      throw new ParseException(
        "Erro ao tentar criar item de enumerao para a enumerao {0}.",
        parameterName);
    }
  }

  /**
   * Obtm um item de enumerao na relao de itens fornecida.
   *
   * @param enumerationName O nome da enumerao (No aceita {@code null}).
   * @param items Os itens disponveis (No aceita {@code null}).
   * @param itemId O identificador do item (No aceita {@code null}).
   *
   * @return O item de enumerao.
   *
   * @throws ParseException Se o item no existir.
   */
  public EnumerationItem getEnumerationItem(String enumerationName,
    List<EnumerationItem> items, String itemId) throws ParseException {
    Iterator<EnumerationItem> itemIterator = items.iterator();
    while (itemIterator.hasNext()) {
      EnumerationItem item = itemIterator.next();
      if (item.getId().equals(itemId)) {
        return item;
      }
    }
    throw new ParseException("Erro ao tentar criar a enumerao {0}:"
      + " O item padro {1} no est definido.", new Object[] { enumerationName,
          itemId });
  }
}
