package csbase.logic.algorithms.parsers;

import java.io.Serializable;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.text.MessageFormat;
import java.util.List;

import csbase.exception.OperationFailureException;
import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;

/**
 * Representa uma fbrica de um tipo de parmetro do sistema que pode ser
 * carregado dinamicamente.
 */
public interface ParameterFactory extends Serializable {

  /**
   * Cria uma instncia do parmetro a partir das informaes do parser do
   * configurador de algoritmos.
   * 
   * @param parser Parser do configurador.
   * @param commandLinePattern Padro de linha de comando do algoritmo.
   * @param group grupo em criao.
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * @return Uma instncia do parmetro criado.
   * @throws ParseException Caso no seja possvel criar o parmetro com as
   *         informaes do parser.
   */
  public abstract Parameter<?> createParameter(XmlParser parser,
    String commandLinePattern, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException;

  /**
   * Obtm os tipos de parmetro criados pela fbrica.
   *
   * @return a lista de classes de parmetros criados por esta fbrica.
   */
  public List<Class<? extends Parameter>> getParameterTypes();

  /**
   * Cria uma instncia, por reflexo, da fbrica de um tipo de parmetro, a
   * partir do nome completo da classe que a implementa.
   *
   * @param factoryClassName O nome completo da classe que implementa a fbrica.
   * @return Uma instncia da fbrica.
   * @throws OperationFailureException Caso no seja possvel criar a fbrica.
   */
  public static ParameterFactory createParameterFactory(String factoryClassName)
    throws OperationFailureException {
    Class<?> factoryClass;
    try {
      factoryClass = Class.forName(factoryClassName);
    }
    catch (ClassNotFoundException e) {
      throw new OperationFailureException("A classe {0} no foi encontrada.",
        factoryClassName);
    }
    if (!ParameterFactory.class.isAssignableFrom(factoryClass)) {
      throw new OperationFailureException(
        "A classe {0} no implementa a interface necessria {1}.",
        factoryClassName, ParameterFactory.class.getName());
    }
    Constructor<?> factoryConstructor;
    try {
      factoryConstructor = factoryClass.getConstructor();
    }
    catch (NoSuchMethodException e) {
      throw new OperationFailureException(
        "No existe um construtor vazio em {0}.", factoryClassName);
    }
    ParameterFactory factory;
    try {
      factory = (ParameterFactory) factoryConstructor.newInstance();
    }
    catch (InstantiationException e) {
      throw new OperationFailureException(MessageFormat.format(
        "No foi possvel construir a fbrica {0}.", factoryClassName), e);
    }
    catch (IllegalAccessException e) {
      throw new OperationFailureException(MessageFormat.format(
        "Construtor da fbrica {0} no aceita os parmetros usados.",
        factoryClassName), e);
    }
    catch (InvocationTargetException e) {
      throw new OperationFailureException(MessageFormat.format(
        "Erro ao construir a fbrica de parmetros {0}.", factoryClassName), e
        .getTargetException());
    }
    return factory;
  }

}
