/*
 * $Id$
 */
package csbase.logic.applicationservice;

import java.io.InputStream;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.PropertyResourceBundle;
import java.util.Set;
import java.util.TreeSet;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.core.lng.LNG.TranslationListener;

/**
 * Esta classe s existe pois precisa-se exportar o mtodo
 * <code>setParent</code> (que  protected na classe original).
 * 
 * @author Tecgraf/PUC-Rio
 */
public class AppPropertyResourceBundle {

  /** O locale usado para carregar esse bundle */
  private Locale locale;

  /** O bundle pai */
  private AppPropertyResourceBundle parent;

  /** Pacote de propriedades */
  private PropertyResourceBundle bundle;

  /** O registro da aplicao */
  private ApplicationRegistry appReg;

  /**
   * Construtor padro.
   * 
   * @param stream o stream de entrada.
   * @param locale o locale usado para esse bundle
   * @param appReg o registro da aplicao associada e esse bundle
   * @throws java.io.IOException se houver priblemas no I/O.
   */
  public AppPropertyResourceBundle(final InputStream stream, Locale locale,
    ApplicationRegistry appReg) throws java.io.IOException {
    bundle = new PropertyResourceBundle(stream);
    this.locale = locale;
    this.appReg = appReg;
  }

  /**
   * Mtodo para publicar ajuste do bundle-pai que em
   * <code>PropertyResourceBundle</code>  protected.
   * 
   * @param parent o bundle.
   */
  public void setParent(final AppPropertyResourceBundle parent) {
    this.parent = parent;
  }

  /**
   * Verifica se o bundle (ou seus pais) possuem uma determina chave.
   * 
   * @param key a chave procurada
   * @return {@code true} se a chave foi encontrada e {@code false} caso
   *         contrrio
   */
  public boolean containsKey(String key) {
    if (bundle.containsKey(key)) {
      return true;
    }
    if (parent == null) {
      return false;
    }
    return parent.containsKey(key);
  }

  /**
   * Obtm um texto a partir de uma chave. Se no encontrar nesse bundle,
   * procura em dos seus pais, na hierarquia.
   * 
   * @param key a chave
   * @return o texto
   */
  public String getString(String key) {
    TranslationListener callback = null;
    try {
      String text = bundle.getString(key);
      callback = appReg.getTranslationListener();
      if (LNG.getLocale() != null && callback != null
        && !inTheSameIdiom(locale, LNG.getLocale())) {
        System.err
          .println("[APP] Chave " + key + " exibida como sendo " + text);
        return callback.keyNotFoundInDefaultLanguage(key, text);
      }
      return text;
    }
    catch (MissingResourceException ex1) {
      if (parent == null) {
        return missingKeyMsg(key, callback);
      }
      try {
        return parent.getString(key);
      }
      catch (MissingResourceException ex2) {
        return missingKeyMsg(key, callback);
      }
    }
  }

  /**
   * Verifica se dois locales correspondem ao mesmo idioma.
   * 
   * @param locale um locale
   * @param other o outro locale
   * @return se a lingua e o pas correspondem
   */
  private static boolean inTheSameIdiom(Locale locale, Locale other) {
    return locale.getCountry().equals(other.getCountry())
      && locale.getLanguage().equals(other.getLanguage());
  }

  /**
   * Retorna texto padro para chave no encontrada. Imprime mensagem de erro na
   * sada adequada.
   * 
   * @param key chave no encontrada.
   * @param callback o objeto que implementa a interface
   *        <code>NotTranslationInterface</code> para tratar chaves no
   *        encontradas nos bundles carregados.
   * 
   * @return texto padro para chave no encontrada.
   */
  private String missingKeyMsg(String key, TranslationListener callback) {
    String defaultText = "<<" + key + ">>";
    if (callback != null) {
      return callback.keyNotFound(key, defaultText);
    }
    System.err.println("[APP] Chave no encontrada: " + key);
    return defaultText;
  }

  /**
   * Obtm o locale desse bundle.
   * 
   * @return o locale usado na carga desse bundle
   */
  public Locale getLocale() {
    return this.locale;
  }

  /**
   * Retorna uma lista com todas as chaves de bundle de idioma carregados para e
   * aplicao.
   * 
   * @return a lista com todas chaves.
   */
  public Set<String> allKeys() {
    Set<String> allKeys = new TreeSet<String>();
    if (bundle != null) {
      allKeys.addAll(bundle.keySet());
    }
    if (parent != null) {
      allKeys.addAll(parent.allKeys());
    }
    return allKeys;
  }
}
