/*
 * $Id$
 */

package csbase.logic;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import csbase.exception.project.FileLockedException;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;
import tecgraf.ftc.common.exception.FailureException;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;

/**
 * Modela a viso que um cliente tem de um arquivo ou diretrio de um projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ClientProjectFile implements ClientFile, Serializable,
  Comparable<ClientProjectFile> {

  /**
   * Canal, no  enviado quando o CPF  serializado.
   */
  private transient SyncRemoteFileChannel channel;

  /**
   * Id do usurio que criou o arquivo.
   */
  private final Object createdBy;

  /**
   * Data de criao.
   */
  private final long creationDate;

  /**
   * Indicativo de diretrio.
   */
  private boolean isDirectory;

  /**
   * Indicativo de "em construo"
   */
  private boolean isUnderConstruction;

  /**
   * Indica se o arquivo est bloqueado.
   */
  private boolean isLocked;

  /**
   * Indica se arquivo est sendo movido.
   */
  private boolean isMoving;

  /**
   * Data de modificao.
   */
  private long modificationDate;

  /**
   * Nome do arquivo.
   */
  private String name;

  /**
   * Caminho para o arquivo, a partir da raiz.
   */
  private String[] path;

  /**
   * Nmero de pedidos de abertura de canais.
   */
  private int numberOfChannelOpens;

  /**
   * Pai (diretrio)
   */
  private ClientProjectFile parent;

  /**
   * Identirficador do projeto.
   */
  private final Object projectId;

  /**
   * Lista de arquivos-filho
   */
  private ClientProjectFileList children = null;

  /**
   * Lock utilizado para manter o sincronismo de threads no acesso ao childList.
   */
  private Lock childrenLock = new ReentrantLock();

  /** Indica se o arquivo est atualizado em relao ao servidor. */
  private boolean updated;

  /**
   * Indica se o arquivo  atualizvel.
   */
  private boolean updatable;

  /** Indica o nome do usurio que agendou a a atualizao do arquivo. */
  private String updateUserLogin;

  /**
   * Indica o intervalo pelo qual o arquivo (somentes os atualizveis) est
   * sendo atualizado.
   * <p>
   * OBS: Para arquivos no atualizveis este valor  sempre 0 (zero).
   */
  private long updateInterval;

  /**
   * Tamanho
   */
  private long size;

  /**
   * Estado
   */
  private boolean opened;

  /**
   * Tipo
   */
  private String type;

  /**
   * Flag que indica se o n possui filhos.
   */
  private boolean hasChildren;

  /**
   * Retorna o nome do arquivo.
   * 
   * @return Nome do arquivo.
   */
  @Override
  public String getName() {
    return this.name;
  }

  /**
   * Muda o nome do arquivo.
   * 
   * @param name novo nome do arquivo.
   */
  public void setName(String name) {
    this.name = name;
    path[path.length - 1] = name;
  }

  /**
   * Muda o tamanho do arquivo.
   * 
   * @param size novo tamanho do arquivo.
   */
  public void setSize(long size) {
    this.size = size;
  }

  /**
   * Muda a data de modificao do arquivo.
   * 
   * @param modificationDate nova data de modificao do arquivo.
   */
  public void setModificationDate(long modificationDate) {
    this.modificationDate = modificationDate;
  }

  /**
   * Retorna o id do projeto.
   * 
   * @return O id do projeto.
   */
  public Object getProjectId() {
    return this.projectId;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getStringPath() {
    String[] pathArray = getPath();
    if ((pathArray == null) || (pathArray.length <= 0)) {
      return "";
    }
    StringBuilder path = new StringBuilder();
    for (int i = 0; i < (pathArray.length - 1); i++) {
      path.append(pathArray[i]);
      path.append('/');
    }
    return path.append(pathArray[pathArray.length - 1]).toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] getPath() {
    return path.clone();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return this.type;
  }

  /**
   * Informa o a data de criao arquivo.
   * 
   * @return a data de criao do arquivo em Milesegundos.
   */
  public long getCreationDate() {
    return this.creationDate;
  }

  /**
   * Retorna o usurio que criou esse arquivo.
   * 
   * @return O nome do usurio que criou esse arquivo.
   */
  public Object whoCreated() {
    return this.createdBy;
  }

  /**
   * {@inheritDoc}
   * 
   * Este mtodo no deve ser utilizado no servidor pois o valor da propriedade
   * a que ele referencia s ser atribuido no cliente.
   */
  @Override
  public ClientProjectFile getParent() {
    return this.parent;
  }

  /**
   * Redefine o pai do arquivo. Tambm altera a path para ficar consistente.
   * 
   * @param newParent novo pai
   */
  public void setParent(ClientProjectFile newParent) {
    this.parent = newParent;
  }

  /**
   * Corrige o path e o dos descendentes de acordo com o path do parent
   * 
   * @throws RemoteException em caso de erro de comunicao.
   */
  public void updatePath() throws RemoteException {
    if (getParent() == null) {
      return;
    }
    String[] parentPath = getParent().getPath();
    String[] newPath = new String[parentPath.length + 1];
    for (int n = 0; n < parentPath.length; n++) {
      newPath[n] = parentPath[n];
    }
    newPath[newPath.length - 1] = path[path.length - 1];
    path = newPath;

    if (isDirectory()) {
      ClientProjectFile[] children = getChildren();
      if (children != null) {
        for (int i = 0; i < children.length; i++) {
          ClientProjectFile child = children[i];
          child.updatePath();
        }
      }
    }
  }

  /**
   * Reseta a lista de filhos.
   */
  public void resetChildren() {
    children = null;
  }

  /**
   * Retorna a lista de arquivos deste diretrio, mas no vai ao servidor buscar
   * os filhos se estes ainda no tiverem sido carregados.
   * 
   * @return a lista de arquivos deste diretrio, ou <code>null</code> se os
   *         filhos ainda no foram obtidos do servidor
   */
  public ClientProjectFile[] getLocalChildren() {
    if (children == null) {
      return null;
    }
    return children.toArray();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientProjectFile[] getChildren() throws RemoteException {
    return getChildren(false, false);
  }

  /**
   * Retorna a lista de filhos deste diretrio.
   * 
   * @return lista de filhos do diretrio (pode ser <code>null</code>)
   */
  ClientProjectFileList getChildrenList() {
    return children;
  }

  /**
   * Obtm os filhos do diretrio corrente, opcionalmente descartando o cache
   * local e obtendo toda a subrvore recursivamente.
   * 
   * @param discardLocal <code>true</code> se o cache local deve ser descartado
   * @param recursive <code>true</code> se a subrvore deve ser obtida
   *        recursivamente
   * @return array com os filhos deste diretrio
   * @throws RemoteException se houver falha de RMI
   */
  public ClientProjectFile[] getChildren(boolean discardLocal, boolean recursive)
    throws RemoteException {
    if (!isDirectory()) {
      throw new RemoteException("Arquivo no  um diretrio");
    }
    buildChildListIfNull(discardLocal, recursive);
    return children.toArray();
  }

  /**
   * Preenche a lista de filhos do diretrio corrente. Por default, se j existe
   * uma lista (atualizada ou no), nada  feito. Porm, se o parmetro
   * <code>discardLocal</code> for <code>true</code> esta ser regerada.
   * 
   * @param discardLocal <code>true</code> se o cache local deve ser descartado
   * @param recursive <code>true</code> se a subrvore deve ser obtida
   *        recursivamente
   * 
   * @throws RemoteException se houver falha de RMI
   */
  private void buildChildListIfNull(boolean discardLocal, boolean recursive)
    throws RemoteException {
    // Se no for um diretrio no vai ao servidor buscar os filhos.
    if (!this.isDirectory()) {
      return;
    }
    childrenLock.lock();
    try {
      if (discardLocal || children == null) {
        // DEBUG
        //        System.out.println(String.format(
        //          "ClientProjectFile '%s' (%s) : indo ao servidor... (%s)", name,
        //          getObjectID(this), discardLocal ? "discardLocal=true"
        //            : "children = null"));

        final ProjectServiceInterface projectService =
          ClientRemoteLocator.projectService;
        final ClientProjectFile[] newChildren =
          projectService.getChildren(this.getProjectId(), this.getPath(),
            recursive);
        children = new ClientProjectFileList(this);
        children.add(true, true, newChildren);
      }
      else {
        // DEBUG
        //        System.out
        //          .println(String
        //            .format(
        //              "ClientProjectFile '%s' (%s) : usando cache local (children NOT null)",
        //              name, getObjectID(this)));
      }
    }
    finally {
      childrenLock.unlock();
    }
    // DEBUG
    //    validateChildrenParents();
  }

  //  /**
  //   * Valida os pais dos filhos diretos deste n. Usado para depurao de
  //   * {@link #buildChildListIfNull(boolean, boolean)}.
  //   * <p>
  //   * <b>ATENO: no habilitar no cdigo de produo.</b>
  //   */
  //  private void validateChildrenParents() {
  //    for (ClientProjectFile child : children) {
  //      if (child.parent != this) {
  //        System.err.println(String.format("%s : parent = %s deveria ser %s",
  //          child.name, getObjectID(child.parent), getObjectID(parent)));
  //      }
  //      else {
  //        System.out.println(child.name + " OK");
  //      }
  //    }
  //  }
  //
  //  /**
  //   * Obtm o ID original de um objeto (usado para depurao pelo mtodo
  //   * {@link #buildChildListIfNull(boolean, boolean)} ).
  //   * 
  //   * @param o objeto
  //   * @return ID original do objeto
  //   */
  //  private String getObjectID(Object o) {
  //    return Integer.toHexString(System.identityHashCode(o));
  //  }

  /**
   * Retorna a lista de arquivos deste diretrio do tipo especificado.
   * 
   * @param fileType Tipo dos arquivos que devem ser retornados.
   * @return A lista de arquivos deste diretrio.
   * @throws RemoteException Erro na comunicao com o servidor.
   */
  public ClientProjectFile[] getChildren(String fileType)
    throws RemoteException {
    ClientProjectFile[] myChildren = getChildren();
    if (fileType == null) {
      return myChildren;
    }
    List<ClientProjectFile> aux = new ArrayList<ClientProjectFile>();
    for (ClientProjectFile child : myChildren) {
      if (child.getType().equals(fileType)) {
        aux.add(child);
      }
    }
    return aux.toArray(new ClientProjectFile[aux.size()]);
  }

  /**
   * Redefine a lista de filhos deste diretrio.
   * 
   * @param clientSide <code>true</code> se o arquivo est sendo inserido no
   *        cliente. Se este parmetro for <code>false</code> o parmetro
   *        <code>fixParentsRecursively</code> ser ignorado.
   * @param fixParentsRecursively <code>true</code> se as referncias para os
   *        pais devem ser corrigidas recursivamente. Neste caso, os novos
   *        filhos apontaro para seu novo pai, e este processo se repetir
   *        recursivamente.<br>
   *        ATENO: as correes afetam apenas os novos filhos, os filhos j
   *        existentes no so alterados. Se este parmetro for
   *        <code>false</code>, apenas as referncias dos novos filhos diretos
   *        sero corrigidas.<br>
   * @param newChildren a nova lista de filhos
   */
  public void setChildren(boolean clientSide, boolean fixParentsRecursively,
    ClientProjectFile[] newChildren) {
    children = new ClientProjectFileList(this);
    children.add(clientSide, fixParentsRecursively, newChildren);
  }

  /**
   * Obtm um {@link ClientProjectFile arquivo} deste diretrio ou {@code null}
   * caso ele no seja encontrado ou este arquivo no represente um diretrio.
   * 
   * @param childName - path para o filho, a partir deste diretrio
   * 
   * @return o {@link ClientProjectFile filho} deste diretrio ou {@code null}
   *         caso ele no seja encontrado ou este arquivo no represente um
   *         diretrio.
   * 
   * @throws RemoteException se houver falha de RMI
   * 
   * @see csbase.logic.ClientProjectFile#getChild(java.lang.String)
   */
  public ClientProjectFile getChild(String childName) throws RemoteException {
    childrenLock.lock();
    try {
      buildChildListIfNull(false, false);
      /*
       * utiliza o mtodo getChildHelper(String) para buscar o arquivo. Se este
       * no for encontrado, considerando que haja uma inconsistncia entre o
       * cliente e o servidor, limpa a memria e tenta novamente.
       */
      ClientProjectFile child = getChildHelper(childName);
      if (null == child) {
        final ProjectServiceInterface projectService =
          ClientRemoteLocator.projectService;
        child =
          projectService.getChild(this.getProjectId(), this.getPath(),
            childName);
        /*
         * [CSBASE-4199] Teste temporrio at resolver o caso do diretrio que 
         * retornado como filho dele mesmo, se o mtodo getChild(String) for
         * chamado com valor vazio -> "".
         */
        if (child != null && !child.equals(this)) {
          children.add(true, true, child);
        }
      }

      return child;
    }
    finally {
      childrenLock.unlock();
    }
  }

  /**
   * Obtm um {@link ClientProjectFile arquivo} deste diretrio ou {@code null}
   * caso ele no seja encontrado ou este arquivo no represente um diretrio.
   * <p>
   * O que difere este mtodo do {@link #getChild(String)}  que apenas se este
   * diretrio nunca tiver carregado seus filhos, que ele ir procurar no
   * servidor. A partir desta carga este mtodo ir confiar no mtodo
   * {@link #addChild(boolean, boolean, ClientProjectFile)} para garantir a
   * consistncia com o servidor.
   * 
   * @param childName nome do filho
   * 
   * @return o {@link ClientProjectFile filho} deste diretrio ou {@code null}
   *         caso ele no seja encontrado ou este arquivo no represente um
   *         diretrio.
   * 
   * @throws RemoteException se houver falha de RMI
   */
  private ClientProjectFile getChildHelper(String childName)
    throws RemoteException {
    final ClientProjectFile[] myChildren = getChildren();
    for (ClientProjectFile file : myChildren) {
      if (file.name.equals(childName)) {
        return file;
      }
    }

    return null;
  }

  /**
   * Adiciona um novo arquivo neste diretrio se ele ainda no existir. Aps a
   * incluso, os arquivos deste diretrio so ordenados por nome.
   * 
   * @param clientSide <code>true</code> se o arquivo est sendo inserido no
   *        cliente. Se este parmetro for <code>false</code> o parmetro
   *        <code>fixParentsRecursively</code> ser ignorado.
   * @param fixParentsRecursively <code>true</code> se as referncias para os
   *        pais devem ser corrigidas recursivamente. Neste caso, os novos
   *        filhos apontaro para seu novo pai, e este processo se repetir
   *        recursivamente.<br>
   *        ATENO: as correes afetam apenas os novos filhos, os filhos j
   *        existentes no so alterados. Se este parmetro for
   *        <code>false</code>, apenas as referncias dos novos filhos diretos
   *        sero corrigidas.
   * @param newChild O arquivo (ou diretrio) a ser includo
   * 
   * @throws RemoteException se houver falha de RMI
   */
  public void addChild(boolean clientSide, boolean fixParentsRecursively,
    ClientProjectFile newChild) throws RemoteException {
    buildChildListIfNull(false, false);
    childrenLock.lock();
    try {
      /*
       * como o novo n provavelmente veio do servidor, garantimos que os ns da
       * sua subrvore estejam com as referncias corretas para seus pais
       */
      children.addIfDoesntExist(clientSide, fixParentsRecursively, newChild);
      hasChildren = true;
    }
    finally {
      childrenLock.unlock();
    }
  }

  /**
   * Obtm o ndice de um arquivo dentro de um diretrio
   * 
   * @param child O arquivo (ou diretrio) do qual deseja-se o ndice
   * 
   * @return O ndice do arquivo ou -1 caso o arquivo no esteja presente na
   *         lista de filhos.
   * @throws RemoteException se houver falha de RMI
   */
  public int getChildIndex(ClientProjectFile child) throws RemoteException {
    final ClientProjectFile[] myChildren = getChildren();
    final int sz = myChildren.length;
    for (int i = 0; i < sz; i++) {
      ClientProjectFile file = myChildren[i];
      if (child.equals(file)) {
        return i;
      }
    }
    return -1;
  }

  /**
   * Remove um arquivo deste diretrio. Aps a incluso, os arquivos deste
   * diretrio so ordenados por nome.
   * 
   * @param child O arquivo (ou diretrio) a ser removido
   * 
   * @return O ndice do arquivo removido ou -1 caso ocorra algum problema ao
   *         remover.
   * @throws RemoteException se houver falha de RMI
   */
  public int removeChild(ClientProjectFile child) throws RemoteException {
    buildChildListIfNull(false, false);

    childrenLock.lock();
    try {
      int index = children.remove(child);
      if (children.isEmpty()) {
        hasChildren = false;
      }
      return index;
    }
    finally {
      childrenLock.unlock();
    }
  }

  /**
   * Cria um arquivo dentro deste diretrio.
   * 
   * @param fileName Nome do arquivo a ser criado.
   * @param fileType Tipo do arquivo a ser criado.
   * 
   * @throws RemoteException falha de rmi
   */
  public void createFile(String fileName, String fileType)
    throws RemoteException {
    ClientRemoteLocator.projectService.createFile(this.projectId, getPath(),
      fileName, fileType);
  }

  /**
   * Cria vrios arquivos simultaneamente dentro deste diretrio.
   * 
   * @param fileInfoList lista de dados dos arquivos a serem criados.
   * 
   * @throws RemoteException falha de rmi
   */
  public void createFiles(List<ProjectFileInfo> fileInfoList)
    throws RemoteException {
    ClientRemoteLocator.projectService.createFiles(this.projectId, getPath(),
      fileInfoList);
  }

  /**
   * Troca o nome deste arquivo.
   * 
   * @param fileName Novo nome do arquivo.
   * @throws RemoteException falha de rmi
   */
  public void rename(String fileName) throws RemoteException {
    ClientRemoteLocator.projectService.renameFile(this.projectId, getPath(),
      fileName);
  }

  /**
   * Copia este arquivo para outro diretrio.
   * 
   * @param directory Diretrio destino.
   * 
   * @throws RemoteException Falha de rmi
   */
  public void copy(ClientProjectFile directory) throws RemoteException {
    Object targetProjectId = directory.getProjectId();
    Object sourceProjectId = this.projectId;
    if (targetProjectId.equals(sourceProjectId)) {
      ClientRemoteLocator.projectService.copyFile(sourceProjectId, getPath(),
        directory.getPath());
    }
    else {
      ClientRemoteLocator.projectService.copyFile(sourceProjectId, getPath(),
        targetProjectId, directory.getPath());
    }
  }

  /**
   * Move este arquivo para outro diretrio.
   * 
   * @param directory Diretrio destino.
   * 
   * @throws RemoteException erro no acesso ao servidor
   */
  public void move(ClientProjectFile directory) throws RemoteException {
    Object targetProjectId = directory.getProjectId();
    Object sourceProjectId = this.projectId;
    if (targetProjectId.equals(sourceProjectId)) {
      ClientRemoteLocator.projectService.moveFile(this.projectId, getPath(),
        directory.getPath());
    }
    else {
      ClientRemoteLocator.projectService.moveFile(sourceProjectId, getPath(),
        targetProjectId, directory.getPath());
    }
  }

  /**
   * Remove este arquivo. Aps chamado este mtodo, o objeto que representa o
   * arquivo no poder mais ser utilizado.
   * 
   * @throws RemoteException falha de comunicao com o servidor
   */
  public void remove() throws RemoteException {
    ClientRemoteLocator.projectService.removeFile(this.projectId, getPath());
  }

  /**
   * Atualiza as informaes desse arquivo buscando os dados reais no servidor.
   * 
   * @throws RemoteException Falha de rmi
   */
  public void updateInfo() throws RemoteException {
    ClientProjectFileInfo info =
      ClientRemoteLocator.projectService.getUpdatedFileInfo(this.projectId,
        getPath());
    this.type = info.getType();
    this.size = info.getSize();
    this.modificationDate = info.getModificationDate();
    this.isUnderConstruction = info.isUnderConstruction();
    this.isLocked = info.isLocked();
  }

  /**
   * Indica se este arquivo  um diretrio.
   * 
   * @return Verdadeiro caso seja um diretrio e falso caso contrrio.
   */
  @Override
  public boolean isDirectory() {
    return this.isDirectory;
  }

  /**
   * Verifica se o n possui filhos. Se o {@link ClientProjectFile} foi criado
   * pelo servidor a partir de um diretrio que contm filhos, este mtodo pode
   * retornar <code>true</code> mesmo que a lista de filhos no tenha sido
   * obtida ainda do servidor. Esta informao  tipicamente usada pela rvore
   * do projeto para indicar se um n  folha.
   * 
   * @return <code>true</code> se o n possui filhos
   */
  public boolean hasChildren() {
    return hasChildren;
  }

  /**
   * Define se o n possui filhos.
   * 
   * @param hasChildren flag indicando se o n possui filhos
   */
  public void setHasChildren(boolean hasChildren) {
    this.hasChildren = hasChildren;
  }

  /**
   * Indica se este arquivo  um arquivo em construo.
   * 
   * @return Verdadeiro caso seja um arquivo em construo e falso caso
   *         contrrio.
   */
  public boolean isUnderConstruction() {
    return this.isUnderConstruction;
  }

  /**
   * Muda o estado "em construo" do arquivo.
   * 
   * @param isUnderConstruction true se o arquivo estiver em construo.
   */
  public void setUnderConstruction(boolean isUnderConstruction) {
    this.isUnderConstruction = isUnderConstruction;
  }

  /**
   * Muda o estado do arquivo para bloqueado ou desbloqueado
   * 
   * @param isLocked true se o arquivo estiver bloqueado
   */
  public void setLocked(boolean isLocked) {
    this.isLocked = isLocked;
  }

  /**
   * Obtm a descrio atual deste arquivo.
   * 
   * @return O texto com a descrio do arquivo.
   * 
   * @throws RemoteException falha de rmi
   */
  public String getDescription() throws RemoteException {
    return ClientRemoteLocator.projectService.getFileDescription(
      this.projectId, getPath());
  }

  /**
   * Muda a descrio atual deste arquivo.
   * 
   * @param text novo texto da descrio do arquivo.
   * 
   * @throws RemoteException falha de rmi
   */
  public void setDescription(String text) throws RemoteException {
    ClientRemoteLocator.projectService.setFileDescription(this.projectId,
      getPath(), text);
  }

  /**
   * Anexa um texto  descrio atual deste arquivo. Permite a gerao de
   * histricos de modificao.
   * 
   * @param text texto a ser anexado.
   * 
   * @throws RemoteException falha de rmi
   */
  public void appendDescription(String text) throws RemoteException {
    ClientRemoteLocator.projectService.appendFileDescription(this.projectId,
      getPath(), text);
  }

  /**
   * Informa o tamanho corrente deste arquivo. Reflete o nmero corrente de
   * bytes do arquivo.
   * 
   * @return Tamanho corrente do arquivo.
   */
  @Override
  public long size() {
    return this.size;
  }

  /**
   * Calcula o tamanho total de um arquivo. Se o arquivo  um diretrio, todos
   * os filhos (diretos e indiretos) so carregados do servidor para se calcular
   * o tamanho total do diretrio (a soma do tamanho do prprio diretrio mais a
   * soma do tamanho de todos os filhos). Se um fitro  passado, o tamanho de um
   * diretrio  calculado considerando-se apenas os arquivos filhos aceitos
   * pelo filtro. Se o arquivo no  um diretrio, esse mtodo  igual a size()
   * e o filtro  desconsiderado.
   * 
   * @param filter Define que arquivos so usados para calcular o tamanho total
   *        de um diretrio.
   * 
   * @return O tamanho total do arquivo.
   * @throws RemoteException Quando no consegue carregar a subrvore de um
   *         diretrio do servidor.
   */
  public long getTotalSize(ProjectFileFilter filter) throws RemoteException {
    if (this.isDirectory()) {
      // Acessa o servidor para atualizar todos os filhos recursivamente.
      this.getChildren(true, true);
      return this.getLocalTotalSize(filter);
    }
    return this.size();
  }

  /**
   * Calcula o tamanho total de um arquivo. Se o arquivo  um diretrio, o
   * tamanho do diretrio  somado ao tamanho de todos os filhos (diretos e
   * indiretos) para se calcular o tamanho total do diretrio. Esse mtodo no
   * carrega os filhos do servidor se eles no tiverem sido carregados ainda. Se
   * um fitro  passado, o tamanho de um diretrio  calculado considerando-se
   * apenas os arquivos filhos aceitos pelo filtro. Se o arquivo no  um
   * diretrio, esse mtodo  igual a size() e o filtro  desconsiderado.
   * 
   * @param filter Define que arquivos so usados para calcular o tamanho total
   *        de um diretrio.
   * 
   * @return O tamanho da transferncia.
   */
  public long getLocalTotalSize(ProjectFileFilter filter) {
    long transferSize = 0;
    if (this.isDirectory()) {
      transferSize = this.size();
      // Obtm os filhos localmente.
      final ClientProjectFile[] myChildren = this.getLocalChildren();
      if (myChildren != null) {
        for (final ClientProjectFile child : myChildren) {
          if (filter == null || filter.accept(child)) {
            transferSize += child.getLocalTotalSize(filter);
          }
        }
      }
    }
    else {
      transferSize = this.size();
    }
    return transferSize;
  }

  /**
   * Informa a data da ltima modificao deste arquivo (formato numrico).
   * 
   * @return data da ltima modificao deste arquivo (formato numrico)
   */
  @Override
  public long getModificationDate() {
    return this.modificationDate;
  }

  /**
   * Indica se o arquivo est atualizado em relao ao servidor.
   * 
   * @return <code>true</code> se est atualizado
   */
  public boolean isUpdated() {
    return updated;
  }

  /**
   * Informa se o arquivo est atualizado em relao ao servidor.
   * 
   * @param updated <code>true</code> se est atualizado.
   */
  public void setUpdated(boolean updated) {
    this.updated = updated;
  }

  /**
   * Obtm o n filho indicado pelo caminho fornecido.
   * 
   * @param path Array de Strings com o caminho para o arquivo, <b>no</b>
   *        incluindo o nome do diretrio corrente.
   * @return O n filho pedido, ou null caso este no exista.
   * @throws RemoteException se houver falha de RMI
   */
  public ClientProjectFile getChild(String[] path) throws RemoteException {
    if (path == null || path.length == 0) {
      return null;
    }
    if (!isDirectory) {
      return null;
    }
    ClientProjectFile child = this;
    for (int i = 0; i < path.length; i++) {
      child = child.getChild(path[i]);
      if (child == null) {
        return null;
      }
    }
    return child;
  }

  /**
   * Obtm o filho do diretrio corrente a partir do {@link ProjectFileInfo} que
   * o representa.
   * 
   * @param prjFileInfo - propriedades do arquivo a ser obtido
   * @return o arquivo solicitado ou null caso no exista ou se o arquivo
   *         corrente no  um diretrio
   * @throws RemoteException Se houver falha de comunicao.
   */
  public ClientProjectFile getChild(ProjectFileInfo prjFileInfo)
    throws RemoteException {
    return getChild(prjFileInfo.getPath());
  }

  /**
   * Verifica se o arquivo existe no servidor.
   * 
   * @return true se o arquivo existe no servidor.
   * @throws RemoteException se houver falha de RMI
   */
  @Override
  public boolean exists() throws RemoteException {
    return ClientProjectFile.existsFile(this.projectId, getPath());
  }

  /**
   * Verifica se o arquivo existe no servidor.
   * 
   * @param projectId id do projeto
   * @param path o path do arquivo.
   * 
   * @return true se o arquivo existe no servidor.
   * @throws RemoteException se houver falha de RMI
   */
  static public boolean existsFile(final Object projectId, final String[] path)
    throws RemoteException {
    return ClientRemoteLocator.projectService.existsFile(projectId, path);
  }

  /**
   * Mtodo interno para decomposio de um caminho de arquivo (<i>path</i>) em
   * um array de strings; onde cada elemento  montado no separador
   * <code>"/"</code>.
   * 
   * @param path o caminho lgico do arquivo.
   * 
   * @return o array.
   */
  static public String[] splitPath(final String path) {
    String[] pathArray = path.split("/");
    int i = 0;
    while ((i < pathArray.length) && pathArray[i].equals("")) {
      i++;
    }
    if (i > 0) {
      final String[] newPath = new String[pathArray.length - i];
      System.arraycopy(pathArray, i, newPath, 0, newPath.length);
      pathArray = newPath;
    }
    return pathArray;
  }

  /**
   * Ajusta o tamanho corrente deste arquivo.
   * 
   * @param newSize o novo tamanho a ser ajustado.
   * 
   * @throws IOException Em caso de erro.
   */
  public void truncate(long newSize) throws IOException {
    try {
      this.channel.setSize(newSize);
    }
    catch (Exception e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * Cria um canal de acesso ao contedo deste arquivo.
   * 
   * @param readOnly Indica se o acesso ser de leitura ou leitura e escrita.
   * @return O canal construdo e j aberto.
   * @throws RemoteException Em caso de falha na operao.
   */
  private SyncRemoteFileChannel openChannel(boolean readOnly)
    throws RemoteException {
    RemoteFileChannelInfo info =
      ClientRemoteLocator.projectService.openFileChannel(this.projectId,
        getPath(), readOnly);
    SyncRemoteFileChannel srfc =
      new SyncRemoteFileChannel(info.getIdentifier(), info.isWritable(),
        info.getHost(), info.getPort(), info.getKey());
    try {
      srfc.open(readOnly);
      return srfc;
    }
    catch (Exception e) {
      this.channel = null;
      throw new RemoteException(e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void open(final boolean readOnly) throws RemoteException {
    updateInfo(); // atualiza info do arquivo antes de abrir
    if (!this.opened) {
      this.channel = openChannel(readOnly);
      this.numberOfChannelOpens = 1;
      this.opened = true;
    }
    else {
      this.numberOfChannelOpens++;
    }
  }

  /**
   * Tenta obter lock exclusivo de escrita para o arquivo no servidor. Ver
   * ProjectService#acquireExclusiveLock(Object, String[]).
   * 
   * @return identificador do lock se o lock foi obtido, ou null caso contrrio
   * @throws RemoteException falha de rmi
   */
  public Object acquireExclusiveLock() throws RemoteException {
    return ClientRemoteLocator.projectService.acquireExclusiveLock(
      getProjectId(), getPath());
  }

  /**
   * Tenta obter um lock de escrita exclusivo para o arquivo. Locks exclusivos
   * so obtidos apenas uma vez.
   * 
   * Este mtodo cadastra uma solicitao de lock numa fila priorizada pela
   * ordem de chegada dos pedidos de lock e espera pela notificao de que o
   * lock foi obtido ou seu timeout foi atingido. Quando o lock  obtido, o
   * mtodo FileLockListenerInterface.fileLocked  chamado. Quando o tempo
   * mximo de espera do lock  atingido e o lock no foi obtido o mtodo
   * FileLockListenerInterface.fileLockExpired  chamado.
   * 
   * @param listener observador do lock cadastrado no servidor
   * @param timeout tempo mximo de espera pelo lock em milisegundos. A
   *        constante ProjectServiceInterface.INFINITE_TIMOUT foi criada para
   *        timeout infinito.
   * @return identificador do lock se o lock foi obtido, ou null caso contrrio
   * 
   * @throws RemoteException falha de rmi
   */
  public Object acquireExclusiveLock(ClientFileLockListener listener,
    long timeout) throws RemoteException {
    return ClientRemoteLocator.projectService.acquireExclusiveLock(
      getProjectId(), getPath(), listener, timeout);
  }

  /**
   * Tenta obter lock compartilhado de escrita para o arquivo no servidor. Ver
   * ProjectService#acquireExclusiveLock(Object, String[]).
   * 
   * @return identificador do lock se o lock foi obtido, ou null caso contrrio
   * @throws RemoteException falha de rmi
   */
  public Object acquireSharedLock() throws RemoteException {
    return ClientRemoteLocator.projectService.acquireSharedLock(getProjectId(),
      getPath());
  }

  /**
   * Tenta obter lock compartilhado para o arquivo. Locks compartilhados podem
   * ser obtidos mltiplas vezes.
   * 
   * Este mtodo cadastra uma solicitao de lock compartilhado em uma fila
   * priorizada pela ordem de chegada dos pedidos de lock e espera pela
   * notificao de que o lock foi obtido ou seu timeout foi atingido. Quando o
   * lock  obtido, o mtodo FileLockListenerInterface.fileLocked  chamado.
   * Quando o tempo mximo de espera do lock  atingido e o lock no foi obtido
   * o mtodo FileLockListenerInterface.fileLockExpired  chamado.
   * 
   * @param listener observador do lock cadastrado no servidor
   * @param timeout tempo mximo de espera pelo lock em milisegundos. A
   *        constante ProjectServiceInterface.INFINITE_TIMOUT foi criada para
   *        timout infinito.
   * @return o identificador do lock se o lock foi obtido com sucesso, ou null
   *         caso contrrio
   * 
   * @throws RemoteException falha de rmi
   */
  public Object acquireSharedLock(ClientFileLockListener listener, long timeout)
    throws RemoteException {
    return ClientRemoteLocator.projectService.acquireSharedLock(getProjectId(),
      getPath(), listener, timeout);
  }

  /**
   * Libera o lock para o arquivo no servidor. Ver
   * ProjectService#releaseLock(Object, String[], Object).
   * 
   * @param lockId identificador do lock
   * 
   * @return contador de locks ainda ativos sobre o arquivo. Para locks
   *         exclusivos, este contador tem que ser zero se a operao foi bem
   *         sucedida. Para locks compartilhados, pode ainda haver outros locks
   *         ativos.
   * 
   * @throws RemoteException falha de rmi
   */
  public int releaseLock(Object lockId) throws RemoteException {
    return ClientRemoteLocator.projectService.releaseLock(getProjectId(),
      getPath(), lockId);
  }

  /**
   * Retorna boolean indicando se este arquivo est bloqueado.
   * 
   * @return Boolean indicando se este arquivo est bloqueado.
   */
  public boolean isLocked() {
    return isLocked;
  }

  /**
   * Verifica se o usurio corrente  o dono do lock do arquivo.
   * 
   * @return <code>true</code> se o usurio corrente  o dono do lock
   * @throws RemoteException em caso de erro de comunicao.
   */
  public boolean userOwnsLock() throws RemoteException {
    return ClientRemoteLocator.projectService.userOwnsLock(getProjectId(),
      getPath());
  }

  /**
   * Indica se este arquivo est aberto, ou seja, se j foi criado um canal para
   * acesso ao seu contedo.
   * 
   * @return Verdadeiro caso este arquivo esteja aberto, falso caso contrrio.
   */
  public boolean isOpen() {
    return opened;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long position() {
    return this.channel.getPosition();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void position(long newPosition) throws IOException {
    try {
      this.channel.setPosition(newPosition);
    }
    catch (FailureException e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * L uma sequncia de bytes do arquivo. Esse mtodo tenta fazer leituras
   * consecutivas at conseguir a quantidade de bytes solicitada, ou atingir o
   * final do arquivo. Caso no consiga recuperar a quantidade solicitada sem
   * ter atingido o final do arquivo (isto , por problemas de comunicao ou
   * outros), o mtodo lana uma exceo.
   * 
   * @param dst O array a ser preenchido.
   * @param position A posio do arquivo a partir da qual a leitura deve ser
   *        iniciada.
   * @return A quantidade de bytes lidos. Se for menor que o nmero de bytes
   *         solicitados, o final do arquivo foi atingido.
   * 
   * @throws IOException Caso ocorra alguma falha no procedimento.
   */
  @Override
  public int read(byte[] dst, long position) throws IOException {
    try {
      // S  necessrio limitar em avail por causa do erro [FTC-13].
      int length = (int) Math.min(dst.length, channel.getSize() - position);
      return channel.syncRead(dst, 0, length, position);
    }
    catch (FailureException e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * L uma sequncia de bytes do arquivo. Esse mtodo tenta fazer leituras
   * consecutivas at conseguir a quantidade de bytes solicitada, ou atingir o
   * final do arquivo. Caso no consiga recuperar a quantidade solicitada sem
   * ter atingido o final do arquivo (isto , por problemas de comunicao ou
   * outros), o mtodo lana uma exceo.
   * 
   * @param target O array a ser preenchido.
   * @param offset O deslocamento no array a partir do qual os bytes sero
   *        armazenados.
   * @param length A quantidade de bytes a serem lidos do arquivo.
   * @param position A posio do arquivo a partir da qual a leitura deve ser
   *        iniciada.
   * @return A quantidade de bytes lidos. Se for menor que o nmero de bytes
   *         solicitados, o final do arquivo foi atingido.
   * 
   * @throws IOException Caso ocorra alguma falha no procedimento.
   */
  @Override
  public int read(byte[] target, int offset, int length, long position)
    throws IOException {
    try {
      // S  necessrio limitar em avail por causa do erro [FTC-13].
      length = (int) Math.min(length, channel.getSize() - position);
      return channel.read(target, offset, length, position);
    }
    catch (FailureException e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] src, long position) throws IOException,
    FileLockedException {
    try {
      channel.syncWrite(src, 0, src.length, position);
    }
    catch (tecgraf.ftc.common.exception.FileLockedException e) {
      throw new FileLockedException(getName(), isDirectory());
    }
    catch (Exception e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] src, int off, int len, long position)
    throws IOException, FileLockedException {
    try {
      channel.syncWrite(src, off, len, position);
    }
    catch (tecgraf.ftc.common.exception.FileLockedException e) {
      throw new FileLockedException(getName(), isDirectory());
    }
    catch (Exception e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * Envia o contedo de um stream de entrada para o arquivo do projeto.
   * Qualquer contedo que o arquivo do projeto possua ser substitudo pelo
   * contedo sendo enviado. O arquivo do projeto no precisa ser aberto para
   * que este mtodo seja chamado.
   * 
   * @param in O stream de entrada que definir o contedo do arquivo do
   *        projeto.
   * @param chunkSize O tamanho do buffer intermedirio usado na transferncia.
   * @param listener O listener que ser chamado para acompanhar a
   *        transferncia. Esse listener ser chamado, no mximo, a cada buffer
   *        completo que seja transferido. Pode ser nulo.
   * 
   * @throws IOException Se houver falha de I/O.
   * @throws FileLockedException Caso o arquivo do projeto esteja bloqueado para
   *         escrita.
   */
  public void upload(final InputStream in, final long chunkSize,
    final ProjectFileChannelLoadListener listener) throws IOException,
    FileLockedException {
    this.open(false);
    try {
      long transfered = 0;
      while (true) {
        try {
          long part = this.channel.syncTransferFrom(in, transfered, chunkSize);
          if (part < chunkSize) {
            return;
          }
          transfered += part;
          if (listener != null) {
            listener.transferedBytes(transfered, part);
          }
        }
        catch (tecgraf.ftc.common.exception.FileLockedException e) {
          throw new FileLockedException(this.name, this.isDirectory);
        }
        catch (Exception e) {
          throw new IOException(e.getMessage(), e);
        }
      }
    }
    finally {
      this.close(false);
    }
  }

  /**
   * Recupera o contedo completo de um arquivo projeto para um stream de sada.
   * O arquivo do projeto no precisa ser aberto para que este mtodo seja
   * chamado.
   * 
   * @param out O stream de sada que receber o contedo do arquivo do projeto.
   * @param chunkSize O tamanho do buffer intermedirio usado na transferncia.
   * @param listener O listener que ser chamado para acompanhar a
   *        transferncia. Esse listener ser chamado, no mximo, a cada buffer
   *        completo que seja transferido. Pode ser nulo.
   * 
   * @throws IOException Se houver falha de I/O.
   */
  public void download(final OutputStream out, final long chunkSize,
    final ProjectFileChannelLoadListener listener) throws IOException {
    this.open(true);
    try {
      long transfered = 0;
      long size = this.channel.getSize();
      while (transfered < size) {
        try {
          long part = this.channel.syncTransferTo(transfered, chunkSize, out);
          transfered += part;
          if (listener != null) {
            listener.transferedBytes(transfered, part);
          }
        }
        catch (Exception e) {
          throw new RemoteException(e.getMessage(), e);
        }
      }
    }
    finally {
      this.close(false);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close(boolean force) throws RemoteException, IOException {
    if (this.opened) {
      if (--this.numberOfChannelOpens == 0) {
        try {
          this.channel.close();
          this.channel = null;
          this.opened = false;
        }
        catch (FailureException e) {
          if (force) {
            this.channel = null;
            this.opened = false;
          }
          throw new RemoteException(e.getMessage(), e);
        }
      }
    }
  }

  /**
   * Retorna um texto descritivo do arquivo.
   * 
   * @return .
   */
  @Override
  public String toString() {
    return this.name;
  }

  /**
   * Muda o tipo do arquivo localmente.
   * 
   * @param newType O novo tipo do arquivo.
   */
  public void setType(String newType) {
    this.type = newType;
  }

  /**
   * Solicita ao servidor para modificar o tipo do arquivo
   * 
   * @param newType O novo tipo do arquivo.
   * 
   * @throws RemoteException falha de rmi
   */
  public void changeType(String newType) throws RemoteException {
    ClientRemoteLocator.projectService.changeFileType(this.projectId,
      getPath(), newType);
  }

  /**
   * Testa a igualdade de dois arquivos. Dois <code>ClientProjectFile</code> so
   * considerados iguais se seus paths absolutos forem os mesmos.
   * 
   * @param o Objeto a ser comparado com o atual
   * 
   * @return true se ambos os objetos forem considerados iguais
   */
  @Override
  public boolean equals(Object o) {
    if (o == null) {
      return false;
    }
    if (!this.getClass().equals(o.getClass())) {
      return false;
    }
    ClientProjectFile other = (ClientProjectFile) o;
    return this.projectId.equals(other.projectId)
      && Arrays.equals(path, other.path);
  }

  /**
   * Redefine a chave pela qual o arquivo deve ser buscado/inserido em
   * hashtables.
   * 
   * @return chave para insero/busca em hashtables.
   */
  @Override
  public int hashCode() {
    int hash = projectId.hashCode();
    for (String part : path) {
      hash += part.hashCode();
    }
    return hash;
  }

  /**
   * Construtor
   * 
   * @param projectId identificador do projeto
   * @param name nome do arquivo
   * @param path caminho para o arquivo, a partir da raiz
   * @param parent diretrio-pai. Se estiver sendo construido no servidor, este
   *        valor deve ser null.
   * @param type tipo do arquivo
   * @param isDirectory indicativo de diretrio
   * @param hasChildren flag indicando se o n possui filhos. Pode ser
   *        <code>true</code> mesmo que <code>children</code> seja
   *        <code>null</code>
   * @param isUnderConstruction indicaivo de construo
   * @param createdBy usurio que criou o arquivo/diretrio
   * @param creationDate data de criao
   * @param size tamanho do arquivo
   * @param modificationDate data da ltima modificao.
   * @param isLocked indica se o arquivo est bloqueado
   * @param updatable indica se o arquivo  atualizvel.
   * @param updateUserLogin Indica o login do usurio que agendou a atualizao
   *        do arquivo.
   * @param updateInterval Indica o intervalo entre as atualizaes de um
   *        arquivo.
   */
  public ClientProjectFile(Object projectId, String name, String[] path,
    ClientProjectFile parent, String type, boolean isDirectory,
    boolean hasChildren, boolean isUnderConstruction, Object createdBy,
    long creationDate, long size, long modificationDate, boolean isLocked,
    boolean updatable, String updateUserLogin, long updateInterval) {
    this.projectId = projectId;
    this.name = name;
    this.path = path;
    this.parent = parent;
    this.type = type;
    this.isDirectory = isDirectory;
    this.hasChildren = hasChildren;
    this.isLocked = isLocked;
    this.isUnderConstruction = isUnderConstruction;
    this.channel = null;
    this.opened = false;
    this.numberOfChannelOpens = 0;
    this.createdBy = createdBy;
    this.creationDate = creationDate;
    this.size = size;
    this.modificationDate = modificationDate;
    this.updated = true;
    this.updatable = updatable;
    this.updateUserLogin = updateUserLogin;
    this.updateInterval = updateInterval;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int compareTo(ClientProjectFile other) {
    return this.name.compareTo(other.name);
  }

  /**
   * Indica se este arquivo/diretrio est sendo movido de lugar. Esta
   * informao ser utilizada para definir o cone do arquivo/diretrio.
   * 
   * @param isMoving indicativo
   */
  public void setMoving(boolean isMoving) {
    if (this.isMoving != isMoving) {
      this.isMoving = isMoving;
    }
  }

  /**
   * @return true se este arquivo/diretrio e/ou seu diretrio pai estiver sendo
   *         movido. Esta informao ser utilizada para definir o cone do
   *         arquivo/diretrio.
   */
  public boolean isMoving() {
    return this.isMoving || (null != getParent() && getParent().isMoving());
  }

  /**
   * Verifica se algum arquivo de projeto  diretrio.
   * 
   * @param projectFiles Os arquivos que se deseja verificar.
   * 
   * @return true, caso algum dos arquivos de projeto seja diretrio, ou false,
   *         caso contrrio.
   */
  public static boolean hasDirectories(ClientProjectFile[] projectFiles) {
    for (int i = 0; i < projectFiles.length; i++) {
      if (projectFiles[i].isDirectory()) {
        return true;
      }
    }
    return false;
  }

  /**
   * Verifica se algum arquivo de projeto est em construo.
   * 
   * @param projectFiles Os arquivos que se deseja verificar.
   * 
   * @return true, caso algum dos arquivos de projeto esteja em construo, ou
   *         false, caso contrrio.
   */
  public static boolean hasFileUnderConstruction(
    ClientProjectFile[] projectFiles) {
    for (int i = 0; i < projectFiles.length; i++) {
      if (projectFiles[i].isUnderConstruction()) {
        return true;
      }
    }
    return false;
  }

  /**
   * Verifica se o arquivo  atualizvel.
   * 
   * @return {@code true} caso o arquivo seja atualizvel, ou {@code false},
   *         caso contrrio.
   */
  public boolean isUpdatable() {
    return this.updatable;
  }

  /**
   * Obtm o nome do usurio que agendou a a atualizao do arquivo.
   * 
   * @return O nome do usurio que agendou a a atualizao do arquivo.
   */
  public String getUpdateUserLogin() {
    return this.updateUserLogin;
  }

  /**
   * Obtm o intervalo pelo qual o arquivo (somentes os atualizveis) est sendo
   * atualizado.
   * 
   * @return O intervalo pelo qual o arquivo est sendo atulizado. Caso o
   *         arquivo no esteja sendo atualizado ou no seja atualizvel,
   *         retorna 0 (zero).
   */
  public long getUpdateInterval() {
    return this.updateInterval;
  }

  /**
   * Define informaes sobre o agendamento da atualizado de um arquivo.
   * 
   * @param updateUserLogin O login do usurio que agendou a atualizao.
   * @param updateInterval O intervalo pelo qual o arquivo est sendo atulizado.
   *        Caso o arquivo no esteja sendo atulizado ou no seja atualizvel, o
   *        valor deve ser {@literal 0} (zero).
   */
  public void setUpdateInfo(String updateUserLogin, long updateInterval) {
    this.updateUserLogin = updateUserLogin;
    this.updateInterval = updateInterval;
  }

  /**
   * Indica se este arquivo  o root da arvore.
   * 
   * @return Retorna boolean indicando se este arquivo  o root da arvore.
   */
  public boolean isRoot() {
    return parent == null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public InputStream getInputStream() throws IOException {
    SyncRemoteFileChannel srfc = openChannel(true);
    String id = projectId.toString() + "/" + getStringPath();
    return new RemoteFileInputStream(id.getBytes(), srfc);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public OutputStream getOutputStream() throws IOException {
    SyncRemoteFileChannel srfc = openChannel(false);
    String id = projectId.toString() + "/" + getStringPath();
    return new RemoteFileOutputStream(id.getBytes(), srfc);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientFileType getClientFileType() {
    return ClientFileType.REMOTE;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canExecute() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canRead() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canWrite() {
    return true;
  }
}
