package csbase.logic.algorithms.parsers;

import java.util.List;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.EnumerationColumn;
import csbase.logic.algorithms.parameters.EnumerationItem;
import csbase.logic.algorithms.parameters.EnumerationListParameter;
import csbase.logic.algorithms.parameters.EnumerationParameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.TableParameter;

/**
 * Classe abstrata que serve de base para analisadores de parmetros {@code <T>}
 * com valores do tipo enumerao como {@link EnumerationParameter} e
 * {@link EnumerationListParameter}.
 * 
 * @param <T> Tipo do parmetro do qual essa classe faz parsing
 */
public abstract class AbstractEnumerationParameterParser<T extends SimpleParameter<?>>
  extends SimpleParameterParser<T> {

  /**
   * <p>
   * O atributo {@value #ENUMERATION_ELEMENT_MUST_SORT_ATTRIBUTE} dos elementos:
   * </p>
   * <ul>
   * <li>{@link EnumerationColumn} de um {@link TableParameter}</li>
   * <li>{@link EnumerationParameter}</li>
   * </ul>
   * <p>
   * Indica se o itens de enumerao devem ser ordenados.  opcional, o seu
   * valor-padro  {@link #ENUMERATION_ELEMENT_MUST_SORT_DEFAULT_VALUE} e o seu
   * tipo  booleano.
   * </p>
   */
  public static final String ENUMERATION_ELEMENT_MUST_SORT_ATTRIBUTE =
    "ordenar";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #ENUMERATION_ELEMENT_MUST_SORT_ATTRIBUTE} dos elementos:
   * </p>
   * <ul>
   * <li>{@link EnumerationColumn} de um {@link TableParameter}</li>
   * <li>{@link EnumerationParameter}</li>
   * </ul>
   * <p>
   * O seu valor  {@value #ENUMERATION_ELEMENT_MUST_SORT_DEFAULT_VALUE}.
   * </p>
   */
  public static final boolean ENUMERATION_ELEMENT_MUST_SORT_DEFAULT_VALUE =
    true;

  /**
   * A fbrica de itens de enumerao.
   */
  private EnumerationItemFactory enumerationItemFactory;

  /**
   * Cria o parser.
   */
  protected AbstractEnumerationParameterParser() {
    enumerationItemFactory = new EnumerationItemFactory();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public T createSimpleParameter(XmlParser parser, String name, String label,
    String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    List<EnumerationItem> items = enumerationItemFactory.loadEnumerationItems(
      parser, name);
    String defaultItemId = parser.extractAttributeValue(
      PARAMETER_ELEMENT_DEFAULT_VALUE_ATTRIBUTE, null);
    boolean mustSort = parser.extractAttributeValueAsBoolean(
      ENUMERATION_ELEMENT_MUST_SORT_ATTRIBUTE,
      ENUMERATION_ELEMENT_MUST_SORT_DEFAULT_VALUE);
    return createEnumerationParameter(parser, name, label, description,
      isOptional, isVisible, commandLinePattern, mustSort, items,
      defaultItemId);
  }

  /**
   * Cria uma instncia do parmetro de tipo {@code <T>}, a partir dos atributos
   * bsicos de um parmetro do tipo enumerao. As subclasses devem implementar
   * esse mtodo, fazendo a extrao dos demais atributos do parmetro.
   *
   * @param parser Parser xml do configurador.
   * @param name Nome do parmetro.
   * @param label Rtulo do parmetro.
   * @param description Dica do parmetro.
   * @param isOptional Indica se o parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern Padro da linha de comando do parmetro.
   * @param mustSort Indica se a enumerao deve ser ordenada.
   * @param items Os itens da enumerao, o valor-padro deve ser um dos itens.
   * @param defaultItemId O valor-padro do parmetro.
   * @return Uma instncia do parmetro com os atributos bsicos e especficos
   *         preenchidos.
   * @throws ParseException Caso no seja possvel criar a instncia do
   *         parmetro com os atributos especificados.
   */
  protected abstract T createEnumerationParameter(XmlParser parser, String name,
    String label, String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, boolean mustSort, List<EnumerationItem> items,
    String defaultItemId) throws ParseException;

  /**
   * Obtm a fbrica de itens de enumerao.
   *
   * @return .
   */
  protected final EnumerationItemFactory getEnumerationItemFactory() {
    return enumerationItemFactory;
  }

}
