package csbase.logic.algorithms.parsers.triggers;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.triggers.Trigger;
import csbase.logic.algorithms.parsers.XmlParser;

/**
 * <p>
 * Analisador de {@link Trigger}.
 * </p>
 * 
 * <p>
 * Este parser l os gatilhos.
 * </p>
 */
public final class TriggerParser {
  /**
   * Um mapa as fbricas de gatilhos indexadas pelo nome do elemento no XML.
   */
  private Map<String, TriggerFactory> factoriesByElementName;

  /**
   * Cria o parser.
   */
  public TriggerParser() {
    this.factoriesByElementName = new HashMap<String, TriggerFactory>();

    addFactory(new EnableParameterTriggerFactory());
    addFactory(new DisableParameterTriggerFactory());

    addFactory(new ShowParameterTriggerFactory());
    addFactory(new HideParameterTriggerFactory());

    addFactory(new ShowGroupTriggerFactory());
    addFactory(new HideGroupTriggerFactory());

    addFactory(new ShowEnumerationItemTriggerFactory());
    addFactory(new HideEnumerationItemTriggerFactory());

    addFactory(new AllowEditParameterTriggerFactory());
    addFactory(new ProhibitEditParameterTriggerFactory());

    addFactory(new ChangeDefaultValueTriggerFactory());

    addFactory(new ChangeLabelTriggerFactory());

    addFactory(new ChangeMaximumTriggerFactory());
    addFactory(new ChangeMinimumTriggerFactory());

    addFactory(new ChangeFileTypeTriggerFactory());
  }

  /**
   * <p>
   * Carregar os gatilhos do configurador de algoritmos.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @throws ParseException Se houver um erro no XML.
   */
  public void loadTriggers(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    if (parser.goToFirstChild()) {
      do {
        Set<Trigger<?>> triggers = null;
        String elementName = parser.getElementName();
        TriggerFactory factory = getFactory(elementName);
        if (factory != null) {
          triggers = factory.createTriggers(parser, configurator);
          for (Trigger<?> trigger : triggers) {
            if (!configurator.addTrigger(trigger)) {
              throw new ParseException(
                "Foi encontrado um gatilho repetido no algoritmo {0}.\n"
                  + "Gatilho repetido: {1}.\n", configurator, trigger);
            }
          }
        }
      } while (parser.goToNextSibling());
      parser.goToParent();
    }
  }

  /**
   * Obtm uma fbrica cujo nome do elemento  o informado.
   * 
   * @param elementName O nome do elemento (no pode ser {@code null}).
   * 
   * @return A fbrica ou {@code null} se a fbrica no existir.
   */
  TriggerFactory getFactory(String elementName) {
    return factoriesByElementName.get(elementName);
  }

  /**
   * Obtm as fbricas.
   * 
   * @return As fbricas (se no houver fbricas, retornar um conjunto vazio).
   *         O conjunto retornado  imutvel (
   *         {@link Collections#unmodifiableSet(Set)}).
   */
  Set<TriggerFactory> getFactories() {
    Set<TriggerFactory> factories = new HashSet<TriggerFactory>();
    for (TriggerFactory factory : factoriesByElementName.values()) {
      factories.add(factory);
    }
    return Collections.unmodifiableSet(factories);
  }

  /**
   * Adiciona uma fbrica de gatilhos a este analisador.
   * 
   * @param factory A fbrica.
   */
  private void addFactory(TriggerFactory factory) {
    factoriesByElementName.put(factory.getElementName(), factory);
  }
}
