/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.logic.algorithms.parameters;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.validators.IntegerParameterValidator;
import csbase.logic.algorithms.parameters.validators.SimpleParameterValidator;

/**
 * Coluna de inteiros.
 *
 * @author lmoreira
 */
public final class IntegerColumn extends BasicColumn<Integer> {
  /**
   * O valor mximo.
   */
  private Integer maximum;

  /**
   * O valor mnimo.
   */
  private Integer minimum;

  /**
   * Cria a coluna.
   *
   * @param id O identificador (No aceita {@code null}).
   * @param label O rtulo (No aceita {@code null}).
   * @param defaultValue O valor-padro das clulas (Aceita {@code null}).
   * @param isOptional Indica se o valor da clula  opcional.
   * @param isEditable Indica se a coluna  editvel.
   * @param maximum O mximo (Aceita {@code null}).
   * @param minimum O mnimo (Aceita {@code null}).
   */
  public IntegerColumn(String id, String label, Integer defaultValue,
    boolean isOptional, boolean isEditable, Integer maximum, Integer minimum) {
    super(id, label, defaultValue, isOptional, isEditable);
    setMaximum(maximum);
    setMinimum(minimum);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Integer getItemValueFromText(String itemValue) throws ParseException {
    if (itemValue == null) {
      return null;
    }
    try {
      int intValue = Integer.parseInt(itemValue);
      return new Integer(intValue);
    }
    catch (NumberFormatException e) {
      throw new ParseException(e, "O valor {0} foi atribuido  coluna {1}, "
        + "porm ele no  um valor vlido para um nmero inteiro.", itemValue,
        this);
    }
  }

  /**
   * Obtm o mximo.
   *
   * @return O mximo (Pode ser {@code null}).
   */
  public Integer getMaximum() {
    return this.maximum;
  }

  /**
   * Obtm o mnimo.
   *
   * @return O mnimo (Pode ser {@code null}).
   */
  public Integer getMinimum() {
    return this.minimum;
  }

  /**
   * Atribui o mximo.
   *
   * @param maximum O mximo (Aceita {@code null}).
   */
  private void setMaximum(Integer maximum) {
    if (this.minimum != null) {
      if (maximum != null && maximum.compareTo(this.minimum) < 0) {
        throw new IllegalArgumentException(
          "O valor mximo no pode ser menor do que o valor mnimo.");
      }
    }
    this.maximum = maximum;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public SimpleParameterValidator<Integer> getValidator() {
    return new IntegerParameterValidator(isOptional(), getMaximum(),
      getMinimum());
  }

  /**
   * Atribui o mnimo.
   *
   * @param minimum O mnimo (Aceita {@code null}).
   */
  private void setMinimum(Integer minimum) {
    if (this.maximum != null) {
      if (minimum != null && minimum.compareTo(this.maximum) > 0) {
        throw new IllegalArgumentException(
          "O valor mnimo no pode ser maior do que o valor mximo.");
      }
    }
    this.minimum = minimum;
  }
}
