package csbase.logic.algorithms.parsers.elements.attributes;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parsers.elements.ParsedElement;

/**
 * Atributo do tipo real.
 */
public class DoubleAttribute extends AbstractElementAttribute<Double> {

  /**
   * Atributo que define o valor mnimo deste atributo.
   */
  private DoubleAttribute minimumValueAttribute;
  /**
   * Atributo que define o valor mximo deste atributo.
   */
  private DoubleAttribute maximumValueAttribute;
  /**
   * Valor mnimo.
   */
  private Double minimumValue;
  /**
   * Valor mximo.
   */
  private Double maximumValue;

  /**
   * Construtor.
   *
   * @param name nome do atributo.
   * @param isOptional indicativo de opcionalidade.
   * @param defaultValue valor padro do atributo.
   * @param defaultValueAttribute atributo com valor padro.
   * @param minimumValue valor mnimo do atributo.
   * @param minimumValueAttribute atributo que define o valor mnimo deste
   * atributo.
   * @param maximumValue valor mximo do atributo.
   * @param maximumValueAttribute atributo que define o valor mximo deste
   * atributo.
   */
  protected DoubleAttribute(String name, boolean isOptional,
    Double defaultValue, DoubleAttribute defaultValueAttribute,
    Double minimumValue, DoubleAttribute minimumValueAttribute,
    Double maximumValue, DoubleAttribute maximumValueAttribute) {
    super(name, Double.class, isOptional, defaultValue, defaultValueAttribute);
    this.minimumValue = minimumValue;
    this.maximumValue = maximumValue;
    this.minimumValueAttribute = minimumValueAttribute;
    this.maximumValueAttribute = maximumValueAttribute;
  }

  /**
   * Construtor para atributos opcionais.
   *
   * @param name nome do atributo.
   * @param defaultValue valor padro do atributo.
   * @param minimumValue valor mnimo do atributo.
   * @param maximumValue valor mximo do atributo.
   */
  public DoubleAttribute(String name, Double defaultValue, Double minimumValue,
    Double maximumValue) {
    this(name, true, defaultValue, null, minimumValue, null, maximumValue,
      null);
  }

  /**
   * Construtor para atributos opcionais.
   *
   * @param name nome do atributo.
   * @param defaultValue valor padro do atributo.
   */
  public DoubleAttribute(String name, Double defaultValue) {
    this(name, defaultValue, null, null);
  }

  /**
   * Construtor para atributos obrigatrios.
   *
   * @param name nome do atributo.
   * @param minimumValue valor mnimo do atributo.
   * @param maximumValue valor mximo do atributo.
   */
  public DoubleAttribute(String name, Double minimumValue,
    Double maximumValue) {
    this(name, false, null, null, minimumValue, null, maximumValue, null);
  }

  /**
   * Construtor para atributos obrigatrios.
   *
   * @param name nome do atributo.
   */
  public DoubleAttribute(String name) {
    this(name, null, null);
  }

  /**
   * Atribui o valor mnimo do atributo.
   *
   * @param minimumValue o valor mnimo.
   */
  public void setMinimumValue(Double minimumValue) {
    this.minimumValue = minimumValue;
  }

  /**
   * Atribui o valor mximo do atributo.
   *
   * @param maximumValue o valor mximo.
   */
  public void setMaximumValue(Double maximumValue) {
    this.maximumValue = maximumValue;
  }

  /**
   * Define o atributo que determina o valor mximo deste atributo.
   *
   * @param maximumValueAttribute o valor mximo.
   */
  public void setMaximumValueAttribute(DoubleAttribute maximumValueAttribute) {
    this.maximumValueAttribute = maximumValueAttribute;
  }

  /**
   * Define o atributo que determina o valor mnimo deste atributo.
   *
   * @param minimumValueAttribute o valor mnimo.
   */
  public void setMinimumValueAttribute(DoubleAttribute minimumValueAttribute) {
    this.minimumValueAttribute = minimumValueAttribute;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean validate(ParsedElement definition) throws ParseException {
    return super.validate(definition) && validateMaximum(
      definition) && validateMinimum(definition);
  }

  /**
   * Valida o valor mnimo do atributo, seja por um valor definido ou seja por
   * um valor que seja determinado por outro atributo.
   *
   * @param definition o elemento lido.
   * @return verdadeiro se a validao for bem sucedida ou falso, caso
   * contrrio.
   * @throws ParseException em caso de erro.
   */
  private boolean validateMinimum(ParsedElement definition) throws ParseException {
    Double doubleValue = definition.getAttributeValue(getName());
    if (doubleValue != null) {
      if (minimumValueAttribute != null) {
        this.minimumValue =
          definition.getAttributeValue(minimumValueAttribute.getName());
      }
      if (minimumValue != null) {
        if (doubleValue < minimumValue) {
          throw new ParseException(
            "O valor do atributo {1} est abaixo do valor mnimo permitido" +
              ".\nValor encontrado: ({2}).\nValor mnimo: {3}.\n",
            getName(), doubleValue, minimumValue);
        }
      }
    }
    return true;
  }

  /**
   * Valida o valor mximo do atributo, seja por um valor definido ou seja por
   * um valor que seja determinado por outro atributo.
   *
   * @param definition o elemento lido.
   * @return verdadeiro se a validao for bem sucedida ou falso, caso
   * contrrio.
   * @throws ParseException em caso de erro.
   */
  private boolean validateMaximum(ParsedElement definition) throws ParseException {
    Double doubleValue = definition.getAttributeValue(getName());
    if (doubleValue != null) {
      if (maximumValueAttribute != null) {
        this.maximumValue =
          definition.getAttributeValue(maximumValueAttribute.getName());
      }
      if (maximumValue != null) {
        if (doubleValue > maximumValue) {
          throw new ParseException(
            "O valor do atributo {1} est acima do valor mximo permitido" +
              ".\nValor encontrado: ({2}).\nValor mximo: {3}.\n",
            getName(), doubleValue, maximumValue);
        }
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public StringToValueConverter<Double> getValueConverter() {
    return new StringToDoubleConverter();
  }

  /**
   * Conversor de valores reais para string.
   */
  private class StringToDoubleConverter implements
    StringToValueConverter<Double> {

    /**
     * {@inheritDoc}
     */
    @Override
    public Double valueOf(String attributeValue) throws ParseException {
      if (attributeValue == null) {
        return null;
      }

      Double doubleValue;
      try {
        doubleValue = new Double(attributeValue);
      }
      catch (final NumberFormatException e) {
        throw new ParseException(e,
          "O valor do atributo {1} do elemento {0} deveria ser um nmero " +
            "real .\nValor encontrado: ({2}).",
          getName(), attributeValue);
      }
      return doubleValue;
    }
  }
}