/**
 * $Id$
 */
package csbase.logic;

import java.io.Serializable;
import java.util.Properties;

/**
 * Contm informaes sobre a localizao do OpenBus.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class BusInfo implements Serializable {
  /**
   * A mquina onde o barramento est executando.
   */
  private final String host;
  /**
   * A porta onde o barramento est executando.
   */
  private final int port;

  /**
   * Propriedades usadas para a configurao do ORB, na inicializao do
   * OpenBus. Tipicamente lidos do arquivo OpenbusService.ORB.properties
   */
  private final Properties orbProperties;
  
  /**
   * Propriedades usadas para a configurao do ORB, na inicializao do
   * OpenBus. Tipicamente lidos do arquivo OpenbusService.Client.ORB.properties
   */
  private final Properties clientORBProperties;

  /**
   * Identificador da entidade do servidor no barramento. Este identificador 
   * transiente e muda a cada autenticao com o barramento.
   */
  private String loginId;

  /**
   * Cria a localizao do OpenBus.
   * 
   * @param host A mquina onde o barramento est executando.
   * @param port A porta onde o barramento est executando.
   * @param orbProperties As propriedades do ORB CORBA
   * @param clientORBProperties As propriedades do ORB CORBA a serem usadas no cliente
   */
  public BusInfo(String host, int port, Properties orbProperties,
			Properties clientORBProperties) {
    this.host = host;
    this.port = port;
    this.orbProperties = orbProperties;
    this.clientORBProperties = clientORBProperties;
  }

  /**
   * Obtm a mquina onde o barramento est executando.
   * 
   * @return A mquina onde o barramento est executando.
   */
  public String getHost() {
    return this.host;
  }

  /**
   * Obtm a porta onde o barramento est executando.
   * 
   * @return A porta onde o barramento est executando.
   */
  public int getPort() {
    return this.port;
  }

  /**
   * Propriedades usadas para a configurao do ORB, na inicializao do
   * OpenBus. Tipicamente lidos do arquivo OpenbusService.ORB.properties
   * 
   * @return Propriedades do ORB, para inicializao do barramento.
   */
  public final Properties getOrbProperties() {
    return orbProperties;
  }

  /**
   * Propriedades usadas para a configurao do ORB no cliente que usa o
   * OpenBus. Tipicamente lidos do arquivo OpenbusService.Client.ORB.properties
   * 
   * @return Propriedades do ORB, para inicializao do barramento.
   */
  public final Properties getClientORBProperties() {
    return clientORBProperties;
  }
  
  /**
   * Retorna o identificador da entidade do servidor no barramento.
   * 
   * @return loginId o identificador
   */
  public String getLoginId() {
    return loginId;
  }

  /**
   * Atribui o identificador da entidade do servidor no barramento.
   * 
   * @param loginId o dentificador
   */
  public void setLoginId(String loginId) {
    this.loginId = loginId;
  }
}
