package csbase.logic.algorithms.parsers;

import static csbase.logic.algorithms.parsers.elements
  .SimpleParameterStructure.PARAMETER_ELEMENT_DEFAULT_VALUE_ATTRIBUTE;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.BooleanColumn;
import csbase.logic.algorithms.parameters.BooleanParameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.TableParameter;
import csbase.logic.algorithms.parsers.elements.attributes.BooleanAttribute;
import csbase.logic.algorithms.parsers.elements.attributes.StringAttribute;
import csbase.logic.algorithms.parsers.elements.ParameterStructure;
import csbase.logic.algorithms.parsers.elements.ParsedSimpleParameter;
import csbase.logic.algorithms.parsers.elements.SimpleParameterStructure;

/**
 * <p>
 * Analisador de {@link BooleanParameter}.
 * </p>
 *
 * <p>
 * Este parser l os atributos de parmetros do tipo booleano. O elemento
 * corrente do {@link XmlParser analisador de XML} precisa ser um elemento
 * {@link BooleanParameter}.
 * </p>
 */
public class BooleanParameterFactory extends
  SimpleParameterParser<BooleanParameter> {

  /**
   * <p>
   * O elemento {@value #BOOLEAN_PARAMETER_ELEMENT}: descreve as propriedades de
   * um {@link BooleanParameter parmetro do tipo booleano}.
   * </p>
   * <p>
   *  filho do elemento {@link ParameterGroup}.
   * </p>
   */
  public static final String BOOLEAN_PARAMETER_ELEMENT = "booleano";

  /**
   * O atributo {@value #BOOLEAN_ELEMENT_FALSE_ATTRIBUTE} do elemento
   * {@link BooleanColumn} de um {@link TableParameter} ou
   * {@link BooleanParameter}: define o valor a ser utilizado na linha de
   * comando, quando o valor de uma clula da coluna for falso ou quando o valor
   * do parmetro for falso,  opcional e  do tipo string.
   */
  public static final String BOOLEAN_ELEMENT_FALSE_ATTRIBUTE = "falso";

  /**
   * O atributo {@value #BOOLEAN_ELEMENT_TRUE_ATTRIBUTE} do elemento
   * {@link BooleanColumn} de um {@link TableParameter} ou
   * {@link BooleanParameter}: define o valor a ser utilizado na linha de
   * comando, quando o valor de uma clula da coluna for verdadeiro ou quando o
   * valor do parmetro for verdadeiro,  opcional e  do tipo string.
   */
  public static final String BOOLEAN_ELEMENT_TRUE_ATTRIBUTE = "verdadeiro";

  /**
   * {@inheritDoc}
   */
  @Override
  public BooleanParameter createSimpleParameter(XmlParser parser,
    ParsedSimpleParameter definition, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    boolean defaultValue =
      definition.getAttributeValue(PARAMETER_ELEMENT_DEFAULT_VALUE_ATTRIBUTE);
    String falseCommand =
      definition.getAttributeValue(BOOLEAN_ELEMENT_FALSE_ATTRIBUTE);
    String trueCommand =
      definition.getAttributeValue(BOOLEAN_ELEMENT_TRUE_ATTRIBUTE);
    return new BooleanParameter(definition.getName(), definition.getLabel(),
      definition.getDescription(), defaultValue, definition.isVisible(),
      definition.getCommandLinePattern(), falseCommand, trueCommand);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ParameterStructure<BooleanParameter> getParameterStructure() {
    ParameterStructure<BooleanParameter> structure =
      new SimpleParameterStructure<>(BOOLEAN_PARAMETER_ELEMENT,
        BooleanParameter.class);
    structure
      .addAttribute(new StringAttribute(BOOLEAN_ELEMENT_TRUE_ATTRIBUTE, null));
    structure
      .addAttribute(new StringAttribute(BOOLEAN_ELEMENT_FALSE_ATTRIBUTE, null));
    structure.addAttribute(
      new BooleanAttribute(PARAMETER_ELEMENT_DEFAULT_VALUE_ATTRIBUTE, false));
    return structure;
  }
}
