/*
 * ProjectAdminInfo.java
 * 
 * $Id: ProjectAdminInfo.java 116325 2011-03-18 03:17:58Z cassino $
 */
package csbase.logic;

import java.rmi.RemoteException;
import java.util.Date;
import java.util.Observable;

import csbase.remote.ClientRemoteLocator;
import csbase.util.restart.RestartListener;
import csbase.util.restart.RestartManager;

/**
 * Modela informaes administrativas sobre um projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ProjectAdminInfo extends UserProjectInfo {

  /**
   * Objeto que permite a observao da classe ProjectAdminInfo.
   */
  private static Observable observable = new Observable() {
    @Override
    public void notifyObservers(Object arg) {
      setChanged();
      super.notifyObservers(arg);
    }
  };

  /**
   * Caminho fsico do diretrio do projeto.
   */
  private String absolutePath;

  /**
   * Tamanho da rea solicitada.
   */
  private long areaRequestSize;

  /**
   * Data em que a rea foi solicitada.
   */
  private Date areaRequestDate;

  /**
   * Tamanho da rea alocada para o projeto.
   */
  private long areaLockedSize;

  /**
   * Situao atual de alocao do projeto.
   */
  private ProjectAllocationState state;

  /**
   * Obtem todas as informaes de projetos.
   * 
   * @return Todas as informaes de projetos.
   * 
   * @throws RemoteException Erro durante invocao de mtodo de objeto remoto.
   */
  public static ProjectAdminInfo[] getAll() throws RemoteException {
    return ClientRemoteLocator.projectService.getLockedProjectAdminInfo();
  }

  /**
   * Informa que um ou mais projetos tiveram sua rea concedida. O projeto 
   * desbloqueado.
   * 
   * @param infos Um array com os objetos <code>ProjectAdminInfo</code> com as
   *        informaes de desbloqueio de projetos devido a alocao de rea
   *        pelo administrador.
   * 
   * @throws RemoteException Erro durante invocao de mtodo de objeto remoto.
   */
  public static void allocateAreaForProjects(ProjectAdminInfo[] infos)
    throws RemoteException {
    ClientRemoteLocator.projectService.unlockProjectsWithAllocation(infos);
  }

  /**
   * Informa que um ou mais projetos tiveram sua rea liberada.
   * 
   * @param infos Um array com os objetos <code>ProjectAdminInfo</code> com as
   *        informaes de desbloqueio de projetos devido a alocao de rea
   *        pelo administrador.
   * 
   * @throws RemoteException Erro durante invocao de mtodo de objeto remoto.
   */
  public static void freeAreaForProjects(ProjectAdminInfo[] infos)
    throws RemoteException {
    ClientRemoteLocator.projectService.freeAreaForProjects(infos);
  }

  /**
   * Obtm o caminho fsico do diretrio do projeto.
   * 
   * @return O caminho fsico do diretrio do projeto.
   */
  public String getAbsolutePath() {
    return absolutePath;
  }

  /**
   * Obtm o tamanho da rea solicitada para o projeto.
   * 
   * @return O tamanho da rea solicitada para o projeto.
   */
  public long getAreaRequestSize() {
    return areaRequestSize;
  }

  /**
   * Obtm o tamanho da rea alocada para o projeto.
   * 
   * @return O tamanho da rea alocada para o projeto.
   */
  public long getAreaLockedSize() {
    return areaLockedSize;
  }

  /**
   * Obtm o estado atual do projeto.
   * 
   * @return O estado do projeto.
   */
  public ProjectAllocationState getState() {
    return state;
  }

  /**
   * Verifica se o projeto teve sua solicitao de espao cancelada.
   * 
   * @return verdadeiro, se o projeto teve sua solicitao de espao cancelada
   *         ou falso, caso contrrio.
   */
  public boolean isLocked() {
    return (state == ProjectAllocationState.WAITING_AREA_ALLOCATION);
  }

  /**
   * Verifica se o projeto est desbloqueado utilizando rea reservada.
   * 
   * @return verdadeiro, se o projeto est desbloqueado utilizando rea
   *         reservada ou falso, caso contrrio.
   */
  public boolean isUnlockedWithAreaAllocated() {
    return (state == ProjectAllocationState.UNLOCKED_WITH_AREA_ALLOCATED);
  }

  /**
   * Verifica se o projeto est aguardando liberao de rea.
   * 
   * @return verdadeiro, se o projeto est aguardando liberao de rea
   *         reservada ou falso, caso contrrio.
   */
  public boolean isWaitingAreaFree() {
    return (state == ProjectAllocationState.WAITING_AREA_FREE);
  }

  /**
   * Obtm a data em que foi solicitada uma rea para o projeto.
   * 
   * @return A data de solicitao de reserva de rea.
   */
  public Date getAreaRequestDate() {
    return areaRequestDate;
  }

  /**
   * Muda o estado do projeto.
   * 
   * @param state O novo estado do projeto.
   */
  public void setState(ProjectAllocationState state) {
    this.state = state;
  }

  /**
   * Informa o tamanho da rea requisitada para o projeto.
   * 
   * @param size O tamanho da rea requisitada.
   */
  public void setAreaRequestSize(long size) {
    this.areaRequestSize = size;
  }

  /**
   * Informa o tamanho da rea alocada para o projeto.
   * 
   * @param size O tamanho da rea alocada pelo administrador.
   */
  public void setAreaLockedSize(long size) {
    this.areaLockedSize = size;
  }

  /**
   * Informa a data da requisio de rea para o projeto.
   * 
   * @param date A data em que a requisio de rea foi solicitada.
   */
  public void setAreaRequestDate(Date date) {
    this.areaRequestDate = date;
  }

  /**
   * Esse mtodo  chamado quando o servidor sofreu alguma alterao nas
   * informaes administrativas de projetos.
   * 
   * @param action a ao que ocorreu no provedor de usurios
   */
  public static void update(AdministrationEvent action) {
    observable.notifyObservers(action);
  }

  /**
   * Constri um <code>ProjectAdminInfo</code> com as informaes sobre um
   * projeto.
   * 
   * @param projectId Identificador do projeto.
   * @param projectName Nome do projeto.
   * @param ownerId Identificador do usurio dono do projeto.
   * @param absolutePath Caminho fsico do diretrio do projeto.
   * @param state O estado do projeto.
   */
  public ProjectAdminInfo(Object projectId, String projectName, Object ownerId,
    String absolutePath, ProjectAllocationState state) {
    super(projectId, projectName, ownerId);
    this.absolutePath = absolutePath;
    this.state = state;
    this.areaRequestSize = 0;
    this.areaRequestDate = null;
  }

  static {
    RestartManager.getInstance().addListener(new RestartListener() {
      public void restart() {
        ProjectAdminInfo.observable.deleteObservers();
      }
    });
  }
}
