package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import tecgraf.javautils.parsers.FiniteAutomaton;
import tecgraf.javautils.parsers.Parser;
import tecgraf.javautils.parsers.State;
import tecgraf.javautils.parsers.Token;
import tecgraf.javautils.parsers.actions.AppendAction;
import tecgraf.javautils.parsers.actions.DiscardAction;
import tecgraf.javautils.parsers.actions.GenerateTokenAction;
import tecgraf.javautils.parsers.exception.AutomatonException;
import tecgraf.javautils.parsers.iterators.CharSymbolIterator;
import tecgraf.javautils.parsers.symbols.CharSymbol;
import tecgraf.javautils.parsers.symbols.Symbol;

/**
 * Analisador do texto que representa o valor de um parmetro do tipo lista.
 */
final class ListParameterParser extends Parser {

  /**
   * Caractere que define o incio de uma lista.
   */
  static final char START_LINE_CHAR = '{';
  /**
   * Caractere que separa os elementos da lista.
   */
  static final char ELEMENT_SEPARATOR_CHAR = ',';
  /**
   * Caractere que define o fim de uma lista.
   */
  static final char END_LINE_CHAR = '}';

  /**
   * Construtor.
   */
  ListParameterParser() {
    super(new ListParameterFiniteAutomaton());
  }

  /**
   * Analisa o contedo do leitor e retorna a lista de elementos lidos.
   * 
   * @param reader o leitor a ser analisado.
   * @return list a lista de elementos lidos.
   * 
   * @throws IOException Em caso de erro de I/O.
   * @throws AutomatonException em caso de falha no autmato do analisador.
   */
  List<String> parseText(Reader reader) throws IOException, AutomatonException {
    StringBuilder buffer = new StringBuilder();
    int current;
    while ((current = reader.read()) != -1) {
      buffer.append((char) current);
    }
    String text = buffer.toString();
    return parseText(text);
  }

  /**
   * Analisa o contedo da string e retorna a lista de elementos lidos.
   * 
   * @param text o texto a ser analisado.
   * @return list a lista de elementos lidos.
   * 
   * @throws AutomatonException em caso de falha no autmato do analisador.
   */
  List<String> parseText(String text) throws AutomatonException {
    List<Token> tokenList = parse(new CharSymbolIterator(text));
    return this.createStringList(tokenList);
  }

  /**
   * Cria uma lista de strings a partir de uma lista de tokens.
   * 
   * @param tokens a lista de tokens.
   * @return list a lista de strings resultante.
   */
  private List<String> createStringList(List<Token> tokens) {
    List<String> stringList = new ArrayList<String>(tokens.size());
    Iterator<Token> tokenIterator = tokens.iterator();
    while (tokenIterator.hasNext()) {
      Token token = tokenIterator.next();
      List<Symbol<?>> symbolList = token.getSymbolList();
      Iterator<Symbol<?>> symbolListIterator = symbolList.iterator();
      StringBuffer buffer = new StringBuffer();
      while (symbolListIterator.hasNext()) {
        CharSymbol symbol = (CharSymbol) symbolListIterator.next();
        Character object = symbol.getObject();
        buffer.append(object);
      }
      stringList.add(buffer.toString());
    }
    return stringList;
  }

  /**
   * O autmato do analisador de listas.
   */
  private static class ListParameterFiniteAutomaton extends FiniteAutomaton {

    /** O smbolo que representa o incio de uma lista */
    private static final CharSymbol START_LINE_SYMBOL = new CharSymbol(
      ListParameterParser.START_LINE_CHAR);

    /** O smbolo que representa o separador de elementos da lista */
    private static final CharSymbol ELEMENT_SEPARATOR_SYMBOL = new CharSymbol(
      ListParameterParser.ELEMENT_SEPARATOR_CHAR);

    /** O smbolo que representa o final de uma lista */
    private static final CharSymbol END_LINE_SYMBOL = new CharSymbol(
      ListParameterParser.END_LINE_CHAR);

    /**
     * Construtor.
     */
    private ListParameterFiniteAutomaton() {
      super(new State(false));
      State startLineState = getInitialState();
      State elementState = new State(false);
      State endLineState = new State(true);
      startLineState.addTransition(START_LINE_SYMBOL, DiscardAction
        .getInstance(), elementState);
      elementState.addTransition(ELEMENT_SEPARATOR_SYMBOL, GenerateTokenAction
        .getInstance(), elementState);
      elementState.addTransition(END_LINE_SYMBOL, DiscardAction.getInstance(),
        endLineState);
      elementState.setDefaultTransition(AppendAction.getInstance(),
        elementState);
    }
  }
}
