/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.logic.algorithms.parameters.validators;

import java.rmi.RemoteException;

import csbase.logic.algorithms.parameters.DoubleParameter;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationError;
import csbase.logic.algorithms.validation.ValidationSuccess;

/**
 * Validador do {@link csbase.logic.algorithms.parameters.DoubleParameter
 * Parmetro do Tipo Real}
 *
 * @author lmoreira
 */
public final class DoubleParameterValidator extends
  SimpleParameterValidator<Double> {
  /**
   * Indica se o valor mximo est entre os valores vlidos.
   */
  private boolean isMaximumIncluded;
  /**
   * Indica se o valor mnimo est entre os valores vlidos.
   */
  private boolean isMinimumIncluded;
  /**
   * O valor mximo.
   */
  private Double maximum;
  /**
   * O valor mnimo.
   */
  private Double minimum;

  /**
   * Cria um Validador.
   *
   * @param isOptional Indica se o valor  opcional/obrigatrio.
   * @param maximum O mximo (Aceita {@code null}).
   * @param isMaximumIncluded Indica se o valor mximo est entre os valores
   *        vlidos.
   * @param minimum O mnimo (Aceita {@code null}).
   * @param isMinimumIncluded Indica se o valor mnimo est entre os valores
   *        vlidos.
   */
  public DoubleParameterValidator(boolean isOptional, Double maximum,
    boolean isMaximumIncluded, Double minimum, boolean isMinimumIncluded) {
    super(isOptional);
    setMaximum(maximum, isMaximumIncluded);
    setMinimum(minimum, isMinimumIncluded);
  }

  /**
   * Obtm o mximo.
   *
   * @return O mximo ou {@code null} se ele no existir.
   */
  public Double getMaximum() {
    return this.maximum;
  }

  /**
   * Obtm o mnimo.
   *
   * @return O mnimo ou {@code null} se ele no existir.
   */
  public Double getMinimum() {
    return this.minimum;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object convert(Object value) throws Exception {
    if (Double.class.isInstance(value)) {
      return value;
    }
    if (Integer.class.isInstance(value)) {
      Integer integerValue = (Integer) value;
      return new Double(integerValue.intValue());
    }
    if (Long.class.isInstance(value)) {
      Long longValue = (Long) value;
      return new Double(longValue.longValue());
    }

    final String className = value.getClass().getName();
    throw new Exception("Bad conversion from " + className + ".");
  }

  /**
   * Indica se o valor mximo est entre os valores vlidos.
   *
   * @return .
   */
  public boolean isMaximumIncluded() {
    return this.isMaximumIncluded;
  }

  /**
   * Indica se o valor mnimo est entre os valores vlidos.
   *
   * @return .
   */
  public boolean isMinimumIncluded() {
    return this.isMinimumIncluded;
  }

  /**
   * Modifica o mximo.
   *
   * @param maximum O mximo (Aceita {@code null}).
   *
   * @param isMaximumIncluded Indica se o valor mximo est entre os valores
   *        vlidos.
   */
  public void setMaximum(Double maximum, boolean isMaximumIncluded) {
    if (this.minimum != null) {
      if (maximum != null && maximum.compareTo(this.minimum) < 0) {
        throw new IllegalArgumentException(
          "O valor mximo no pode ser menor do que o valor mnimo.");
      }
    }
    if (maximum == null) {
      this.isMaximumIncluded = false;
    }
    else {
      this.isMaximumIncluded = isMaximumIncluded;
    }
    this.maximum = maximum;
  }

  /**
   * Modifica o mnimo.
   *
   * @param minimum O mnimo (Aceita {@code null}).
   *
   * @param isMinimumIncluded Indica se o valor mnimo est entre os valores
   *        vlidos.
   */
  public void setMinimum(Double minimum, boolean isMinimumIncluded) {
    if (this.maximum != null) {
      if (minimum != null && minimum.compareTo(this.maximum) > 0) {
        throw new IllegalArgumentException(
          "O valor mnimo no pode ser maior do que o valor mximo.");
      }
    }
    if (minimum == null) {
      this.isMinimumIncluded = false;
    }
    else {
      this.isMinimumIncluded = isMinimumIncluded;
    }
    this.minimum = minimum;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Validation validateValue(SimpleParameter<?> parameter, Double value,
    ValidationContext context) throws RemoteException {
    Validation validationResult = super.validateValue(parameter, value,
      context);
    if (!validationResult.isWellSucceeded()) {
      return validationResult;
    }
    if (parameter.isEnabled() && parameter.isVisible()) {
      if (value != null) {
        if (value.equals(DoubleParameter.ERROR_VALUE)) {
          return new ValidationError(new LocalizedMessage(
            DoubleParameterValidator.class, "error_value", new Object[] {
                parameter.getLabel() }));
        }
        if (this.minimum != null) {
          if (this.isMinimumIncluded) {
            if (value.compareTo(this.minimum) < 0) {
              LocalizedMessage message = new LocalizedMessage(
                DoubleParameterValidator.class, "less_than_minimum",
                new Object[] { parameter.getLabel(), value, this.minimum });
              return new ValidationError(message);
            }
          }
          else {
            if (value.compareTo(this.minimum) <= 0) {
              LocalizedMessage message = new LocalizedMessage(
                DoubleParameterValidator.class, "less_or_equals_to_minimum",
                new Object[] { parameter.getLabel(), value, this.minimum });
              return new ValidationError(message);
            }
          }
        }
        if (this.maximum != null) {
          if (this.isMaximumIncluded) {
            if (value.compareTo(this.maximum) > 0) {
              LocalizedMessage message = new LocalizedMessage(
                DoubleParameterValidator.class, "larger_than_maximum",
                new Object[] { parameter.getLabel(), value, this.maximum });
              return new ValidationError(message);
            }
          }
          else {
            if (value.compareTo(this.maximum) >= 0) {
              LocalizedMessage message = new LocalizedMessage(
                DoubleParameterValidator.class, "larger_or_equals_to_maximum",
                new Object[] { parameter.getLabel(), value, this.maximum });
              return new ValidationError(message);
            }
          }
        }
      }
    }
    return new ValidationSuccess();
  }
}
