package csbase.logic.algorithms.parameters.validators;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;

import csbase.logic.algorithms.parameters.RowValue;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.TableColumn;
import csbase.logic.algorithms.parameters.TableParameter;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationError;
import csbase.logic.algorithms.validation.ValidationMode;
import csbase.logic.algorithms.validation.ValidationSuccess;

/**
 * Validador do {@link csbase.logic.algorithms.parameters.TableParameter
 * Parmetro do Tipo Tabela}.
 */
public final class TableParameterValidator extends
  SimpleParameterValidator<List<RowValue>> {
  /**
   * Os validadores das colunas.
   */
  private List<SimpleParameterValidator<?>> columnValidators;

  /**
   * ndice da coluna na tabela.
   */
  private int columnIndex;
  /**
   * ndice da linha na tabela.
   */
  private int rowIndex;

  /**
   * Cria um validador de tabela.
   *
   * @param optional Indica se o valor  opcional/obrigatrio.
   * @param columns Lista de colunas da tabela
   */
  public TableParameterValidator(boolean optional,
    List<TableColumn<?>> columns) {
    super(optional);
    this.columnValidators = new ArrayList<SimpleParameterValidator<?>>(columns
      .size());
    for (TableColumn<?> column : columns) {
      this.columnValidators.add(column.getValidator());
    }
  }

  /**
   * @see csbase.logic.algorithms.parameters.validators.SimpleParameterValidator#setMode(csbase.logic.algorithms.validation.ValidationMode)
   */
  @Override
  public void setMode(ValidationMode mode) {
    for (SimpleParameterValidator<?> validator : this.columnValidators) {
      validator.setMode(mode);
    }
    super.setMode(mode);
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings("unchecked")
  @Override
  public Validation validateValue(SimpleParameter<?> parameter,
    List<RowValue> value, ValidationContext context) throws RemoteException {
    if (!parameter.isEnabled() || !parameter.isVisible()) {
      return new ValidationSuccess();
    }
    TableParameter tableParameter = (TableParameter) parameter;
    if (tableParameter.getRowCount() == 0) {
      if (isOptional() || getMode().equals(ValidationMode.ALLOW_EMPY_VALUES)) {
        return new ValidationSuccess();
      }
      LocalizedMessage message = new LocalizedMessage(
        TableParameterValidator.class, "empty_value", new Object[] { parameter
          .getLabel() });
      return new ValidationError(message);
    }
    for (int i = 0; i < tableParameter.getRowCount(); i++) {
      this.rowIndex = i;
      List<TableColumn<?>> columns = tableParameter.getColumns();
      for (int j = 0; j < columns.size(); j++) {
        this.columnIndex = j;
        Object itemValue = tableParameter.getItemValue(i, j);
        @SuppressWarnings("rawtypes")
        SimpleParameterValidator validator = this.columnValidators.get(j);
        try {
          itemValue = validator.convert(itemValue);
        }
        catch (Exception e) {
          final TableColumn<?> column = columns.get(j);
          final String parName = parameter.getLabel();
          final Integer linNum = Integer.valueOf(i + 1);
          final Integer colNum = Integer.valueOf(j + 1);
          final String colName = column.getLabel();
          LocalizedMessage message = new LocalizedMessage(
            TableParameterValidator.class, "bad_cell_value", new Object[] {
                parName, linNum, colNum, colName, e.getMessage() });
          return new ValidationError(message);
        }
        Validation result = validator.validateValue(tableParameter, itemValue,
          context);
        if (!result.isWellSucceeded()) {
          return result;
        }
      }
    }
    return new ValidationSuccess();
  }

  /**
   * Retorna o ndice da linha na tabela.
   *
   * @return O ndice da linha.
   */
  public int getRowIndex() {
    return rowIndex;
  }

  /**
   * Retorna o ndice da coluna na tabela.
   *
   * @return O ndice da coluna.
   */
  public int getColumnIndex() {
    return columnIndex;
  }
}