package csbase.logic.server;

import java.io.Serializable;
import java.security.cert.Certificate;

import csbase.logic.ServerURI;
import csbase.remote.ServerServiceInterface;

/**
 * <p>
 * Encapsula as informaes sobre um servidor.<br>
 * Obs.: Este objeto  imutvel.
 * </p>
 *
 * @author Tecgraf/PUC-Rio
 */
public final class ServerInfo implements Serializable, Comparable<ServerInfo> {
  /** O identificador do servidor */
  private Integer id;

  /** O nome do servidor  */
  private String name;

  /** Indica se o servidor est suspenso. */
  private boolean suspended;

  /** Indica se o servidor  local */
  private boolean isLocal;

  /** A URI para conexo com o servidor */
  private ServerURI uri;

  /**
   * Indica se o servidor est completo (j possui certificado
   * cadastrado).
   */
  private boolean completed;

  /** O certificado do servidor  */
  private transient Certificate certificate;

  /**
   * <p>
   * Construtor.
   * </p>
   * 
   * <p>
   * Esse construtor deve ser utilizado apenas pelo
   * {@link ServerServiceInterface servio de servidores}.
   * </p>
   *
   * @param id Identificador nico.
   * @param name O nome do servidor.
   * @param suspended Indica se o servidor est suspenso.
   * @param isLocal Indica se o servidor  local.
   * @param uri A URI para conexo com o servidor.
   *
   * @throws IllegalArgumentException Caso o algum dos parmetros recebido seja
   *         nulo.
   */
  public ServerInfo(Integer id, String name, boolean suspended,
    boolean isLocal, ServerURI uri) {
    if (id == null) {
      throw new IllegalArgumentException(
        "Identificador do servidor no pode ser nulo.");
    }
    if (name == null) {
      throw new IllegalArgumentException("Nome do servidor no pode ser nulo.");
    }
    if (uri == null)
      throw new IllegalArgumentException("A URI no pode ser nula");

    this.id = id;
    this.name = name;
    this.suspended = suspended;
    this.completed = false;
    this.isLocal = isLocal;
    this.uri = uri;
  }

  /**
   * <p>
   * Construtor.
   * </p>
   * 
   * <p>
   * Esse construtor deve ser utilizado apenas pelo
   * {@link ServerServiceInterface servio de servidores}.
   * </p>
   *
   * @param id Identificador nico.
   * @param name O nome do servidor.
   * @param suspended Indica se o servidor est suspenso.
   * @param isLocal Indica se o servidor  local.
   * @param uri A URI para conexo com o servidor.
   * @param certificate O certificado do servidor. 
   *
   * @throws IllegalArgumentException Caso o algum dos parmetros recebido seja
   *         nulo.
   */
  private ServerInfo(Integer id, String name, boolean suspended,
    boolean isLocal, ServerURI uri, Certificate certificate) {
    this(id, name, suspended, isLocal, uri);
    if (certificate == null) {
      throw new IllegalArgumentException(
        "O certificado do servidor no pode ser nulo.");
    }
    this.certificate = certificate;
    this.completed = true;
  }

  /**
   * <p>
   * Construtor.
   * </p>
   * 
   * <p>
   * Esse construtor deve ser utilizado apenas pelo
   * {@link ServerServiceInterface servio de servidores}.
   * </p>
   * 
   * @param serverInfo Informaes de um servidor que ser utilizado como base
   *        para a criao do novo {@link ServerInfo}.<br>
   *        Obs.: O identificador desta instncia no pode ser nulo.
   */
  public ServerInfo(ServerInfo serverInfo) {
    this(serverInfo.id, serverInfo.name, serverInfo.suspended,
      serverInfo.isLocal, serverInfo.uri);
  }

  /**
   * <p>
   * Construtor.
   * </p>
   * 
   * <p>
   * Esse construtor deve ser utilizado apenas pelo
   * {@link ServerServiceInterface servio de servidores}.
   * </p>
   * 
   * @param id O identificador do {@link ServerInfo} que est sendo criado.
   * @param serverInfo Informaes de um servidor que ser utilizado como base
   *        para a criao do novo {@link ServerInfo}.<br>
   */
  public ServerInfo(Integer id, ServerInfoData serverInfo) {
    this(id, serverInfo.getName(), serverInfo.isSuspended(), serverInfo
      .isLocal(), serverInfo.getURI());
  }

  /**
   * <p>
   * Construtor.
   * </p>
   * 
   * <p>
   * Esse construtor deve ser utilizado apenas pelo
   * {@link ServerServiceInterface servio de servidores}.
   * </p>
   *
   * @param serverInfo Informaes de um servidor que ser utilizado como base
   *        para a criao do novo {@link ServerInfo}.<br>
   *        Obs.: O identificador desta instncia no pode ser nulo.
   * @param certificate O certificado do servidor.
   */
  public ServerInfo(ServerInfo serverInfo, Certificate certificate) {
    this(serverInfo.id, serverInfo.name, serverInfo.suspended,
      serverInfo.isLocal, serverInfo.getURI(), certificate);
  }

  /**
   * <p>
   * Cria um servidor
   * </p>
   * 
   * <p>
   * Esse construtor deve ser utilizado apenas pelo
   * {@link ServerServiceInterface servio de servidores}.
   * </p>
   * 
   * @param id O identificador do {@link ServerInfo} que est sendo criado.
   * @param serverInfoData Os dados necessrios para a criao do novo servidor
   *        {@link ServerInfo}.
   * @param certificate O certificado do servidor.
   */
  public ServerInfo(Integer id, ServerInfoData serverInfoData,
    Certificate certificate) {
    this(id, serverInfoData.getName(), serverInfoData.isSuspended(),
      serverInfoData.isLocal(), serverInfoData.getURI(), certificate);
  }

  /**
   * Obtm o nome do servidor.
   *
   * @return O nome do servidor.
   */
  public String getName() {
    return this.name;
  }

  /**
   * Obtm o identificador do servidor.
   *
   * @return O identificador do servido.
   */
  public Object getId() {
    return this.id;
  }

  /**
   * Verifica se o servidor est suspenso.
   *
   * @return true, se estiver suspenso, ou false, caso contrrio.
   */
  public boolean isSuspended() {
    return this.suspended;
  }

  /**
   * Verifica se o servidor est completo.
   *
   * @return true, se estiver completo, ou false, caso contrrio.
   */
  public boolean isCompleted() {
    return this.completed;
  }

  /**
   * Obtm o certificado do servidor.
   *
   * @return O certificado do servidor, ou null, caso o servidor 
   *         ainda no possua certificado cadastrado.
   */
  public Certificate getCertificate() {
    return this.certificate;
  }

  /**
   * Verifica se o servidor  local.
   * 
   * @return true se o servidor  local, false caso contrrio.
   */
  public boolean isLocal() {
    return this.isLocal;
  }

  /**
   * @return a URI do servidor.
   */
  public ServerURI getURI() {
    return this.uri;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object object) {
    if (object == null) {
      return false;
    }
    if (!this.getClass().equals(object.getClass())) {
      return false;
    }
    ServerInfo localServer = (ServerInfo) object;
    return this.name.equals(localServer.name);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return this.name.hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return this.uri.toString();
  }

  /**
   * A comparao natural do {@link ServerInfo}  feita pelo seu nome.
   * {@inheritDoc}
   */
  @Override  
  public int compareTo(ServerInfo other) {
    return this.name.compareTo(other.name);
  }
}
