/**
 * 
 */
package csbase.logic;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import tecgraf.javautils.core.io.FileUtils;

/**
 * Permisso que define quais so os paths do SGA visveis para um usurio.
 * Primeiro,  necessrio definir o SGA a que esta permisso se aplica. Para
 * isso, basta definir o atributo {@code "sga=<expresso regular>"}. Logo em seguida,
 * basta definir os paths visveis para o usurio.
 * 
 * Esta implementao permite especificarmos os paths permitidos de 3 formas:
 * ----------------------------------------------------------------------------
 * 1 - Podemos descrever especificamente os paths permitidos atravs de
 * expresses regulares. Para isso, o atributo deve ser definido da seguinte
 * forma: {@code permitir=<expresso regular>;}
 * 
 * Por exemplo:<br>
 * {@code permitir=.*} (permite a visualizao de todos os paths)
 * {@code permitir=/local/.*} (permite a visualizao de todos os paths de
 * "/local/")
 * 
 * -----------------------------------------------------------------------------
 * 2 - Podemos descrever especificamente os paths que no podem ser visualizados
 * pelo usurio. Para isso, o atributo deve ser definido da seguinte forma:
 * {@code excluir=<expresso regular>;}
 * 
 * Por exemplo:<br>
 * {@code excluir=/bin/.*} (impede a visualizao dos paths de "/bin/")
 * {@code excluir=.*lib.*} (impede a visualizao dos paths com a substring "lib")
 * 
 * -----------------------------------------------------------------------------
 * 3 - H casos em que queremos especificar o caminho de um diretrio, por
 * exemplo, /a/b/c. Nesses casos, poderiamos definir a permisso da seguinte
 * forma:
 * 
 * {@code permitir=(/a|/a/b|/a/b/c|/a/b/c/.*)}
 *
 * Porm, existe uma aucar sinttico para esse caso especfico que permite que
 * o administrador especifique da mesma forma mas com uma sintaxe mais
 * intuitiva. Para isso, devemos usar o atributo
 * {@code "caminho=<path de um diretrio>"}. O mesmo exemplo acima pode ser
 * descrito da seguinte forma:
 * 
 * {@code caminho=/a/b/c}
 * 
 * -----------------------------------------------------------------------------
 * 
 * NOTA 1: Atributos de excluso possuem prioridade sobre os atributos de
 * permisso.
 * 
 * NOTA 2: Atributos com erros de sintaxe sero desconsiderados.
 * 
 * @author tecgraf
 */
public class AccessSGAPathPermission extends AttributesPermission {

  /** Atributo que define o SGA em questo. */
  public static final String SGA_ATT = "sga=";

  /** Atributo que define um path permitido. */
  public static final String ALLOW_ATT = "permitir=";

  /** Atributo que define um path negado. */
  public static final String DENY_ATT = "excluir=";

  /** Atributo que define o caminho at um diretrio que ser permitido. */
  public static final String PATH_ATT = "caminho=";

  /** Construtor padro. */
  public AccessSGAPathPermission() {
    super();
  }

  /**
   * Constri uma permisso para acesso de paths do SGA.
   * 
   * @param name nome da permisso.
   * @param description descrio.
   * @param attributes atributos da permisso.
   */
  public AccessSGAPathPermission(String name, String description,
    String[] attributes) {
    super(name, description, attributes);
  }

  /**
   * Retorna true se o dado usurio tem permisso para visualizar o path no dado
   * SGA, false caso contrrio.
   * 
   * @param user usurio.
   * @param sga nome do SGA.
   * @param path path a ser verificado.
   * @return true se o dado usurio tem permisso para visualizar o path, false
   *         caso contrrio.
   */
  public static boolean canReadPath(User user, String sga, String path) {
    if (user == null) {
      throw new IllegalArgumentException("user no pode ser nulo.");
    }

    if (user.isAdmin()) {
      return true;
    }

    List<AccessSGAPathPermission> permissions =
      getAllAccessSGAPathPermissions(user);

    // Se o usurio no possui essa permisso ento ele no pode ver nada.
    if (permissions.isEmpty()) {
      return false;
    }

    for (AccessSGAPathPermission p : permissions) {
      Pattern sgaPatt = createSGAPattern(p.getAttributes());
      if (sgaPatt == null || sga == null) {
        continue;
      }

      Matcher m = sgaPatt.matcher(sga);
      if (m.matches()) {
        // excluso de paths tem maior prioridade.
        Pattern denyPatt = createDenyPattern(p.getAttributes());
        if (denyPatt != null && path != null) {
          m = denyPatt.matcher(path);
          if (m.matches()) {
            continue;
          }
        }

        Pattern allowPatt = createAllowPattern(p.getAttributes());
        if (allowPatt != null && path != null) {
          m = allowPatt.matcher(path);
          if (m.matches()) {
            return true;
          }
        }

        Pattern pathPatt = createPathPattern(p.getAttributes());
        if (pathPatt != null && path != null) {
          m = pathPatt.matcher(path);
          if (m.matches()) {
            return true;
          }
        }
      }
    }
    return false;
  }

  /**
   * Retorna todas as permisses do dado usurio que so da classe
   * {@link AccessSGAPathPermission}.
   * 
   * @param user usurio.
   * @return lista de permisses {@link AccessSGAPathPermission}.
   */
  private static List<AccessSGAPathPermission> getAllAccessSGAPathPermissions(
    User user) {
    List<AccessSGAPathPermission> permissions =
      new ArrayList<AccessSGAPathPermission>();

    try {
      for (Object permissionId : user.getAllPermissionIds()) {
        Permission permission = Permission.getPermission(permissionId);
        if (permission.getClass().equals(AccessSGAPathPermission.class)) {
          permissions.add((AccessSGAPathPermission) permission);
        }
      }
    }
    catch (Exception e) {
      //faz nada
    }

    return permissions;
  }

  /**
   * Constroi o padro que reconhece o SGA a que esta permisso se aplica.
   * 
   * @param values valores definidos na permisso.
   * @return padro que reconhece o SGA a que esta permisso se aplica.
   */
  private static Pattern createSGAPattern(String[] values) {
    StringBuilder builder = new StringBuilder();

    boolean hasPatt = false;
    for (String value : values) {
      if (isSGAAtt(value)) {
        String regex = getValue(value);
        if (regex != null && isPatternValid(regex)) {
          if (hasPatt) {
            builder.append("|");
          }
          builder.append(regex);
          hasPatt = true;
        }
      }
    }

    if (!hasPatt) {
      return null;
    }

    return Pattern.compile(builder.toString());
  }

  /**
   * Constroi o padro que reconhece os paths que devem ser excluidos.
   * 
   * @param values valores definidos na permisso.
   * @return padro que reconhece os paths que devem ser excluidos.
   */
  private static Pattern createDenyPattern(String[] values) {
    StringBuilder builder = new StringBuilder();

    boolean hasPatt = false;
    for (String value : values) {
      if (isDenyAtt(value)) {
        String regex = getValue(value);
        if (regex != null && isPatternValid(regex)) {
          if (hasPatt) {
            builder.append("|");
          }
          builder.append(regex);
          hasPatt = true;
        }
      }
    }

    if (!hasPatt) {
      return null;
    }

    return Pattern.compile(builder.toString());
  }

  /**
   * Constroi o padro que reconhece os paths que podem ser visualizados pelo
   * usurio.
   * 
   * @param values valores definidos na permisso.
   * @return padro que reconhece os paths que podem ser visualizados pelo
   *         usurio.
   */
  private static Pattern createAllowPattern(String[] values) {
    StringBuilder builder = new StringBuilder();

    boolean hasPatt = false;
    for (String value : values) {
      if (isAllowAtt(value)) {
        String regex = getValue(value);
        if (regex != null && isPatternValid(regex)) {
          if (hasPatt) {
            builder.append("|");
          }
          builder.append(regex);
          hasPatt = true;
        }
      }
    }

    if (!hasPatt) {
      return null;
    }

    return Pattern.compile(builder.toString());
  }

  /**
   * Constri o padro que permite chegar at um determinado diretrio.
   * 
   * @param values valores definidos na permisso.
   * @return padro que permite chegar at um determinado diretrio.
   */
  private static Pattern createPathPattern(String[] values) {
    StringBuilder builder = new StringBuilder();

    boolean hasPatt = false;
    for (String value : values) {
      if (isPathAtt(value)) {
        String regex = buildRegex(getValue(value));
        if (regex != null && isPatternValid(regex)) {
          if (hasPatt) {
            builder.append("|");
          }
          builder.append(regex);
          hasPatt = true;
        }
      }
    }

    if (!hasPatt) {
      return null;
    }

    return Pattern.compile(builder.toString());
  }

  /**
   * Retorna true se a expresso regular  vlida, false caso contrrio.
   * 
   * @param regex expresso regular.
   * @return true se a expresso regular  vlida, false caso contrrio.
   */
  private static boolean isPatternValid(String regex) {
    try {
      Pattern.compile(regex);
      return true;
    }
    catch (PatternSyntaxException e) {
      return false;
    }
  }

  /**
   * Constri a expresso regular que permite chegar at o caminho especificado.
   * 
   * @param value caminho completo.
   * @return expresso regular.
   */
  private static String buildRegex(String value) {
    boolean isAbsolute = FileUtils.isAbsolutePath(value);

    String[] splited = FileUtils.splitPath(value, '/');

    StringBuilder b = new StringBuilder();
    for (int i = 0; i < splited.length; i++) {
      String[] subArray = Arrays.copyOfRange(splited, 0, i + 1);

      String subPath =
        (isAbsolute) ? FileUtils.joinPath(true, '/', subArray) : FileUtils
          .joinPath(subArray);

      b.append(subPath);

      if (i < splited.length - 1) {
        b.append("|");
      }
      else {
        b.append("|");
        b.append(subPath);
        b.append("/.*");
      }
    }

    return String.format("(%s)", b.toString());
  }

  /**
   * Retorna o valor do atributo. Null caso o atributo no possua um "=".
   * 
   * @param att atributo.
   * @return valor do atributo. Null caso o atributo no possua um "=".
   */
  private static String getValue(String att) {
    int index = att.indexOf("=");
    if (index == -1) {
      return null;
    }

    return att.substring(index + 1);
  }

  /**
   * Retorna true se o atributo comear com "sga=", false caso contrrio.
   * 
   * @param att atributo.
   * @return true se o atributo comear com "sga=", false caso contrrio.
   */
  private static boolean isSGAAtt(String att) {
    return att.startsWith(SGA_ATT);
  }

  /**
   * Retorna true se o atributo comear com "permitir=", false caso contrrio.
   * 
   * @param att atributo.
   * @return true se o atributo comear com "permitir=", false caso contrrio.
   */
  private static boolean isAllowAtt(String att) {
    return att.startsWith(ALLOW_ATT);
  }

  /**
   * Retorna true se o atributo comear com "excluir=", false caso contrrio.
   * 
   * @param att atributo.
   * @return true se o atributo comear com "excluir=", false caso contrrio.
   */
  private static boolean isDenyAtt(String att) {
    return att.startsWith(DENY_ATT);
  }

  /**
   * Retorna true se o atributo comear com "caminho=", false caso contrrio.
   * 
   * @param att atributo.
   * @return true se o atributo comear com "caminho=", false caso contrrio.
   */
  private static boolean isPathAtt(String att) {
    return att.startsWith(PATH_ATT);
  }
}
