package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import csbase.exception.ParseException;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.parameters.validators.EnumerationParameterValidator;
import csbase.logic.algorithms.parameters.validators.SimpleParameterValidator;

/**
 * Enumerao para seleo mltipla.
 *
 * @author lmoreira
 */
public final class EnumerationListParameter extends
  ListParameter<EnumerationItem> {

  /**
   * A representao textual do tipo do parmetro.
   */
  public static final String TYPE = "ENUMERATION_LIST";

  /**
   * Expresso regular utilizada pelo mtodo {@link #setValueAsText(String)}
   * para separar a lista de {@link #items} da lista de objetos selecionados.
   */
  private static final Pattern valueAsTextPattern = Pattern.compile(
    "^\\{(.*)\\}:(\\{.*\\})?$");

  /**
   * Expresso regular utilizada pelo mtodo {@link #setValueAsText(String)}
   * para separar a lista de {@link #items} da lista de objetos selecionados.
   */
  private static final Pattern selectedItemsAsTextPattern = Pattern.compile(
    "^(\\{.*\\})$");

  /**
   * Expresso regular utilizada pelo mtodo {@link #setItemsAsText(String)}
   * para recriar cada item e adicion-los  lista de {@link #items}
   */
  private static final Pattern itemsPattern = Pattern.compile(
    "\\{id=(.*?)/label=(.*?)/value=(.*?)/description=(.*?)/visible=(.*?)\\}");

  /**
   * Formato da representao de um {@link EnumerationItem item} como um texto.
   */
  private static final String itemAsTextFormat =
    "{id=%s/label=%s/value=%s/description=%s/visible=%s}";

  /**
   * Os tens desta enumerao indexados pelos seus rtulos.
   */
  private Map<String, EnumerationItem> itemsByLabel;

  /**
   * Os tems preservando a ordem de insero.
   */
  private List<EnumerationItem> items;

  /**
   * Os observadores deste parmetro.
   */
  private transient List<EnumerationListParameterListener> listeners;

  /**
   * Indica se os itens devem ser ordenados.
   */
  private boolean mustSortItems;

  /**
   * TODO Este construtor ser removido numa reestruturao dos testes. Cria uma
   * enumerao para seleo mltipla.
   *
   * @param name O nome do parmetro (No aceita {@code null}).
   * @param label O rtulo do parmetro (No aceita {@code null}).
   * @param description A descrio do parmetro (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se o parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   */
  public EnumerationListParameter(String name, String label, String description,
    List<EnumerationItem> defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern) {
    this(name, label, description, defaultValue, isOptional, isVisible,
      commandLinePattern, false);
  }

  /**
   * Cria uma enumerao para seleo mltipla.
   *
   * @param name O nome do parmetro (No aceita {@code null}).
   * @param label O rtulo do parmetro (No aceita {@code null}).
   * @param description A descrio do parmetro (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se o parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param mustSortItems indica se a enumerao deve ficar ordenada.
   */
  public EnumerationListParameter(String name, String label, String description,
    List<EnumerationItem> defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern, boolean mustSortItems) {
    super(name, label, description, defaultValue, isOptional, isVisible, false,
      true, commandLinePattern);
    this.itemsByLabel = new HashMap<String, EnumerationItem>();
    this.items = new LinkedList<EnumerationItem>();
    this.listeners = new LinkedList<EnumerationListParameterListener>();
    this.mustSortItems = mustSortItems;
    /* TODO Passar essa informao como atributo para a superclasse. */
    setEnabled(false);
  }

  /**
   * Adiciona um observador de parmetros.
   *
   * @param listener O observador (No aceita {@code null}).
   */
  public final void addEnumerationListParameterListener(
    EnumerationListParameterListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    this.listeners.add(listener);
  }

  /**
   * <p>
   * Obtm os tens desta enumerao.
   * </p>
   *
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * @return A lista de tens (se no houver tens a lista estar vazio).
   */
  public List<EnumerationItem> getItems() {
    return Collections.unmodifiableList(this.items);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return TYPE;
  }

  /**
   * Modifica os itens da enumerao. Remove os itens anteriores.
   *
   * @param items Os tens (No aceita {@code null}).
   */
  public void setItems(List<EnumerationItem> items) {
    if (items == null) {
      throw new IllegalArgumentException("O parmetro items est nulo.");
    }
    setValue(null);
    this.items.clear();
    this.itemsByLabel.clear();
    for (EnumerationItem item : items) {
      addItem(item);
    }
    if (mustSortItems) {
      Collections.sort(this.items);
    }
    setEnabled(!items.isEmpty());
    for (EnumerationListParameterListener listener : this.listeners) {
      listener.itemsWereChanged(this);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean setValue(List<EnumerationItem> value) {
    List<EnumerationItem> selectedItems = value;
    if (value != null && mustSortItems) {
      selectedItems = new ArrayList<EnumerationItem>(value);
      Collections.sort(selectedItems);
    }
    return super.setValue(selectedItems);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getValueAsText() {
    StringBuilder sb = new StringBuilder();
    sb.append('{');
    for (int inx = 0; inx < items.size(); inx++) {
      EnumerationItem item = items.get(inx);
      if (0 < inx) {
        sb.append(',');
      }
      sb.append(getItemAsText(item));
    }
    sb.append('}');
    sb.append(':');
    String selectedValues = super.getValueAsText();
    if (null != selectedValues) {
      sb.append(selectedValues);
    }

    return sb.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAsText(String parameterValue) throws ParseException {

    if (parameterValue == null) {
      setValue(null);
    }
    else {
      Matcher match = valueAsTextPattern.matcher(parameterValue);
      if (match.find() && 0 < match.groupCount()) {
        /*
         * A lista de itens indica o universo de itens de enumerao
         * selecionveis.
         */
        String itemList = match.group(1);
        setItemsAsText(itemList);
        /*
         * A lista de labels indica os valores selecionados.  o valor da lista.
         * {label1,label2,label3}
         */
        String selecteds = 2 == match.groupCount() ? match.group(2) : null;
        super.setValueAsText(selecteds);
      }
      else {
        Matcher itemsMatch = selectedItemsAsTextPattern.matcher(parameterValue);
        if (itemsMatch.find() && itemsMatch.groupCount() == 1) {
          String selecteds = itemsMatch.group(1);
          super.setValueAsText(selecteds);
        }
        else {
          throw new ParseException("No  possivel atribuir o valor\n({0}),"
            + "j que ele no representa uma codificao vlida uma lista de "
            + "itens de enumerao.\nParmetro envolvido: {1}.", new Object[] {
                parameterValue, this });
        }
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getCommandItemValue(CommandLineContext context,
    EnumerationItem itemValue) {
    if (itemValue == null) {
      return null;
    }
    return itemValue.getCommandValue(context);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EnumerationItem getItemValueFromText(String itemValue)
    throws ParseException {
    EnumerationItem item = this.itemsByLabel.get(itemValue);
    if (item == null) {
      throw new ParseException(
        "{0} no  identificador de item da enumerao {1}.\nItems vlidos:{2}",
        itemValue, this, getItems());
    }
    return item;
  }

  /**
   * Adicionar tem de enumerao.
   *
   * @param item O tem (No aceita {@code null}).
   *
   * @return {@code true} se for possvel adicionar ou {@code false} se j
   *         existe um tem com este nome.
   */
  private boolean addItem(EnumerationItem item) {
    if (item == null) {
      throw new IllegalArgumentException("O parmetro item est nulo.");
    }
    if (this.itemsByLabel.containsKey(item.getLabel())) {
      return false;
    }
    this.itemsByLabel.put(item.getLabel(), item);
    this.items.add(item);
    return true;
  }

  /**
   * Cria os atributos transientes.
   *
   * @param in Leitor de objetos
   *
   * @throws IOException em caso de erro na leitura
   * @throws ClassNotFoundException se no encontrar a classe do objeto sendo
   *         lido.
   */
  private void readObject(java.io.ObjectInputStream in) throws IOException,
    ClassNotFoundException {
    in.defaultReadObject();
    listeners = new LinkedList<EnumerationListParameterListener>();
  }

  /**
   * Obtm o valor de um {@link EnumerationItem item} sob a forma de String.
   *
   * @param item O item.
   *
   * @return O valor de um {@link EnumerationItem item} sob a forma de String.
   */
  private String getItemAsText(EnumerationItem item) {
    String id = (item.getId() == null ? "" : item.getId());
    String label = (item.getLabel() == null ? "" : item.getLabel());
    String value = (item.getValue() == null ? "" : item.getValue());
    String description = (item.getDescription() == null ? ""
      : item.getDescription());
    String visible = Boolean.toString(item.isVisible());

    return String.format(itemAsTextFormat, id, label, value, description,
      visible);
  }

  /**
   * Modifica os itens da enumerao. Remove os itens anteriores.
   *
   * @param itemsAsText Os itens sob a forma de String (No aceita {@code null}
   *        ).
   *
   * @throws ParseException Se o valor passado no estiver em um formato aceito
   *         por este parmetro.
   */
  private void setItemsAsText(String itemsAsText) throws ParseException {
    List<EnumerationItem> itemList = new ArrayList<EnumerationItem>();
    Matcher matcher = itemsPattern.matcher(itemsAsText);
    while (matcher.find()) {
      if (matcher.groupCount() == 5) {
        String id = matcher.group(1);
        String label = matcher.group(2);
        String value = matcher.group(3);
        String description = matcher.group(4);
        boolean visible = Boolean.valueOf(matcher.group(5));

        EnumerationItem item = new DefaultEnumerationItem(id, label, value,
          description);
        item.setVisible(visible);
        itemList.add(item);
      }
      else {
        throw new ParseException("No  possivel atribuir o valor\n({0}),"
          + "j que ele no representa uma codificao vlida para itens de "
          + "enumerao.\nParmetro envolvido: {1}.", new Object[] { matcher
            .group(0), this });
      }
    }
    setItems(itemList);
  }

  /**
   * Indica se a enumerao deve ser ordenada.
   *
   * @return <code>true</code> caso a enumerao esteja ordenada ou
   *         <code>false</code> caso contrrio.
   */
  public boolean mustSortItems() {
    return mustSortItems;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected SimpleParameterValidator<EnumerationItem> createItemValidator() {
    return new EnumerationParameterValidator(isOptional());
  }
}
