package csbase.logic.algorithms.parameters;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import csbase.exception.ParseException;
import csbase.logic.algorithms.CommandLineBuilder;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.EnvironmentVariable;
import csbase.logic.algorithms.parameters.validators.URLListParameterValidator;
import csbase.logic.algorithms.parameters.validators.URLParameterValidator;

/**
 * <p>
 * Parmetro do tipo lista de URLs.
 * </p>
 */
public abstract class URLListParameter extends ListParameter<FileURLValue> {

  /**
   * Chave utilizada para informar o <b>caminho</b> do arquivo no mapa de
   * parmetros dos mtodos de importao ({@link #importValue(Map)}) e
   * exportao ({@link #exportValue()}).
   */
  private static String PATH = ".path";

  /**
   * Chave utilizada para informar o <b>tipo</b> do arquivo no mapa de
   * parmetros dos mtodos de importao ({@link #importValue(Map)}) e
   * exportao ({@link #exportValue()}).
   */
  private static String TYPE = ".type";

  /**
   * Chave utilizada para informar o <b>protocolo</b> da URL no mapa de
   * parmetros dos mtodos de importao ({@link #importValue(Map)}) e
   * exportao ({@link #exportValue()}).
   */
  private static final String PROTOCOL = ".protocol";

  /**
   * Chave utilizada para informar o <b>host</b> da URL no mapa de parmetros
   * dos mtodos de importao ({@link #importValue(Map)}) e exportao (
   * {@link #exportValue()}).
   */
  private static final String HOST = ".host";

  /**
   * Define o argumento na linha de comando informando o protoloco usado pelo
   * usurio.
   */
  private String localization;

  /**
   * Os tipos de arquivo que este parmetro aceita. Se ele se no se importa com
   * o tipo do arquivo, ele ser um array vazio.
   */
  protected String[] fileTypes;

  /**
   * O modo de funcionamento atual (apenas arquivos, apenas diretrios ou
   * ambos).
   */
  private FileParameterMode mode;

  /**
   * Indica se a URL deve existir.
   */
  private final boolean mustExist;

  /** Conjunto de protocolos aceitos na URL. */
  private EnumSet<URLProtocol> allowedProtocols;

  /**
   * Cria um parmetro do tipo lista de arquivos.
   * 
   * @param name O nome deste parmetro (No aceita {@code null}).
   * @param label O rtulo deste parmetro (No aceita {@code null}).
   * @param description A descrio deste parmetro (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se o valor do parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param localization define o argumento na linha de comando informando o
   *        protoloco usado pelo usurio.
   * @param fileTypes Os tipos dos arquivos aceitos neste parmetro (Aceita
   *        {@code null}).
   * @param mode O modo de funcionamento deste parmetro (No aceita
   *        {@code null}).
   * @param mustSort Indica se os caminhos para os arquivos precisam ser
   *        ordenados.
   * @param mustExist Indica se os arquivso devem existir.
   * @param allowedProtocols conjunto de protocolos aceitos na URL.
   */
  public URLListParameter(String name, String label, String description,
    List<FileURLValue> defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern, String localization, String[] fileTypes,
    FileParameterMode mode, boolean mustSort, boolean mustExist,
    EnumSet<URLProtocol> allowedProtocols) {
    super(name, label, description, defaultValue, isOptional, isVisible,
      mustSort, false, commandLinePattern);
    this.localization = localization;
    setFileTypes(fileTypes);
    this.mustExist = mustExist;
    this.allowedProtocols = allowedProtocols;
    setMode(mode);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, Object> exportValue() {
    Map<String, Object> valuesByName = new HashMap<String, Object>();
    Collection<Map<String, String>> exportableValues =
      new LinkedList<Map<String, String>>();
    List<FileURLValue> urls = getValue();
    if (urls != null) {
      for (FileURLValue url : urls) {
        Map<String, String> exportableValue = new HashMap<String, String>();
        exportableValue.put(PATH, url.getPath());
        exportableValue.put(TYPE, url.getType());
        URLProtocol protocol = url.getProtocol();
        if (protocol != null) {
          exportableValue.put(PROTOCOL, protocol.getType());
        }
        exportableValue.put(HOST, url.getHost());
        exportableValues.add(Collections.unmodifiableMap(exportableValue));
      }
      valuesByName.put(getName(), Collections
        .unmodifiableCollection(exportableValues));
    }
    return Collections.unmodifiableMap(valuesByName);
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings("unchecked")
  @Override
  public void importValue(Map<String, Object> parameterValues) {
    Collection<Map<String, String>> exportableValues =
      (Collection<Map<String, String>>) parameterValues.get(getName());
    if (exportableValues != null) {
      List<FileURLValue> urls = new ArrayList<FileURLValue>();
      for (Map<String, String> exportableValue : exportableValues) {
        String path = exportableValue.get(PATH);
        String type = exportableValue.get(TYPE);
        String protocolName = exportableValue.get(PROTOCOL);
        String host = exportableValue.get(HOST);
        FileURLValue url;
        if (protocolName != null) {
          try {
            URLProtocol protocol =
              new URLProtocolConverter().valueOf(protocolName);
            url = new FileURLValue(path, type, protocol, host);
          }
          catch (ParseException e) {
            throw new IllegalArgumentException("Protocolo " + protocolName
              + " invlido");
          }
        }
        else {
          url = new FileURLValue(path, type);
        }
        if (!urls.contains(url)) {
          urls.add(url);
        }
      }
      if (urls.isEmpty()) {
        setValue(null);
      }
      else {
        setValue(urls);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getCommandValue(CommandLineContext context) {
    String commandValue = super.getCommandValue(context);
    if (commandValue == null) {
      return null;
    }
    List<FileURLValue> value = getValue();
    if (value == null || value.isEmpty()) {
      return null;
    }

    /*
     * Retorna o protocolo de qualquer url selecionada, j que assumimos que
     * todas as URLs selecionadas usam o mesmo protocolo.
     */
    FileURLValue url = value.iterator().next();
    URLProtocol protocol = url.getProtocol();

    String f = "%s %s=%s";
    return String
      .format(f, commandValue, getLocalization(), protocol.getType());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getCommandItemValue(CommandLineContext context, FileURLValue url) {
    if (url == null) {
      return null;
    }
    String urlPath = url.getPath();
    if (urlPath.startsWith(CommandLineBuilder.REFERENCE_VAR_CHAR)) {
      return urlPath;
    }
    char fileSeparator = context.getFileSeparator();
    String path = url.getPath(fileSeparator);
    URLProtocol protocol = url.getProtocol();
    switch (protocol) {
      case PROJECT:
        String filePath = url.getPath(fileSeparator);
        if (filePath.startsWith("/")) {
          filePath = filePath.substring(1);
        }
        return CommandLineBuilder.makePathWithEnvironmentVariable(
          EnvironmentVariable.PROJECT_DIR, filePath, fileSeparator);
      default:
        String commandValue = protocol.getCommandValue();
        if (commandValue != null) {
          return commandValue + path;
        }
        else {
          return path;
        }
    }
  }

  /**
   * Obtm os tipos de arquivo que este parmetro aceita.
   * 
   * @return Os tipos de arquivo ou um array vazio se ele no se importar com um
   *         tipo especfico.
   */
  public String[] getFileTypes() {
    return fileTypes;
  }

  /**
   * Obtm o modo de funcionamento (apenas arquivos, apenas diretrios ou ambos)
   * deste parmetro.
   * 
   * @return O modo de funcionamento.
   */
  public FileParameterMode getMode() {
    return mode;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getValueAsText() {
    List<FileURLValue> urls = getValue();
    if (urls == null || urls.isEmpty()) {
      return null;
    }
    String separator = "";
    StringBuilder valueAsText = new StringBuilder();

    for (FileURLValue url : urls) {
      String stringValue = FileURLValue.getStringValue(url);
      if (stringValue != null) {
        valueAsText.append(separator);
        valueAsText.append(stringValue);
        separator = "|";
      }
    }
    return valueAsText.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FileURLValue getItemValueFromText(String elementAsText)
    throws ParseException {
    return FileURLValue.getURLFromString(mode, elementAsText);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAsText(String parameterValue) throws ParseException {
    if ((parameterValue == null) || (parameterValue.length() == 0)) {
      setValue(null);
      return;
    }
    List<FileURLValue> urls = new ArrayList<FileURLValue>();
    String[] components = parameterValue.split("\\|");
    for (String component : components) {
      FileURLValue url = getItemValueFromText(component);
      urls.add(url);
    }
    setValue(urls);
  }

  /**
   * Move para cima o elemento do ndice
   * 
   * @param index ndice
   */
  public void moveUp(final int index) {
    final List<FileURLValue> urls = getValue();
    if (urls == null) {
      return;
    }
    final int size = urls.size();
    if (size == 0 || index == 0) {
      return;
    }

    final FileURLValue element = urls.remove(index);
    urls.add(index - 1, element);
    fireValueWasChangedEvent();
  }

  /**
   * Move para cima o elemento do ndice
   * 
   * @param index ndice
   */
  public void moveDown(final int index) {
    final List<FileURLValue> urls = getValue();
    if (urls == null) {
      return;
    }
    final int size = urls.size();
    if (size == 0 || index == size - 1) {
      return;
    }

    final FileURLValue element = urls.remove(index);
    urls.add(index + 1, element);
    fireValueWasChangedEvent();
  }

  /**
   * Atribui um tipo de arquivo especfico a este parmetro.
   * 
   * @param fileTypes Os tipos de arquivo (Aceita {@code null}).
   */
  public void setFileTypes(String[] fileTypes) {
    if (fileTypes == null) {
      this.fileTypes = new String[0];
    }
    else {
      this.fileTypes = fileTypes;
    }
    fireVisiblityWasChangedEvent();
  }

  /**
   * Atribui o modo de funcionamento (apenas arquivo, apenas diretrio ou ambos)
   * deste parmetro.
   * 
   * @param mode O modo de funcionamento (No aceita {@code null}).
   */
  protected void setMode(FileParameterMode mode) {
    if (mode == null) {
      throw new IllegalArgumentException("O parmetro mode est nulo.");
    }
    this.mode = mode;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getExpressionValue() {
    return null;
  }

  /**
   * Define o argumento na linha de comando informando o protoloco usado pelo
   * usurio.
   * 
   * @return localizao.
   */
  public String getLocalization() {
    return localization;
  }

  /**
   * Obtm o conjunto de protocolos aceitos na URL.
   * 
   * @return conjunto de protocolos.
   */
  public EnumSet<URLProtocol> getAllowedProtocols() {
    return allowedProtocols;
  }

  /**
   * Obtm o protocolo padro do parmetro (caso no tenha sido especificado
   * nenhum).
   * 
   * @return protocolo padro.
   */
  public URLProtocol getDefaultProtocol() {
    if (allowedProtocols.size() == 0) {
      return FileURLValue.DEFAULT_PROTOCOL;
    }
    return allowedProtocols.iterator().next();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public URLListParameterValidator createParameterValidator() {
    return new URLListParameterValidator(createItemValidator());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected URLParameterValidator createItemValidator() {
    return new URLParameterValidator(isOptional());
  }

  /**
   * Indica se os arquivos selecionados precisam existir.
   * 
   * @return mustExist verdadeiro se os arquivos precisam existir, ou falso,
   *         caso contrrio.
   */
  public boolean mustExist() {
    return mustExist;
  }

}