package csbase.logic.algorithms.parsers;

import csbase.exception.ParseException;
import csbase.logic.ProjectFileType;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.logic.algorithms.parameters.OutputFileParameter;
import csbase.logic.algorithms.parameters.OutputURLParameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parsers.elements.ParameterStructure;
import csbase.logic.algorithms.parsers.elements.ParsedSimpleParameter;
import csbase.logic.algorithms.parsers.elements.attributes.BooleanAttribute;
import csbase.logic.algorithms.parsers.elements.attributes.FileElementPipeAcceptanceAttribute;

/**
 * <p>
 * Analisador de {@link OutputFileParameter}.
 * </p>
 *
 * <p>
 * Este parser l os atributos de parmetros do tipo "arquivo de sada". O
 * elemento corrente do {@link XmlParser analisador de XML} precisa ser um
 * elemento {@link OutputFileParameter}.
 * </p>
 */
public class OutputFileParameterParser extends
  AbstractFileParameterParser<OutputFileParameter> {

  /**
   * <p>
   * O elemento {@value #OUTPUT_FILE_PARAMETER_ELEMENT}: descreve as
   * propriedades de um {@link OutputFileParameter parmetro do tipo arquivo de
   * sada}.
   * </p>
   * <p>
   *  filho do elemento {@link ParameterGroup}.
   * </p>
   */
  public static final String OUTPUT_FILE_PARAMETER_ELEMENT = "arquivo_de_saida";

  /**
   * <p>
   * O atributo
   * {@value #OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_ATTRIBUTE} do
   * elemento {@link OutputFileParameter}.
   * </p>
   *
   * <p>
   * Indica se o {@link OutputFileParameter arquivo de sada} deve permitir
   * sadas mltiplas,  opcional, o seu valor-padro 
   * {@link #OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_DEFAULT_VALUE} e
   *  do tipo booleano.
   * </p>
   */
  protected static final String OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_ATTRIBUTE =
    "forcar_extensao";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_ATTRIBUTE} do
   * elemento {@link OutputFileParameter}
   * </p>
   * <p>
   * O seu valor 
   * {@value #OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_DEFAULT_VALUE}.
   * </p>
   */
  protected static final boolean OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_DEFAULT_VALUE =
    false;

  /**
   * <p>
   * O atributo
   * {@value #OUTPUT_FILE_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_ATTRIBUTE} do
   * elemento {@link OutputURLParameter}.
   * </p>
   *
   * <p>
   * Indica se o {@link OutputFileParameter arquivo de sada} deve criar uma
   * extenso se ela no for informada,  opcional, o seu valor-padro 
   * {@link #OUTPUT_FILE_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_DEFAULT_VALUE}
   * e  do tipo booleano.
   * </p>
   */
  protected static final String OUTPUT_FILE_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_ATTRIBUTE =
    "permitir_bifurcacao";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #OUTPUT_FILE_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_ATTRIBUTE} do
   * elemento {@link OutputURLParameter}
   * </p>
   * <p>
   * O seu valor 
   * {@value #OUTPUT_FILE_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_DEFAULT_VALUE}.
   * </p>
   */
  protected static final boolean OUTPUT_FILE_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_DEFAULT_VALUE =
    false;

  /**
   * {@inheritDoc}
   */
  @Override
  public OutputFileParameter createFileParameter(
    ParsedSimpleParameter definition, String[] types, FileParameterMode mode)
    throws ParseException {

    FileParameterPipeAcceptance usesPipe = definition.getAttributeValue(
      FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_ATTRIBUTE);
    boolean mustForceExtension = definition.getAttributeValue(
      OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_ATTRIBUTE);
    boolean useRootDirectoryAsDefault = definition.getAttributeValue(
      FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_ATTRIBUTE);
    FileURLValue defaultValue = null;
    if (useRootDirectoryAsDefault) {
      if (mode == FileParameterMode.REGULAR_FILE) {
        throw new ParseException(
          "O atributo {0}  invlido quando o parmetro de sada  um arquivo.",
          FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_ATTRIBUTE);
      }
      defaultValue = new FileURLValue(".", ProjectFileType.DIRECTORY_TYPE);
    }

    boolean useFilter = definition.getAttributeValue(
      FILE_PARAMETER_ELEMENT_USE_FILTER_ATTRIBUTE);

    boolean allowMultipleOutput = definition.getAttributeValue(
      OUTPUT_FILE_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_ATTRIBUTE);

    if (types != null && types.length > 1) {
      throw new ParseException(
        "Arquivo de sada no admite mltiplos tipos de arquivo");
    }

    OutputFileParameter parameter = new OutputFileParameter(definition
      .getName(), definition.getLabel(), definition.getDescription(),
      defaultValue, definition.isOptional(), definition.isVisible(), definition
        .getCommandLinePattern(), types, mode, usesPipe, mustForceExtension,
      useFilter, allowMultipleOutput);
    return parameter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ParameterStructure<OutputFileParameter> getParameterStructure() {
    ParameterStructure<OutputFileParameter> fileStructure = getFileStructure(
      OUTPUT_FILE_PARAMETER_ELEMENT, OutputFileParameter.class);
    fileStructure.addAttribute(new FileElementPipeAcceptanceAttribute(
      FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_ATTRIBUTE,
      FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_DEFAULT_VALUE));
    fileStructure.addAttribute(new BooleanAttribute(
      OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_ATTRIBUTE,
      OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_DEFAULT_VALUE));
    fileStructure.addAttribute(new BooleanAttribute(
      FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_ATTRIBUTE,
      FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_DEFAULT_VALUE));
    fileStructure.addAttribute(new BooleanAttribute(
      FILE_PARAMETER_ELEMENT_USE_FILTER_ATTRIBUTE,
      FILE_PARAMETER_ELEMENT_USE_FILTER_DEFAULT_VALUE));
    fileStructure.addAttribute(new BooleanAttribute(
      OUTPUT_FILE_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_ATTRIBUTE,
      OUTPUT_FILE_PARAMETER_ELEMENT_ALLOW_MULTIPLE_OUTPUT_DEFAULT_VALUE));
    return fileStructure;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean acceptBothCategory() {
    return false;
  }
}
