package csbase.logic.algorithms.parsers.elements;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import csbase.logic.algorithms.parsers.elements.attributes.IElementAttribute;

/**
 * Estrutura de um elemento.
 */
public class ElementStructure<T> implements IElementStructure<T> {

  /**
   * Mapa de atributos do elemento.
   */
  private final Map<String, IElementAttribute<?>> attributeMap =
    new HashMap<>();

  /**
   * Nome do elemento.
   */
  private final String name;

  /**
   * Lista de elementos filhos.
   */
  private final List<IElementStructure<?>> children;

  /**
   * Classe do elemento.
   */
  private final Class<T> parameterClass;

  /**
   * Construtor.
   *
   * @param name nome do elemento.
   * @param parameterClass classe do elemento.
   */
  public ElementStructure(String name, Class<T> parameterClass) {
    this(name, parameterClass, new ArrayList<>());
  }

  /**
   * Construtor.
   *
   * @param name nome do elemento.
   * @param parameterClass classe do elemento.
   * @param children lista de elementos filhos.
   */
  public ElementStructure(String name, Class<T> parameterClass,
    List<IElementStructure<?>> children) {
    this.name = name;
    this.parameterClass = parameterClass;
    this.children = children;
  }

  /**
   * Adiciona um elemento filho.
   *
   * @param childStructure elemento filho.
   */
  public void addChild(IElementStructure childStructure) {
    children.add(childStructure);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    return name;
  }

  /**
   * Adiciona um atributo.
   *
   * @param attribute o atributo.
   */
  public void addAttribute(IElementAttribute<?> attribute) {
    attributeMap.put(attribute.getName(), attribute);
  }

  /**
   * Obtm um atributo a partir do seu nome.
   *
   * @param name nome do atributo
   * @return o atributo.
   */
  public IElementAttribute<?> getAttribute(String name) {
    return attributeMap.get(name);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<IElementAttribute<?>> getAttributes() {
    return new ArrayList<>(attributeMap.values());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<IElementStructure<?>> getChildElements() {
    return children;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Class<T> getElementClass() {
    return parameterClass;
  }
}
