package csbase.logic.algorithms.parsers.elements.attributes;

import csbase.exception.ParseException;

/**
 * Atributo do tipo booleano.
 */
public class BooleanAttribute extends AbstractElementAttribute<Boolean> {

  /**
   * Strings aceitas para booleanos falsos
   */
  static final String[] FALSE_VALUES =
    new String[] { "falso", "false", "F", "nao", "no", "no", "N" };

  /**
   * Strings aceitas para booleanos verdadeiros
   */
  static final String[] TRUE_VALUES =
    new String[] { "verdadeiro", "true", "V", "T", "sim", "yes", "S", "Y" };

  /**
   * Consturor.
   *
   * @param name nome do atributo
   * @param optional indicativo de opcionalidade.
   * @param defaultValue valor padro do atributo.
   * @param defaultValueAttribute atributo com valor padro.
   */
  BooleanAttribute(String name, boolean optional, Boolean defaultValue,
    BooleanAttribute defaultValueAttribute) {
    super(name, Boolean.class, optional, defaultValue, defaultValueAttribute);
  }

  /**
   * Construtor para atributos obrigatrios.
   *
   * @param name nome do atributo.
   */
  public BooleanAttribute(String name) {
    this(name, false, null, null);
  }

  /**
   * Construtor para atributos opcionais.
   *
   * @param name nome do atributo.
   * @param defaultValue valor padro do atributo.
   */
  public BooleanAttribute(String name, Boolean defaultValue) {
    this(name, true, defaultValue, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public StringToValueConverter<Boolean> getValueConverter() {
    return new StringToBooleanConverter();
  }

  /**
   * Conversor de valores booleanos para string.
   */
  private class StringToBooleanConverter implements
    StringToValueConverter<Boolean> {

    @Override
    public Boolean valueOf(String value) throws ParseException {
      for (final String trueValue : TRUE_VALUES) {
        if (trueValue.equalsIgnoreCase(value)) {
          return true;
        }
      }
      for (final String falseValue : FALSE_VALUES) {
        if (falseValue.equalsIgnoreCase(value)) {
          return false;
        }
      }
      throw new ParseException(
        "O valor {1} do atributo {0} no  do tipo booleano.", getName(),
        value);
    }
  }
}
