/*
 * $Id: FileTransferServiceInterface.java 91988 2009-05-29 18:04:21Z clinio $
 */

package csbase.remote;

import java.rmi.RemoteException;
import java.util.List;

import csbase.logic.User;
import csbase.logic.filetransferservice.FileTransferConnection;
import csbase.logic.filetransferservice.FileTransferElement;
import csbase.logic.filetransferservice.FileTransferPredefinedConnection;
import csbase.logic.filetransferservice.FileTransferRequest;

/**
 * Interface remota para acesso ao servio de transferncia de arquivos via FTP.
 * 
 * @author Tecgraf/PUC-Rio
 */
public interface FileTransferServiceInterface extends ServiceInterface {

  /**
   * Nome do servio para o <code>ServiceManager</code>
   */
  public static final String SERVICE_NAME = "FileTransferService";

  /**
   * Recupera a lista de connexes predefinidas do servidor.
   * 
   * @return a lista
   * @throws RemoteException em caso de falha.
   */
  public List<FileTransferPredefinedConnection> getPredefinedConnections()
    throws RemoteException;

  /**
   * Mtodo para verificar a existncia e acessibilidade de um servidor
   * 
   * @param connection conexo.
   * @return indicativo.
   * @throws RemoteException em caso de falha.
   */
  public boolean checkConnection(final FileTransferConnection connection)
    throws RemoteException;

  /**
   * Recupera um arquivo remoto de um servidor para o projeto.
   * 
   * @param connection a conexo.
   * @param remoteFilePath caminho do arquivo remoto usando o separador '/'.
   * @param localFilePath caminho do arquivo local no projeto.
   * @param size tamanho da transferncia.
   * @throws RemoteException em caso de falha de comunicao.
   */
  public void download(FileTransferConnection connection,
    String remoteFilePath, String[] localFilePath, long size)
    throws RemoteException;

  /**
   * Envia um arquivo do projeto para um servidor.
   * 
   * @param connection a conexo.
   * @param remoteFilePath caminho do arquivo remoto usando o separador '/'.
   * @param localFilePath caminho do arquivo local no projeto.
   * @param size tamanho da transferncia
   * @throws RemoteException em caso de falha de comunicao.
   */
  public void upload(FileTransferConnection connection, String remoteFilePath,
    String[] localFilePath, long size) throws RemoteException;

  /**
   * Retorna a lista de requisies do usurio armazenadas no servidor.
   * 
   * @param user usurio CSBASE (pode ser null)
   * @return a lista completa (<code>user == null</code>) ou a lista de
   *         requisies do usurio.
   * @throws RemoteException em caso de falha de comunicao.
   */
  public List<FileTransferRequest> getAllRequests(final User user)
    throws RemoteException;

  /**
   * Cancelamento de requisio.
   * 
   * @param idList lista de ids de requisio
   * @throws RemoteException em caso de falha de comunicao.
   */
  public void interruptRequest(final List<String> idList)
    throws RemoteException;

  /**
   * Remoo de requisio.
   * 
   * @param idList lista de ids de requisio
   * @throws RemoteException em caso de falha de comunicao.
   */
  public void removeRequest(final List<String> idList) throws RemoteException;

  /**
   * Listagem de arquivos remotos em um servidor FTP
   * 
   * @param connection a conexo.
   * @param remoteDirPath diretrio remoto
   * @return uma lista
   * @throws RemoteException se houver falha
   */
  public List<FileTransferElement> listContent(
    FileTransferConnection connection, final String remoteDirPath)
    throws RemoteException;
}
