package csbase.logic;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * Classe abstrata que deve ser implementada quando se precisa de uma permisso
 * com atributos que podem ser escolhidos de uma lista que ser populada de
 * acordo com a implementao dos mtodos: {@link #getDomain()} e
 * {@link #getRange(String)}.
 */
public abstract class ChoicePermission extends Permission {

  /** Mapa dos atributos desta permisso e seus respectivos valores. */
  private Map<?, ?> attributes;

  /**
   * Construtor.
   */
  public ChoicePermission() {
  }

  /**
   * Construtor.
   * 
   * @param name Nome da permisso.
   * @param description Descrio da permisso.
   * @param attributes Mapa dos atributos desta permisso e seus respectivos
   *        valores.
   */
  public ChoicePermission(String name, String description, Map<?, ?> attributes) {
    super(name, description);
    this.attributes = attributes;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equalContents(Object obj) {
    return super.equalContents(obj)
      && attributes.equals(((ChoicePermission) obj).attributes);
  }

  /**
   * Obtm os atributos na forma de uma <code>Map</code> cuja chave  o nome do
   * atributo e o valor uma Lista(List) de Strings representando os valores do
   * atributo.
   * 
   * @return Map de atributos.
   */
  public Map<?, ?> getAttributes() {
    return attributes;
  }

  /**
   * Verifica se uma Map de atributos est contida na Map de atributos da
   * permisso.
   * 
   * @param attrs atributos a serem verificados.
   * 
   * @return true se os valores dos atributos passados por parmetro estiverem
   *         presente na permisso.
   */
  public boolean hasPermission(Map<?, ?> attrs) {
    Iterator<?> permissionDomainIterator = this.attributes.keySet().iterator();
    List<?> permissionValues;
    while (permissionDomainIterator.hasNext()) {
      String key = (String) permissionDomainIterator.next();
      permissionValues = (List<?>) this.attributes.get(key);
      if (permissionValues.containsAll((Collection<?>) attrs.get(key))) {
        return true;
      }
    }
    return false;
  }

  /**
   * Atribui os atributos na forma de uma <code>Map</code> cuja chave  o nome
   * do atributo e o valor uma Lista(List) de Strings representando os valores
   * do atributo.
   * 
   * @param attributes atributos da permisso.
   */
  public void setAttributes(Map<?, ?> attributes) {
    this.attributes = attributes;
  }

  /**
   * Retorna um Map com os pares {String attributeName, Boolean
   * isMultiSelection} de domnio de uma permisso, onde
   * <code>isMultiSelection</code> indica se  permitido a seleo de mais de um
   * valor para o atributo. OBS.: Se for desejado que se mantenha a ordem em que
   * foi includo os elementos na Map, aconselha-se o uso da implementao
   * <code>LinkedHashMap</code>.
   * 
   * @return Map com o domnio dos atributos.
   */
  public abstract Map<String, Object> getDomain();

  /**
   * Retorna um Map com o conjunto imagem de um atributo do domnio da
   * permisso. A chave do Map  o valor de exibio e o valor  um valor da
   * imagem. OBS.: Se for desejado que se mantenha a ordem em que foi includo
   * os elementos na Map, aconselha-se o uso da implementao
   * <code>LinkedHashMap</code>.
   * 
   * @param attribute Atributo contido no domnio da permisso.
   * 
   * @return Map com o conjunto imagem.
   * 
   * @throws Exception se ocorrer algum problema na recuperao do conjunto
   *         imagem.
   */
  public abstract Map<String, Object> getRange(String attribute)
    throws Exception;
}
