package csbase.logic.algorithms.parsers;

import csbase.exception.ParseException;
import csbase.logic.ProjectFileType;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.logic.algorithms.parameters.InputFileParameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parsers.elements.ParameterStructure;
import csbase.logic.algorithms.parsers.elements.ParsedSimpleParameter;
import csbase.logic.algorithms.parsers.elements.attributes.BooleanAttribute;
import csbase.logic.algorithms.parsers.elements.attributes.FileElementPipeAcceptanceAttribute;

/**
 * <p>
 * Analisador de {@link InputFileParameter}.
 * </p>
 *
 * <p>
 * Este parser l os atributos de parmetros do tipo "arquivo de entrada"
 * simples. O elemento corrente do {@link XmlParser analisador de XML} precisa
 * ser um elemento {@link InputFileParameter}.
 * </p>
 */
public class InputFileParameterParser extends
  AbstractFileParameterParser<InputFileParameter> {

  /**
   * <p>
   * O elemento {@value #INPUT_FILE_PARAMETER_ELEMENT}: descreve as propriedades
   * de um {@link InputFileParameter parmetro do tipo arquivo de entrada}.
   * </p>
   * <p>
   *  filho do elemento {@link ParameterGroup}.
   * </p>
   */
  public static final String INPUT_FILE_PARAMETER_ELEMENT =
    "arquivo_de_entrada";

  /**
   * <p>
   * Atributo do elemento {@value #INPUT_FILE_PARAMETER_ELEMENT} que indica se o
   * arquivo de entrada deve ser utilizado para carregar o valor de outros
   * parmetros. Essa carga  feita no cliente.
   */
  public static final String INPUT_FILE_PARAMETER_USE_TO_LOAD_PARAM = "carregar_parametros";

  /**
   * Valor padro do atributo {@value #INPUT_FILE_PARAMETER_USE_TO_LOAD_PARAM}.
   */
  public static final boolean INPUT_FILE_PARAMETER_USE_TO_LOAD_PARAM_DEFAULT_VALUE = false;

  /**
   * {@inheritDoc}
   */
  @Override
  public InputFileParameter createFileParameter(
    ParsedSimpleParameter definition, String[] types, FileParameterMode mode)
    throws ParseException {

    FileParameterPipeAcceptance usesPipe = definition.getAttributeValue(
      FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_ATTRIBUTE);

    if (usesPipe != FileParameterPipeAcceptance.FALSE && types.length > 1) {
      throw new ParseException(
        "O valor {0} do atributo {1}  invlido quando se tem mais de um "
          + "tipo" + " " + "de arquivo definido no parmetro.", usesPipe,
        FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_ATTRIBUTE);
    }

    boolean useRootDirectoryAsDefault = definition.getAttributeValue(
      FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_ATTRIBUTE);
    FileURLValue defaultValue = null;
    if (useRootDirectoryAsDefault) {
      if (mode == FileParameterMode.REGULAR_FILE) {
        throw new ParseException(
          "O atributo {0}  invlido quando o parmetro de entrada  um "
            + "arquivo.",
          FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_ATTRIBUTE);
      }
      defaultValue = new FileURLValue(".", ProjectFileType.DIRECTORY_TYPE);
    }

    boolean useFilter = definition.getAttributeValue(
      FILE_PARAMETER_ELEMENT_USE_FILTER_ATTRIBUTE);

    boolean useToLoadParam = definition.getAttributeValue(
      INPUT_FILE_PARAMETER_USE_TO_LOAD_PARAM);

    InputFileParameter parameter = new InputFileParameter(definition.getName(),
      definition.getLabel(), definition.getDescription(), defaultValue,
      definition.isOptional(), definition.isVisible(), definition
        .getCommandLinePattern(), types, mode, usesPipe, useFilter, useToLoadParam);
    return parameter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ParameterStructure<InputFileParameter> getParameterStructure() {
    ParameterStructure<InputFileParameter> fileStructure = getFileStructure(
      INPUT_FILE_PARAMETER_ELEMENT, InputFileParameter.class);
    fileStructure.addAttribute(new FileElementPipeAcceptanceAttribute(
      FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_ATTRIBUTE,
      FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_DEFAULT_VALUE));
    fileStructure.addAttribute(new BooleanAttribute(
      FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_ATTRIBUTE,
      FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_DEFAULT_VALUE));
    fileStructure.addAttribute(new BooleanAttribute(
      FILE_PARAMETER_ELEMENT_USE_FILTER_ATTRIBUTE,
      FILE_PARAMETER_ELEMENT_USE_FILTER_DEFAULT_VALUE));
    fileStructure.addAttribute(new BooleanAttribute(
        INPUT_FILE_PARAMETER_USE_TO_LOAD_PARAM,
        INPUT_FILE_PARAMETER_USE_TO_LOAD_PARAM_DEFAULT_VALUE));
    return fileStructure;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean acceptBothCategory() {
    return true;
  }
}
