/**
 * $Id: CommonProjectInfo.java 175530 2016-08-24 13:08:10Z isabella $
 */

package csbase.logic;

import java.io.IOException;
import java.io.InvalidClassException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Hashtable;

/**
 * Modela as informaes de um projeto.
 */
public class CommonProjectInfo implements Serializable {
  /**
   * Identificao da verso da classe, segundo a serializao padro de Java.
   */
  static final long serialVersionUID = 2761371397943092528L;

  /**
   * Identificao interna, para podermos evoluir a classe mantendo
   * compatibilidade com verses anteriores.
   */
  private static long internalVersion = 1;

  /**
   * Identificador do projeto.
   */
  public Object projectId;

  /**
   * Identificao do usurio dono do projeto.
   */
  public Object userId;

  /**
   * Nome do projeto.
   */
  public String name;

  /**
   * Descrio do projeto.
   */
  public String description;

  /**
   * Todos os atributos do CommonProjectInfo so armazenados nesta Hashtable.
   */
  protected Hashtable<String, Object> attributes;

  /**
   * Retorna o valor de um atributo.
   * 
   * @param name Nome do atributo.
   * @return O valor do atributo.
   */
  public Object getAttribute(String name) {
    return attributes.get(name);
  }

  /**
   * Atribui um valor a um atributo.
   * 
   * @param name Nome do atributo.
   * @param value Valor do atributo. Se for nulo, o atributo  removido, caso
   *        exista.
   */
  public void setAttribute(String name, Object value) {
    if (value == null) {
      attributes.remove(name);
    }
    else {
      attributes.put(name, value);
    }
  }

  /**
   * Retorna uma tabela com todos os atributos do projeto.
   * 
   * @return Tabela com todos os atributos do projeto.
   */
  public Hashtable<String, Object> getAttributes() {
    return attributes;
  }

  /**
   * Redefine a tabela com os atributos do projeto.
   * 
   * @param attributes Nova tabela com os atributos do projeto.
   */
  public void setAttributes(Hashtable<String, Object> attributes) {
    this.attributes = attributes;
  }

  /**
   * Construtor.
   */
  public CommonProjectInfo() {
    attributes = new Hashtable<String, Object>();
  }

  /**
   * Cuida da serializao do objeto para podermos evoluir a classe mantendo
   * compatibilidade com verses anteriores.
   * 
   * @param out Stream de sada onde sero gravados os dados.
   * @throws IOException Em caso de erro de IO.
   */
  private void writeObject(ObjectOutputStream out) throws IOException {
    out.writeLong(internalVersion);
    out.writeObject(projectId);
    out.writeObject(userId);
    out.writeObject(name);
    out.writeObject(description);
    out.writeObject(attributes);
  }

  /**
   * Cuida da serializao do objeto para podermos evoluir a classe mantendo
   * compatibilidade com verses anteriores.
   * 
   * @param in input stream.
   * @throws IOException Em caso de erro de IO.
   * @throws ClassNotFoundException Caso alguma classe componente n seja
   *         encontrada.
   */
  @SuppressWarnings("unchecked")
  private void readObject(ObjectInputStream in)
    throws IOException, ClassNotFoundException {
    long version = in.readLong();
    if (version != internalVersion) {
      throw new InvalidClassException(
        "No h converso codificada para ler da verso " + version);
    }
    projectId = in.readObject();
    userId = in.readObject();
    name = (String) in.readObject();
    description = (String) in.readObject();
    attributes = (Hashtable<String, Object>) in.readObject();
  }

}
