/*
 * $Id$
 */
package csbase.logic;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Responsvel por centralizar todas as informaes a respeito dos servidores
 * que so manipulados por uma instncia de {@link ServerManager}
 * 
 * @author Tecgraf/PUC-Rio
 * 
 */
public final class ServerManagerData {

  /** Os dados monitorados de cada servidor */
  private Map<ServerURI, MonitoredServer> servers;

  /** O servidor default. Inicialmente servidor onde foi feito o primeiro login */
  private ServerURI defaultServerURI;

  /** Lock de acesso ao defaultServerURI */
  private final ReentrantLock defaultServerLock = new ReentrantLock();

  /** Lista de ouvintes comum a todos os servidores monitorados */
  private List<MonitoredServerListener> commonListeners;

  /**
   * Indica se mensagens informativas devem ser exibidas. Se for
   * <code>false</code>, apenas erros e alertas sero exibidos.
   */
  private boolean verbose = true;

  /**
   * Instncia nica em caso de compartilhamento dos dados monitorados pelas
   * instncias de ServerManager
   */
  private static ServerManagerData instance = new ServerManagerData();

  /**
   * Construtor.
   */
  protected ServerManagerData() {
    this.servers = new Hashtable<ServerURI, MonitoredServer>();
    this.commonListeners =
      Collections.synchronizedList(new ArrayList<MonitoredServerListener>());
  }

  /**
   * @return A instncia desta classe, nica se estiver sendo compartilhada
   *         entre as instncias de {@link ServerManager}
   */
  protected static ServerManagerData getInstance() {
    return instance;
  }

  /**
   * Recupera a instncia de monitorao de um servidor
   * 
   * @param uri A URI que identifica o servidor
   * @return A instncia de monitorao ({@link MonitoredServer})
   */
  protected MonitoredServer getMonitoredServer(ServerURI uri) {
    synchronized (this.servers) {
      MonitoredServer sdata = this.servers.get(uri);
      if (sdata == null) {
        throw new IllegalStateException(MessageFormat.format(
          "Servidor {0} no est na monitorao.", uri));
      }
      return sdata;
    }
  }

  /**
   * Define o servidor default
   * 
   * @param serverURI A URI do servidor
   */
  protected final void setDefaultServer(ServerURI serverURI) {
    if (serverURI == null) {
      throw new IllegalArgumentException("serverURI == null");
    }

    try {
      this.defaultServerLock.lock();

      synchronized (this.servers) {
        if (this.defaultServerURI != null) {
          this.getMonitoredServer(this.defaultServerURI).setAsDefaultServer(
            false);
        }
        this.getMonitoredServer(serverURI).setAsDefaultServer(true);
      }

      this.defaultServerURI = serverURI;
      printInfo("Servidor default = " + serverURI);
    }
    finally {
      this.defaultServerLock.unlock();
    }
  }

  /**
   * Exibe uma mensagem informativa (se {@link #verbose} = <code>true</code>).
   * 
   * @param msg mensagem
   */
  private void printInfo(String msg) {
    if (verbose) {
      System.out.println(msg);
    }
  }

  /**
   * @return A URI do servidor default
   */
  protected final ServerURI getDefaultURI() {
    try {
      this.defaultServerLock.lock();
      return this.defaultServerURI;
    }
    finally {
      this.defaultServerLock.unlock();
    }
  }

  /**
   * @param serverURI A URI que identifica o servidor
   * @return true se o servidor estiver sendo monitorado, false caso contrrio
   */
  protected final boolean isMonitored(ServerURI serverURI) {
    synchronized (this.servers) {
      return this.servers.containsKey(serverURI);
    }
  }

  /**
   * Remove um servidor da monitorao.
   * 
   * @param serverURI A URI que identifica o servidor
   */
  protected void removeServer(ServerURI serverURI) {
    synchronized (this.servers) {
      this.getMonitoredServer(serverURI).flush();
      this.servers.remove(serverURI);
    }
  }

  /**
   * Adiciona um novo servidor para ser gerenciado. Os ouvintes comuns so
   * adicionados mas a monitorao no  iniciada neste momento. Se j houver
   * uma monitorao a mesma ser perdida e no ser possvel encerr-la, sendo
   * assim  importante que seja removida atravs de um logout antes da adio
   * da nova.
   * 
   * @param monitoredServer A Instncia do {@link MonitoredServer} do servidor a
   *        ser gerenciado
   */
  protected void addServer(MonitoredServer monitoredServer) {
    ServerURI uri = monitoredServer.getURI();
    synchronized (this.servers) {
      this.servers.put(uri, monitoredServer);
      if (this.servers.keySet().size() == 1 && this.getDefaultURI() == null) {
        this.setDefaultServer(uri);
      }
      this.addCommonListeners(uri);
    }
  }

  /**
   * Obtm o conjunto de URI's que identifica os servidores sendo gerenciados
   * 
   * @return Um conjunto de {@link ServerURI}
   */
  protected Set<ServerURI> getManagedServers() {
    synchronized (this.servers) {
      return new HashSet<ServerURI>(this.servers.keySet());
    }
  }

  /**
   * Submete os ouvintes que so comuns a todos os servidores que esto
   * monitorados.
   * 
   */
  private final void submitCommonListeners() {
    synchronized (this.commonListeners) {
      for (MonitoredServerListener l : this.commonListeners) {
        for (MonitoredServer ms : this.servers.values()) {
          ms.addListener(l);
        }
      }
    }
  }

  /**
   * Adiciona o ouvinte comum ao servidor especificado
   * 
   * @param serverURI A URI do servidor que receber o ouvinte.
   */
  private final void addCommonListeners(ServerURI serverURI) {
    MonitoredServer ms = this.getMonitoredServer(serverURI);
    synchronized (this.commonListeners) {
      for (MonitoredServerListener l : this.commonListeners) {
        ms.addListener(l);
      }
    }
  }

  /**
   * Adiciona ouvintes a todos os servidores gerenciados.
   * 
   * @param l A instncia do ouvinte.
   */
  protected final void addCommonListener(MonitoredServerListener l) {
    synchronized (this.commonListeners) {
      if (!this.commonListeners.contains(l)) {
        this.commonListeners.add(l);
        this.submitCommonListeners();
      }
    }
  }

  /**
   * Remove o ouvinte de todos o servidores gerenciados.
   * 
   * @param l A instncia do ouvinte
   */
  protected final void deleteCommonListener(MonitoredServerListener l) {
    synchronized (this.commonListeners) {
      if (!this.commonListeners.contains(l)) {
        return;
      }
      for (MonitoredServer ms : this.servers.values()) {
        ms.deleteListener(l);
      }
      this.commonListeners.remove(l);
    }
  }

  /**
   * Finaliza
   */
  protected void shutdown() {
    synchronized (this.servers) {
      for (MonitoredServer data : this.servers.values()) {
        data.flush();
      }
      this.servers.clear();
    }
  }

  /**
   * Define se mensagens informativas devem ser exibidas.
   * 
   * @param verbose <code>true</code> se informaes devem ser exibidas,
   *        <code>false</code> se apenas erros e alertas devem ser exibidos
   */
  public void setVerbose(boolean verbose) {
    this.verbose = verbose;
  }
}
