/*
 * $Author$ $Date$ $Release:$
 */
package csbase.logic.algorithms;

import java.io.Serializable;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import csbase.exception.BugException;
import csbase.exception.ParseException;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.SGAInfo;
import csbase.logic.algorithms.parameters.AbstractFileParameter;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.logic.algorithms.parameters.InputFileParameter;
import csbase.logic.algorithms.parameters.InputURLParameter;
import csbase.logic.algorithms.parameters.OutputFileParameter;
import csbase.logic.algorithms.parameters.OutputURLParameter;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.Validation;
import csbase.remote.ClientRemoteLocator;

/**
 * <p>
 * O modelo de um configurador de algoritmos.
 * </p>
 *
 * <p>
 * O configurador de algoritmos  responsvel por implementar praticamente todas
 * as responsabilidades relacionadas  montagem da linha de comando utilizada
 * para executar algoritmos em um {@link SGAInfo}. Entre essas responsabilidades
 * pode-se destacar:
 * </p>
 * <ul>
 * <li>Importar e exportar parmetros com {@link #importValues(java.util.Map)} e
 * {@link #exportValues()});</li>
 * <li>Receber os valores dos parmetros fornecidos pelo usurio (mtodo:
 * {@link #setParameterValuesByName(Map)});</li>
 * <li>Montar a linha de comando (mtodo:
 * {@link #makeCommandLine(CommandLineContext)});</li>
 * </ul>
 * 
 * <p>
 * Alguns mtodos podem gerar eventos observveis por instncias da classe
 * {@link AlgorithmConfiguratorListener}.
 * </p>
 *
 * @author lmoreira
 */
public abstract class AlgorithmConfigurator implements Serializable {

  /**
   * Enumera os tipos de configuradores disponveis
   */
  public enum ConfiguratorType {
    /**
     * Configurador para um comando simples.
     */
    SIMPLE,
    /**
     * Configurador para um fluxo de comandos.
     */
    FLOW,
  }

  /** Caractere separador de lista de arquivos na linha de comando. */
  public static final String FILE_LIST_SEPARATOR = ",";
  /** Esta string substitui o acesso da classe da logic ao language */
  private static final String UPDATE_ALGOVERSION_ERROR =
    "Erro ao atualizar verso do algoritmo";
  /**
   * Nome da propriedade que indica a quantidade de CPU necessria para execuo
   * do algoritmo.
   */
  private static final String CPU_AMOUNT = "cpu";

  /**
   * Quantidade de CPU quando no for possvel recuperar do algoritmo.
   */
  private static final float DEFAULT_CPU_AMOUNT = 0.0f;

  /**
   * Nome da propriedade que indica a quantidade de memria necessria em MB
   * para execuo do algoritmo.
   */
  private static final String MEMORY_AMOUNT = "memory";

  /**
   * Total de memria quando no for possvel recuperar do algoritmo.
   */
  private static final float DEFAULT_MEMORY_AMOUNT = 0.0f;

  /**
   * Nome da propriedade que indica os requisitos necessrios para a execuo do
   * algoritmo.
   */
  private static final String REQUIREMENTS = "requirements";

  /**
   * Requisitos quando no for possvel recuperar do algoritmo.
   */
  private static final String DEFAULT_REQUIREMENTS = "";

  /**
   * Nome da propriedade que indica qual o separador dos requisitos necessrios
   * para a execuo do algoritmo.
   */
  private static final String REQUIREMENTS_SEPARATOR = "requirements_separator";

  /**
   * Separador dos requisitos quando no for possvel recuperar do algoritmo.
   */
  private static final String DEFAULT_REQUIREMENTS_SEPARATOR = ",";

  /**
   * Tipo do configurador.
   */
  private final ConfiguratorType configuratorType;

  /**
   * Abreviatura do nome do algoritmo.
   */
  private String abbreviation;

  /**
   * A verso do algoritmo.
   */
  private AlgorithmVersionInfo algorithmVersion;

  /**
   * A descrio do configurador de algoritmos.
   */
  private String algorithmDescription;

  /**
   * A descrio do comando a ser executado. Valor aparecer como default.
   */
  private String commandDescription;

  /**
   * O tipo de execuo utilizado.
   */
  private ExecutionType executionType;

  /**
   * Local de execuo do algoritmo. Pode ser
   * {@link ExecutionLocation#BINARY_DIR},
   * {@link ExecutionLocation#PERSISTENCY_DIR} ou
   * {@link ExecutionLocation#SANDBOX}.
   */
  private ExecutionLocation executionLocation;

  /**
   * O caminho para o arquivo. Este atributo  utilizado para permitir a criao
   * de um configurador de algoritmos que no esteja na rvore de algoritmos.
   */
  private String filePath;

  /**
   * O arquivo de sada (stdout) de um algoritmo sendo executado.
   */
  private FileURLValue standardOutputFile;

  /**
   * O arquivo que armazena os alertas de execuo do algoritmo.
   */
  private FileURLValue warningsFile;

  /**
   * O arquivo que armazena o cdigo de sada do algoritmo.
   */
  private FileURLValue exitCodeLogFile;

  /**
   * Indica que o algoritmo prov um cdigo de sada durante sua execuo.
   */
  private boolean hasExitCode;

  /**
   * O observadores deste algoritmo.
   */
  private List<AlgorithmConfiguratorListener> listeners;

  /**
   * Sinal que indica se  necessrio fornecer o identificador do comando na
   * linha de comando.
   */
  private boolean provideId;

  /**
   * <p>
   * Cria um configurador de algoritmos.
   * </p>
   *
   * @param algorithmVersion A verso do algoritmo (No aceita {@code null}).
   * @param description A descrio do configurador de algoritmos (Aceita
   *        {@code null}).
   * @param executionType O tipo de execuo (No aceita {@code null}).
   * @param executionLocation Local de execuo do algoritmo.
   * @param abbreviation A abreviatura (Aceita {@code null}).
   * @param provideId Sinal que indica se  necessrio fornecer o identificador
   *        do comando na linha de comando.
   * @param commandLinePattern pattern para linha de comando.
   */
  protected AlgorithmConfigurator(AlgorithmVersionInfo algorithmVersion,
    String description, ExecutionType executionType,
    ExecutionLocation executionLocation, String abbreviation, boolean provideId,
    final String commandLinePattern) {
    this(getType(algorithmVersion), description, executionType,
      executionLocation, abbreviation, provideId, commandLinePattern);
    setAlgorithmVersion(algorithmVersion);
  }

  /**
   * Obtm o tipo da verso do algoritmo.
   * 
   * @param algorithmVersion a verso.
   * @return o tipo da verso.
   */
  private static ConfiguratorType getType(
    AlgorithmVersionInfo algorithmVersion) {
    if (algorithmVersion == null) {
      return null;
    }
    return algorithmVersion.getType();
  }

  /**
   * Pattern
   */
  final private String commandLinePattern;

  /**
   * <p>
   * Cria um configurador de algoritmos.
   * </p>
   *
   * @param type Tipo de configurador (@see {@link ConfiguratorType})
   * @param description A descrio do configurador de algoritmos (Aceita
   *        {@code null}).
   * @param executionType O tipo de execuo (No aceita {@code null}).
   * @param executionLocation Local de execuo do algoritmo.
   * @param abbreviation A abreviatura (Aceita {@code null}).
   * @param provideId Sinal que indica se  necessrio fornecer o identificador
   *        do comando na linha de comando.
   * @param commandLinePattern pattern para linha de comando.
   */
  protected AlgorithmConfigurator(ConfiguratorType type, String description,
    ExecutionType executionType, ExecutionLocation executionLocation,
    String abbreviation, boolean provideId, final String commandLinePattern) {
    this.listeners = new LinkedList<AlgorithmConfiguratorListener>();
    this.configuratorType = type;
    this.algorithmDescription = description;
    this.abbreviation = abbreviation;
    this.provideId = provideId;
    this.hasExitCode = false;
    this.commandLinePattern = commandLinePattern;
    setExecutionType(executionType);
    setExecutionLocation(executionLocation);
  }

  /**
   * Retorna
   *
   * @return configuratorType
   */
  public ConfiguratorType getConfiguratorType() {
    return configuratorType;
  }

  /**
   * <p>
   * Cria um configurador de algoritmos.
   * </p>
   *
   * @param type Tipo de configurador (@see {@link ConfiguratorType})
   * @param filePath O caminho para o arquivo que descreve este configurador
   *        (No aceita {@code null}).
   * @param description A descrio do configurador de algoritmos (Aceita
   *        {@code null}).
   * @param executionType O tipo de execuo (No aceita {@code null}).
   * @param executionLocation Local de execuo do algoritmo.
   * @param abbreviation A abreviatura (Aceita {@code null}).
   * @param provideId Sinal que indica se  necessrio fornecer o identificador
   *        do comando na linha de comando.
   * @param commandLinePattern pattern para linha de comando.
   */
  protected AlgorithmConfigurator(ConfiguratorType type, String filePath,
    String description, ExecutionType executionType,
    ExecutionLocation executionLocation, String abbreviation, boolean provideId,
    final String commandLinePattern) {
    this(type, description, executionType, executionLocation, abbreviation,
      provideId, commandLinePattern);
    setFilePath(filePath);
  }

  /**
   * <p>
   * Obtm todos os arquivos de log.
   * </p>
   *
   * <p>
   * O conjunto retornado  imutvel (Collections.unmodifiableSet(Set)).
   * </p>
   *
   * @return o conjunto de arquivos de log (se no existir arquivo de log,
   *         retornar um conjunto vazio).
   */
  public Set<FileURLValue> getLogFiles() {
    Set<FileURLValue> logFiles = new HashSet<FileURLValue>();
    FileURLValue theLogFile = getLogFile();
    if (null != theLogFile) {
      logFiles.add(theLogFile);
    }
    return Collections.unmodifiableSet(logFiles);
  }

  /**
   * Adiciona um observador de configurador de algoritmos a esta instncia.
   *
   * @param listener O observador (No aceita {@code null}).
   */
  public final void addAlgorithmConfiguratorListener(
    AlgorithmConfiguratorListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    this.listeners.add(listener);
  }

  /**
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public final boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!getClass().equals(obj.getClass())) {
      return false;
    }
    AlgorithmConfigurator configurator = (AlgorithmConfigurator) obj;
    if (getAlgorithmVersion() == null) {
      return getFilePath().equals(configurator.getFilePath());
    }
    return getAlgorithmVersion().equals(configurator.getAlgorithmVersion());
  }

  /**
   * <p>
   * Exporta os valores dos parmetros para um {@link Map} que contm apenas
   * objetos que so tipos envelopes para tipos primitivos ou colees padro do
   * API do Java.
   * </p>
   *
   * <p>
   * Este mtodo cria um {@link Map} que pode ser serializado para um arquivo.
   * </p>
   *
   * <p>
   * O mapa retornado neste mtodo pode ser utilizado pelo mtodo
   * {@link #importValues(Map)}.
   * </p>
   *
   * @return O {@link Map} ou uma mapa vazio se no houver valores ou
   *         parmetros.
   */
  public abstract Map<Object, Object> exportValues();

  /**
   * Obtm a abreviatura do algoritmo.
   *
   * @return A abreviatura do algoritmo ou {@code null} se ela no existir.
   */
  public final String getAbbreviation() {
    return this.abbreviation;
  }

  /**
   * Obtm o identificador do algoritmo.
   *
   * @return o identificador do algoritmo ou {@code null} se este configurador no
   * estiver associado a um algoritmo.
   */
  public final String getAlgorithmId() {
    if (this.algorithmVersion != null) {
      return this.algorithmVersion.getInfo().getId();
    }
    return null;
  }

  /**
   * Obtm o nome do algoritmo.
   *
   * @return O nome do algoritmo ou {@code null} se este configurador no
   *         estiver associado a um algoritmo.
   */
  public final String getAlgorithmName() {
    if (this.algorithmVersion != null) {
      return this.algorithmVersion.getInfo().getName();
    }
    return null;
  }

  /**
   * Obtm a verso do algoritmo deste configurador.
   *
   * @return A verso do algoritmo ou {@code null} se este configurador no
   *         estiver uma verso.
   */
  public final AlgorithmVersionInfo getAlgorithmVersion() {
    return this.algorithmVersion;
  }

  /**
   * Obtm o identificador da verso do algoritmo.
   *
   * @return O identificador da verso do algoritmo ou {@code null} se este
   *         configurador no estiver associado a um algoritmo.
   */
  public final AlgorithmVersionId getAlgorithmVersionId() {
    if (this.algorithmVersion != null) {
      return this.algorithmVersion.getId();
    }
    return null;
  }

  /**
   * Obtm uma relao de caminhos para os diretrios dos binrios deste
   * algoritmo para um determinado {@link SGAInfo}.
   *
   * @param platformName O nome da plataforma (No aceita {@code null}).
   * @param fileSeparator O separador de arquivo (No aceita {@code null}).
   *
   *        <p>
   *        Obs.: o conjunto de caminhos retornvel deve ser imutvel (veja
   *        {@link Collections#unmodifiableSet(Set)}).
   *        </p>
   *
   * @return O conjunto dos caminhos.
   */
  public abstract Set<String> getBinaryDirectories(String platformName,
    char fileSeparator);

  /**
   * Obtm uma relao de caminhos em forma de array para os diretrios dos
   * binrios deste algoritmo para um determinado {@link SGAInfo}.
   *
   * @param platformName O nome da plataforma (No aceita {@code null}).
   *
   *        <p>
   *        Obs.: o conjunto de caminhos retornvel deve ser imutvel (veja
   *        {@link Collections#unmodifiableSet(Set)}).
   *        </p>
   *
   * @return O conjunto dos caminhos em forma de array.
   */
  public abstract Set<String[]> getBinaryDirectoriesAsArray(
    String platformName);

  /**
   * Obtm a quantidade de CPU solicitada pelo algoritmo.
   *
   * @return .
   */
  public final float getCpuAmount() {
    return getFloatAlgorithmProperty(CPU_AMOUNT, DEFAULT_CPU_AMOUNT);
  }

  /**
   * Obtm o descritor do arquivo de log.
   *
   * @return O descritor do arquivo ou {@code null} se ele no existir.
   */
  public final FileURLValue getLogFile() {
    OutputFileParameter logFileParameter = getLogFileParameter();
    if (logFileParameter == null) {
      return null;
    }
    return logFileParameter.getValue();
  }

  /**
   * Obtm o parmetro de log.
   *
   * @return O parmetro de log ({@code null} se ele no existir).
   */
  public final OutputFileParameter getLogFileParameter() {
    for (OutputFileParameter parameter : getOutputFileParameters()) {
      if (parameter.isLogFile()) {
        return parameter;
      }
    }
    return null;
  }

  /**
   * Obtm a quantidade de memria solicitada pelo algoritmo.
   *
   * @return .
   */
  public final float getMemoryAmount() {
    return getFloatAlgorithmProperty(MEMORY_AMOUNT, DEFAULT_MEMORY_AMOUNT);
  }

  /**
   * @return o campo commandLinePattern
   */
  public final String getCommandLinePattern() {
    return commandLinePattern;
  }

  /**
   * Obtm os requisitos necessrios ao algoritmo.
   *
   * @return .
   */
  public Set<String> getRequirements() {
    Set<String> requirementsSet = new HashSet<String>();
    String requirements = getStringAlgorithmProperty(REQUIREMENTS,
      DEFAULT_REQUIREMENTS);
    String sep = getStringAlgorithmProperty(REQUIREMENTS_SEPARATOR,
      DEFAULT_REQUIREMENTS_SEPARATOR);
    for (String requirement : requirements.split(sep)) {
      String req = requirement.trim();
      if (req.length() > 0) {
        requirementsSet.add(req);
      }
    }
    return requirementsSet;
  }

  /**
   * Obtm a descrio deste configurador.
   *
   * @return A descrio ou {@code null} caso no hava descrio.
   */
  public final String getDescription() {
    return this.algorithmDescription;
  }

  /**
   * <p>
   * Obtm o conjunto de diretrios de sada.
   * </p>
   *
   * <p>
   * Obs.: o conjunto  imutvel (veja {@link Collections#unmodifiableSet(Set)}
   * ).
   * </p>
   *
   * @return O conjunto ou um conjunto vazio caso no haja diretrios como valor
   *         de nenhum dos parmetros.
   */
  public abstract Set<FileURLValue> getOutputDirectories();

  /**
   * <p>
   * Obtm o conjunto de diretrios de entrada.
   * </p>
   *
   * <p>
   * Obs.: o conjunto  imutvel (veja {@link Collections#unmodifiableSet(Set)}
   * ).
   * </p>
   *
   * @return O conjunto ou um conjunto vazio caso no haja diretrios como valor
   *         de nenhum dos parmetros.
   */
  public abstract Set<FileURLValue> getInputDirectories();

  /**
   * Obtm o tipo de execuo.
   *
   * @return O tipo de execuo.
   */
  public final ExecutionType getExecutionType() {
    return this.executionType;
  }

  /**
   * Obtm o local de execuo do algoritmo.
   *
   * @return O local de execuo do algoritmo.
   */
  public final ExecutionLocation getExecutionLocation() {
    return this.executionLocation;
  }

  /**
   * <p>
   * Obtm a relao com os parmetros que aceitam arquivos e/ou diretrios.
   * </p>
   *
   * <p>
   * A relao  ordenada segundo o critrio natural de ordenao de um
   * parmetro ({@link csbase.logic.algorithms.parameters.Parameter}).
   * </p>
   *
   * <p>
   * Obs.: a relao  imutvel (veja
   * {@link Collections#unmodifiableSortedSet(SortedSet)}).
   * </p>
   *
   * @return A relao de parmetros ou uma relao vazia caso no haja
   *         parmetros que aceitem arquivos.
   */
  public final SortedSet<AbstractFileParameter> getFileParameters() {
    SortedSet<AbstractFileParameter> fileParameters =
      new TreeSet<AbstractFileParameter>();
    /* File */
    List<InputFileParameter> inputFileParameters = getInputFileParameters();
    for (InputFileParameter inputFileParameter : inputFileParameters) {
      fileParameters.add(inputFileParameter);
    }
    List<OutputFileParameter> outputFileParameters = getOutputFileParameters();
    for (OutputFileParameter outputFileParameter : outputFileParameters) {
      fileParameters.add(outputFileParameter);
    }
    /* URL */
    List<InputURLParameter> inputURLParameters = getInputURLParameters();
    for (InputURLParameter inputURLParameter : inputURLParameters) {
      fileParameters.add(inputURLParameter);
    }
    List<OutputURLParameter> outputURLParameters = getOutputURLParameters();
    for (OutputURLParameter outputURLParameter : outputURLParameters) {
      fileParameters.add(outputURLParameter);
    }
    return Collections.unmodifiableSortedSet(fileParameters);
  }

  /**
   * Obtm o caminho para o arquivo que descreve este configurador.
   *
   * @return O caminho para o arquivo ou {@code null} caso no haja um descritor
   *         disponvel.
   */
  public final String getFilePath() {
    return this.filePath;
  }

  /**
   * <p>
   * Obtm a lista com os parmetros que aceitam arquivos e/ou diretrios para
   * entrada de dados.
   * </p>
   *
   * <p>
   * Obs.: a relao  imutvel (veja {@link Collections#unmodifiableList(List)}
   * ).
   * </p>
   *
   * @return A relao de parmetros ou uma relao vazia caso no haja
   *         parmetros que aceitem arquivos.
   */
  public abstract List<InputFileParameter> getInputFileParameters();

  /**
   * <p>
   * Obtm o conjunto de arquivos de entrada.
   * </p>
   *
   * <p>
   * Obs.: o conjunto  imutvel (veja {@link Collections#unmodifiableSet(Set)}
   * ).
   * </p>
   *
   * @return O conjunto ou um conjunto vazio caso no haja arquivos de entrada
   *         como valor de nenhum dos parmetros.
   */
  public abstract Set<FileURLValue> getInputFiles();

  /**
   * <p>
   * Obtm a lista com os parmetros que aceitam arquivos e/ou diretrios para
   * sada de dados.
   * </p>
   *
   * <p>
   * Obs.: a relao  imutvel (veja {@link Collections#unmodifiableList(List)}
   * ).
   * </p>
   *
   * @return A relao de parmetros ou uma relao vazia caso no haja
   *         parmetros que aceitem arquivos.
   */
  public abstract List<OutputFileParameter> getOutputFileParameters();

  /**
   * <p>
   * Obtm o conjunto de arquivos de sada.
   * </p>
   *
   * <p>
   * Obs.: o conjunto  imutvel (veja {@link Collections#unmodifiableSet(Set)}
   * ).
   * </p>
   *
   * @return O conjunto ou um conjunto vazio caso no haja arquivos de sada
   *         como valor de nenhum dos parmetros.
   */
  public abstract Set<FileURLValue> getOutputFiles();

  /**
   * <p>
   * Obtm a lista com os parmetros do tipo URL que aceitam arquivos e/ou
   * diretrios para entrada de dados.
   * </p>
   *
   * <p>
   * Obs.: a relao  imutvel (veja {@link Collections#unmodifiableList(List)}
   * ).
   * </p>
   *
   * @return A relao de parmetros ou uma relao vazia caso no haja
   *         parmetros que aceitem arquivos.
   */
  public abstract List<InputURLParameter> getInputURLParameters();

  /**
   * <p>
   * Obtm o conjunto de arquivos (URLs) de entrada.
   * </p>
   *
   * <p>
   * Obs.: o conjunto  imutvel (veja {@link Collections#unmodifiableSet(Set)}
   * ).
   * </p>
   *
   * @return O conjunto ou um conjunto vazio caso no haja arquivos de entrada
   *         como valor de nenhum dos parmetros.
   */
  public abstract Set<FileURLValue> getInputURLs();

  /**
   * <p>
   * Obtm a lista com os parmetros que aceitam arquivos e/ou diretrios (URLs)
   * para sada de dados.
   * </p>
   *
   * <p>
   * Obs.: a relao  imutvel (veja {@link Collections#unmodifiableList(List)}
   * ).
   * </p>
   *
   * @return A relao de parmetros ou uma relao vazia caso no haja
   *         parmetros que aceitem arquivos.
   */
  public abstract List<OutputURLParameter> getOutputURLParameters();

  /**
   * <p>
   * Obtm o conjunto de arquivos (URLs) de sada.
   * </p>
   *
   * <p>
   * Obs.: o conjunto  imutvel (veja {@link Collections#unmodifiableSet(Set)}
   * ).
   * </p>
   *
   * @return O conjunto ou um conjunto vazio caso no haja arquivos de sada
   *         como valor de nenhum dos parmetros.
   */
  public abstract Set<FileURLValue> getOutputURLs();

  /**
   * Obtm o rtulo de um parmetro dado o seu nome.
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   *
   * @return O rtulo.
   *
   * @throws ParameterNotFoundException Se no existir um parmetro com o nome
   *         fornecido.
   */
  public abstract String getParameterLabel(String parameterName)
    throws ParameterNotFoundException;

  /**
   * <p>
   * Obtm a relao de nomes dos parmetros.
   * </p>
   *
   * <p>
   * A relao  imutvel (veja {@link Collections#unmodifiableSet(Set)}).
   * </p>
   *
   * @return A relao de nomes ou uma relao vazia caso no haja parmetros.
   */
  public abstract Set<String> getParameterNames();

  /**
   * Obtm o tipo de um parmetro dado o seu nome.
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   *
   * @return O tipo.
   *
   * @throws ParameterNotFoundException Se no existir um parmetro com o nome
   *         fornecido.
   */
  public abstract String getParameterType(String parameterName)
    throws ParameterNotFoundException;

  /**
   * Obtm o valor de um parmetro dado o seu nome.
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   *
   * @return O valor ou {@code null} caso no haja um valor para o parmetro.
   *
   * @throws ParameterNotFoundException Se no existir um parmetro com o nome
   *         fornecido.
   */
  public abstract String getParameterValue(String parameterName)
    throws ParameterNotFoundException;

  /**
   * Retorna um mapa com os valores dos parmetros indexado pelos nomes.
   *
   * @return um mapa com os valores dos parmetros indexado pelos nomes
   */
  public Map<String, String> getParameterValuesByName() {
    Map<String, String> parameterValues = new HashMap<String, String>();
    for (String parameterName : getParameterNames()) {
      try {
        parameterValues.put(parameterName, getParameterValue(parameterName));
      }
      catch (ParameterNotFoundException e) {
        throw new BugException(e);
      }
    }
    return parameterValues;
  }

  /**
   * <p>
   * Obtm uma relao de todas as plataformas suportadas por este configurador
   * de algoritmos.
   * </p>
   *
   * <p>
   * A relao  imutvel (veja {@link Collections#unmodifiableSet(Set)}).
   *
   * @return A relao de plataformas.
   */
  public abstract Set<String> getPlatforms();

  /**
   * @see java.lang.Object#hashCode()
   */
  @Override
  public final int hashCode() {
    if (getAlgorithmVersion() == null) {
      return getFilePath().hashCode();
    }
    return getAlgorithmVersion().hashCode();
  }

  /**
   * <p>
   * Importa os valores dos parmetros de um {@link Map}.
   * </p>
   *
   * @param values O mapa de valores de parmetros (No aceita {@code null}).
   *
   * @throws ParseException Se os valores do mapa no puderem ser convertidos
   *         para os valores dos parmetros.
   */
  public abstract void importValues(Map<Object, Object> values)
    throws ParseException;

  /**
   * Indica se o configurador est habilitado.
   *
   * @return {@code true} se estiver habilitado ou {@code false} caso contrrio.
   */
  public abstract boolean isEnabled();

  /**
   * Indica se este configurador utiliza o tipo de execuo
   * {@link ExecutionType#MULTIPLE}.
   *
   * @return {@code true} se utiliza ou {@code false} caso contrrio.
   */
  public final boolean isMultipleExecution() {
    return getExecutionType().equals(ExecutionType.MULTIPLE);
  }

  /**
   * Verifica se os valores deste configurador so iguais aos valores-padro.
   *
   * @return {@code true} se todos os valores so iguais aos valores-padro ou
   *         {@code false} caso contrrio.
   */
  public abstract boolean isSetDefaultValues();

  /**
   * Indica se este configurador utiliza o tipo de execuo
   * {@link ExecutionType#SIMPLE}.
   *
   * @return {@code true} se utiliza ou {@code false} caso contrrio.
   */
  public final boolean isSimpleExecution() {
    return getExecutionType().equals(ExecutionType.SIMPLE);
  }

  /**
   * Retorna a linha de comando necessria para fazer a captura do cdigo de
   * sada de execuo do algoritmo. Esse mtodo deve ser chamado pelos
   * configuradores filhos ao final de makeCommandLine(CommandLineContext
   * context), caso queiram capturar o cdigo de sada do algoritmo.
   *
   * @see SimpleAlgorithmConfigurator (para exemplo de uso)
   *
   * @param context O contexto de execuo do comando
   * @return Uma string com a linha de comando que deve ser concatenada ao
   *         comando, caso se queria capturar o cdigo de sada de sua execuo.
   */
  protected String getExitCodeCaptureCommand(CommandLineContext context) {
    if (hasExitCode()) {
      // Arquivo onde ser armazenado o cdigo de sada do comando
      FileURLValue exitCodeLog = getExitCodeLogFile();
      if (exitCodeLog != null) {
        String exitCodeFilePath = exitCodeLog.getPath();
        CommandLineBuilder builder = new CommandLineBuilder(context.isScript());
        CommandLineBuilder command = builder.appendExitCodeCaptureCommand(
          exitCodeFilePath, context);
        return command.toString();
      }
    }
    return "";
  }

  /**
   * Cria a linha de comando.
   *
   * @param context contexto para criao da linha de comando.
   *
   * @return A linha de comando compatvel com o shell <b>Korn Shell (ksh)</b>.
   */
  public abstract String makeCommandLine(CommandLineContext context);

  /**
   * Cria a linha de comando e a salva em um arquivo de script.
   *
   * @param context contexto para criao da linha de comando.
   *
   * @return A linha de comando compatvel com o shell <b>Korn Shell (ksh)</b>
   *         para a execuo <b>do script</b>.
   */
  public CommandScript[] makeCommandLineAsScript(CommandLineContext context) {
    String scriptContent = makeCommandLine(context);
    CommandScript commandScript = new CommandScript(context, scriptContent);
    return new CommandScript[] { commandScript };
  }

  /**
   * Indica se ir fornecer o identificador do comando na linha de comando.
   *
   * @return {@code true} se ir fornecer ou {@code false} caso contrrio.
   */
  public final boolean provideId() {
    return this.provideId;
  }

  /**
   * <p>
   * Restaura os valores dos parmetros para os valores-padro.
   * </p>
   */
  public abstract void resetValues();

  /**
   * <p>
   * Atribui um arquivo para ser utilizado com o arquivo de entrada padro do
   * configurador.
   * </p>
   *
   * <p>
   * O arquivo de entrada padro  o arquivo que  atribuido a um dos parmetros
   * do tipo arquivo de entrada (veja {@link InputFileParameter}) tipicamente
   * uma operao de duplo-clique na rvore de projetos.
   * </p>
   *
   * <p>
   * Por exemplo: em um algoritmo para descomprimir dados, o arquivo de entrada
   * padro poderia ser o arquivo a ser expandido.
   * </p>
   *
   * @param inputFile O arquivo de entrada (Aceita {@code null}).
   *
   * @return {@code true} se a operao foi realizada com sucesso ou
   *         {@code false} se o configurador no aceita arquivo de entrada
   *         padro ou este tipo de arquivo de entrada padro.
   */
  public abstract boolean setDefaultInputFile(FileURLValue inputFile);

  /**
   * <p>
   * Modifica o estado habilitado/desabilitado do configurador de algoritmos.
   * </p>
   *
   * <p>
   * Dispara o evento
   * {@link AlgorithmConfiguratorListener#wasSetEnabled(AlgorithmConfigurator)}.
   * </p>
   *
   * @param isEnabled Indica se o configurador se tornar habilitado ou
   *        desabilitado.
   *
   * @return {@code true} em caso de sucesso ou {@code false} se no houver
   *         mudana de estado.
   */
  public abstract boolean setEnabled(boolean isEnabled);

  /**
   * Exibe ou oculta o parmetro log.
   *
   * @param isVisible {@code true} para exibir ou {@code false} para ocultar.
   *
   * @return {@code false} se no houver parmetro log ou se foi solicitado
   *         "exibir" e o parmetro est visvel ou se foi solicitado "ocultar"
   *         e o parmetro est oculto.
   */
  public final boolean setLogFileParameterVisible(boolean isVisible) {
    OutputFileParameter logFileParameter = getLogFileParameter();
    if (logFileParameter == null) {
      return false;
    }
    if (logFileParameter.isVisible() == isVisible) {
      return false;
    }
    logFileParameter.setVisible(isVisible);
    return true;
  }

  /**
   * Atribui o valor a um parmetro.
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   * @param parameterValue O valor do parmetro (Aceita {@code null})
   *
   * @throws ParseException Se pelo menos um dos valores no estiver em um
   *         formato que no pode ser aceito pelo seu parmetro.
   *
   * @throws ParameterNotFoundException Caso no exista um parmetro com o nome
   *         fornecido.
   */
  public abstract void setParameterValue(String parameterName,
    String parameterValue) throws ParseException, ParameterNotFoundException;

  /**
   * <p>
   * Atribui valores a este configurador de algoritmos.
   * </p>
   *
   * <p>
   * O mapa de parmetros possui pares cujas as chaves so os nomes dos
   * parmetros e os valores so valores dos parmetros. Caso no haja um
   * parmetro cujo nome no  chave do mapa o seu valor dever ser considerado
   * {@code null}.
   * </p>
   *
   * @param parameterValuesByName O mapa de valores.
   *
   * @throws ParseException Se pelo menos um dos valores no estiver em um
   *         formato que no pode ser aceito pelo seu parmetro.
   * @throws ParameterNotFoundException Caso no exista um parmetro com o nome
   *         fornecido.
   */
  public void setParameterValuesByName(
    Map<String, String> parameterValuesByName) throws ParseException,
    ParameterNotFoundException {
    List<String> missingParameters = new ArrayList<String>();
    for (String parameterName : parameterValuesByName.keySet()) {
      String parameterValue = parameterValuesByName.get(parameterName);
      try {
        setParameterValue(parameterName, parameterValue);
      }
      catch (ParameterNotFoundException e) {
        /*
         * Guarda o nome dos parmetros para relanar a exceo no final,
         * deixando a atribuio dos demais parmetros prosseguir.
         */
        Collections.addAll(missingParameters, e.getParameterNames());
      }
    }
    if (!missingParameters.isEmpty()) {
      throw new ParameterNotFoundException(missingParameters.toArray(
        new String[missingParameters.size()]));
    }
  }

  /**
   * Atualiza a verso deste configurador, que pode ter sido alterada pelo
   * administrador.
   *
   * @throws Exception exceo
   */
  public void updateAlgorithmVersion() throws Exception {
    /* No fluxo annimo, a verso pode ser null */
    if (getAlgorithmVersion() == null) {
      return;
    }
    AlgorithmVersionId versionId = getAlgorithmVersion().getId();
    if (versionId != null) {
      String algorithmName = getAlgorithmVersion().getInfo().getName();
      try {
        AlgorithmInfo algoInfo = ClientRemoteLocator.algorithmService.getInfo(
          algorithmName);
        AlgorithmVersionInfo versionInfo = algoInfo.getVersionInfo(versionId);
        setAlgorithmVersion(versionInfo);
      }
      catch (RemoteException re) {
        throw new Exception(UPDATE_ALGOVERSION_ERROR, re);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    if (getAlgorithmVersion() == null) {
      return "";
    }
    String algorithmName = getAlgorithmVersion().getInfo().getName();
    AlgorithmVersionId algorithmVersionId = getAlgorithmVersion().getId();
    return MessageFormat.format("{0} ({1})", algorithmName, algorithmVersionId);
  }

  /**
   * Atribui o valor do arquivo que deve receber a sada padro da execuo de
   * um algoritmo representado por este configurador.
   *
   * @param standardOutputFile arquivo de sada.
   */
  public void setStandardOutputFile(FileURLValue standardOutputFile) {
    this.standardOutputFile = standardOutputFile;
  }

  /**
   * Obtm a sada padro que ser utilizada durante a execuo de um algortmo
   * representado por este configurador. Se este configurador for uma composio
   * de outros, deve-se utilizar o mtodo {@link #getStandardOutputFiles()} para
   * obter a sada padro dos algoritmos que constituem este.
   *
   * @return a sada padro que ser utilizada durante a execuo de um
   *         algortmo representado por este configurador.
   */
  public FileURLValue getStandardOutputFile() {
    return this.standardOutputFile;
  }

  /**
   * Obtm um conjunto imutvel contendo os arquivos de sada padro. O tamanho
   * do conjuno retornado ser maior que um se este algoritmo  uma composio
   * de outros. Como por exemplo, se ele for um fluxo.
   *
   * @return um conjunto imutvel contendo os arquivos de sada padro.
   */
  public Set<FileURLValue> getStandardOutputFiles() {
    if (null == this.standardOutputFile) {
      return Collections.emptySet();
    }
    return Collections.singleton(standardOutputFile);
  }

  /**
   * Obtm a verso atual do arquivo de parmetros.
   *
   * @return .
   */
  protected abstract String getCurrentParameterFileVersion();

  /**
   * Dispara o evento
   * {@link AlgorithmConfiguratorListener#parameterLabelWasChanged(AlgorithmConfigurator, String, String)}
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   * @param parameterLabel O rtulo do parmetro (No aceita {@code null}).
   */
  protected final void fireParameterLabelWasChanged(String parameterName,
    String parameterLabel) {
    for (AlgorithmConfiguratorListener listener : listeners) {
      listener.parameterLabelWasChanged(this, parameterName, parameterLabel);
    }
  }

  /**
   * Dispara o evento
   * {@link AlgorithmConfiguratorListener#parameterValueWasChanged(AlgorithmConfigurator, String, Object)}
   *
   * @param <V> O tipo do valor.
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   * @param parameterValue O valor do parmetro (Aceita {@code null}).
   */
  protected final <V> void fireParameterValueWasChanged(String parameterName,
    V parameterValue) {
    for (AlgorithmConfiguratorListener listener : listeners) {
      listener.parameterValueWasChanged(this, parameterName, parameterValue);
    }
  }

  /**
   * Dispara o evento
   * {@link AlgorithmConfiguratorListener#parameterWasSetEnabled(AlgorithmConfigurator, String, boolean)}
   * .
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   * @param parameterIsEnabled Indica se o parmetro est habilitado (
   *        {@code true}) ou desabilitado ({@code false}).
   */
  protected final void fireParameterWasSetEnabled(String parameterName,
    boolean parameterIsEnabled) {
    for (AlgorithmConfiguratorListener listener : listeners) {
      listener.parameterWasSetEnabled(this, parameterName, parameterIsEnabled);
    }
  }

  /**
   * Dispara o evento
   * {@link AlgorithmConfiguratorListener#parameterWasSetVisible(AlgorithmConfigurator, String, boolean)}
   * .
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   * @param parameterIsVisible Indica se o parmetro est visvel ({@code true})
   *        ou oculto ({@code false}).
   */
  protected final void fireParameterWasSetVisible(String parameterName,
    boolean parameterIsVisible) {
    for (AlgorithmConfiguratorListener listener : listeners) {
      listener.parameterWasSetVisible(this, parameterName, parameterIsVisible);
    }
  }

  /**
   * Dispara o evento
   * {@link AlgorithmConfiguratorListener#wasSetEnabled(AlgorithmConfigurator)}.
   */
  protected final void fireWasSetEnabled() {
    for (AlgorithmConfiguratorListener listener : listeners) {
      listener.wasSetEnabled(this);
    }
  }

  /**
   * Obtm o valor de uma propriedade do algoritmo como string.
   *
   * @param key chave que identifica a propriedade
   * @param defaultValue valor default no caso de no haver a propriedade.
   *
   * @return .
   */
  private String getStringAlgorithmProperty(String key, String defaultValue) {
    AlgorithmVersionInfo versionInfo = getAlgorithmVersion();
    if (versionInfo == null) {
      return defaultValue;
    }
    String stringValue = versionInfo.getInfo().getPropertyValue(key);
    if (stringValue == null) {
      return defaultValue;
    }
    return stringValue;
  }

  /**
   * Obtm o valor de uma propriedade do algoritmo como float.
   *
   * @param key chave que identifica a propriedade
   * @param defaultValue valor default no caso de no haver a propriedade.
   *
   * @return .
   */
  private float getFloatAlgorithmProperty(String key, float defaultValue) {
    AlgorithmVersionInfo versionInfo = getAlgorithmVersion();
    if (versionInfo == null) {
      return defaultValue;
    }
    String stringValue = versionInfo.getInfo().getPropertyValue(key);
    if (stringValue == null) {
      return defaultValue;
    }
    try {
      return Float.parseFloat(stringValue);
    }
    catch (NumberFormatException e) {
      String msg = MessageFormat.format(
        "A chave {0} do algoritmo {1} cujo valor  {2} no  float. ", key,
        this, stringValue);
      throw new BugException(msg, e);
    }
  }

  /**
   * Atribui uma verso a este configurador.
   *
   * @param algorithmVersion A verso (No aceita {@code null}).
   */
  private void setAlgorithmVersion(AlgorithmVersionInfo algorithmVersion) {
    if (algorithmVersion == null) {
      throw new IllegalArgumentException(
        "O parmetro algorithmVersion est nulo.");
    }
    this.algorithmVersion = algorithmVersion;
  }

  /**
   * Atribui o tipo de execuo a este configurador.
   *
   * @param executionType O tipo de execuo (No aceita {@code null}).
   */
  public void setExecutionType(ExecutionType executionType) {
    if (executionType == null) {
      throw new IllegalArgumentException(
        "O parmetro executionType est nulo.");
    }
    this.executionType = executionType;
  }

  /**
   * Atribui o local e execuo do algoritmo.
   *
   * @param executionLocation O local de execuo do algoritmo.
   */
  public void setExecutionLocation(ExecutionLocation executionLocation) {
    if (executionLocation == null) {
      throw new IllegalArgumentException(
        "O parmetro executionLocation est nulo.");
    }
    this.executionLocation = executionLocation;
  }

  /**
   * Atribui o caminho do arquivo com o descritor deste configurador.
   *
   * @param filePath O caminho (No aceita {@code null}).
   */
  private void setFilePath(String filePath) {
    if (filePath == null) {
      throw new IllegalArgumentException("O parmetro filePath est nulo.");
    }
    this.filePath = filePath;
  }

  /**
   * Obtm a descrio default do comando a ser gerado pelo algoritmo.
   *
   * @return a descrio default do comando
   */
  public String getCommandDescription() {
    return commandDescription;
  }

  /**
   * Estabelece a descrio default do comando a ser gerado pelo algoritmo.
   *
   * @param commandDescription descrio do comando
   */
  public void setCommandDescription(String commandDescription) {
    this.commandDescription = commandDescription;
  }

  /**
   * Indica se o algoritmo prov um cdigo de sada durante sua execuo.
   *
   * @return Verdadeiro se o algoritmo prov o cdigo de sada, ou falso, caso
   *         contrrio.
   */
  public boolean hasExitCode() {
    return hasExitCode;
  }

  /**
   * Determina se o algoritmo prov um cdigo de sada durante sua execuo.
   *
   * @param hasExitCode Verdadeiro se o algoritmo prov o cdigo de sada, ou
   *        falso, caso contrrio.
   */
  public void setHasExitCode(boolean hasExitCode) {
    this.hasExitCode = hasExitCode;
  }

  /**
   * Retorna o arquivo que armazena o cdigo de sada do algoritmo.
   *
   * @return o arquivo que armazena o cdigo de sada.
   */
  public FileURLValue getExitCodeLogFile() {
    return exitCodeLogFile;
  }

  /**
   * Retorna os arquivos que armazenam os cdigos de sada dos algoritmos (so
   * vrios, no caso de fluxo).
   *
   * @return O conjunto de arquivos de cdigo de sada.
   */
  public Set<FileURLValue> getExitCodeLogFiles() {
    if (null == this.exitCodeLogFile) {
      return Collections.emptySet();
    }
    return Collections.singleton(exitCodeLogFile);
  }

  /**
   * Seta o arquivo que armazena o cdigo de sada do algoritmo.
   *
   * @param exitCodeLogFile o arquivo que armazena o cdigo de sada.
   */
  public void setExitCodeLogFile(FileURLValue exitCodeLogFile) {
    this.exitCodeLogFile = exitCodeLogFile;
  }

  /**
   * Retorna o arquivo que armazena os alertas do algoritmo.
   *
   * @return o arquivo que armazena os alertas.
   */
  public FileURLValue getWarningsFile() {
    return warningsFile;
  }

  /**
   * Retorna os arquivos que armazenam os alertas dos algoritmos (so vrios, no
   * caso de fluxo).
   *
   * @return O conjunto de arquivos de alerta.
   */
  public Set<FileURLValue> getWarningsFiles() {
    if (null == this.warningsFile) {
      return Collections.emptySet();
    }
    return Collections.singleton(warningsFile);
  }

  /**
   * Atribui o arquivo que armazena os alertas do algoritmo.
   *
   * @param warningsFile o arquivo que armazena os alertas.
   */
  public void setWarningsFile(FileURLValue warningsFile) {
    this.warningsFile = warningsFile;
  }

  /**
   * Retorna verdadeiro se algum dos parmetros do algoritmo somente podem ser
   * utilizados via pipe. Nesse caso, o algoritmo s pode ser executado como
   * parte de um fluxo. Caso nenhum dos parmetros tenha esse requisito, retorna
   * falso.
   *
   * @return verdadeiro se algum parmetro s pode ser utilizado atravs de pipe
   *         ou falso, caso contrrio.
   */
  public boolean hasParameterThatRequiresPipe() {
    for (AbstractFileParameter param : getFileParameters()) {
      if (param.usesPipe() == FileParameterPipeAcceptance.ALWAYS) {
        return true;
      }
    }
    return false;
  }

  /**
   * Verifica se os valores armazenados no parmetro so vlidos.
   *
   * @param context contexto da validao.
   *
   * @return O resultado da validao.
   * @throws RemoteException em caso de erro na comunicao com servidor.
   */
  public abstract Validation validate(ValidationContext context)
    throws RemoteException;
}
