package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;

import csbase.logic.algorithms.parsers.BooleanParameterFactory;
import csbase.logic.algorithms.parsers.ClientSGAFileParameterFactory;
import csbase.logic.algorithms.parsers.DefaultTableParameterFactory;
import csbase.logic.algorithms.parsers.DoubleListParameterFactory;
import csbase.logic.algorithms.parsers.DoubleParameterFactory;
import csbase.logic.algorithms.parsers.EnumerationParameterFactory;
import csbase.logic.algorithms.parsers.FormulaParameterFactory;
import csbase.logic.algorithms.parsers.InputFileParameterFactory;
import csbase.logic.algorithms.parsers.InputFileParameterParser;
import csbase.logic.algorithms.parsers.InputURLParameterFactory;
import csbase.logic.algorithms.parsers.InputURLParameterParser;
import csbase.logic.algorithms.parsers.IntegerListParameterFactory;
import csbase.logic.algorithms.parsers.IntegerParameterFactory;
import csbase.logic.algorithms.parsers.OutputFileParameterFactory;
import csbase.logic.algorithms.parsers.OutputFileParameterParser;
import csbase.logic.algorithms.parsers.OutputURLParameterFactory;
import csbase.logic.algorithms.parsers.OutputURLParameterParser;
import csbase.logic.algorithms.parsers.ParameterFactory;
import csbase.logic.algorithms.parsers.TextListParameterFactory;
import csbase.logic.algorithms.parsers.TextParameterFactory;
import csbase.logic.algorithms.parsers.elements.IElementStructure;
import csbase.logic.algorithms.parsers.elements.attributes.IElementAttribute;

/**
 * Registro dos parmetros do csbase.
 */
public class DefaultParameterRegistry implements ParameterRegistry {

  /**
   * Mapa de fbricas de parmetros indexado pela tag xml do parmetro.
   */
  private final Map<String, ParameterFactory> parameterFactoryMap;

  /**
   * Construtor.
   */
  public DefaultParameterRegistry() {
    parameterFactoryMap = new HashMap<>();
    parameterFactoryMap.put(
      OutputFileParameterParser.OUTPUT_FILE_PARAMETER_ELEMENT,
      new OutputFileParameterFactory());
    parameterFactoryMap.put(
      InputFileParameterParser.INPUT_FILE_PARAMETER_ELEMENT,
      new InputFileParameterFactory());
    parameterFactoryMap.put(BooleanParameterFactory.BOOLEAN_PARAMETER_ELEMENT,
      new BooleanParameterFactory());
    parameterFactoryMap.put(
      EnumerationParameterFactory.ENUMERATION_PARAMETER_ELEMENT,
      new EnumerationParameterFactory());
    parameterFactoryMap.put(FormulaParameterFactory.FORMULA_PARAMETER_ELEMENT,
      new FormulaParameterFactory());
    parameterFactoryMap.put(IntegerParameterFactory.INTEGER_PARAMETER_ELEMENT,
      new IntegerParameterFactory());
    parameterFactoryMap.put(
      IntegerListParameterFactory.INTEGER_LIST_PARAMETER_ELEMENT,
      new IntegerListParameterFactory());
    parameterFactoryMap.put(
      DoubleListParameterFactory.DOUBLE_LIST_PARAMETER_ELEMENT,
      new DoubleListParameterFactory());
    parameterFactoryMap.put(
      TextListParameterFactory.TEXT_LIST_PARAMETER_ELEMENT,
      new TextListParameterFactory());
    parameterFactoryMap.put(DoubleParameterFactory.DOUBLE_PARAMETER_ELEMENT,
      new DoubleParameterFactory());
    parameterFactoryMap.put(
      DefaultTableParameterFactory.TABLE_PARAMETER_ELEMENT,
      new DefaultTableParameterFactory());
    parameterFactoryMap.put(TextParameterFactory.TEXT_PARAMETER_ELEMENT,
      new TextParameterFactory());
    parameterFactoryMap.put(
      ClientSGAFileParameterFactory.CLIENT_SGA_PARAMETER_ELEMENT,
      new ClientSGAFileParameterFactory());
    parameterFactoryMap.put(InputURLParameterParser.INPUT_URL_PARAMETER_ELEMENT,
      new InputURLParameterFactory());
    parameterFactoryMap.put(
      OutputURLParameterParser.OUTPUT_URL_PARAMETER_ELEMENT,
      new OutputURLParameterFactory());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, ParameterFactory> getParameterFactories() {
    return parameterFactoryMap;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ParameterDocumentation getParameterDocumentation(
    IElementStructure<?> structure, Locale locale) {

    Class<?> parameterClass = structure.getElementClass();
    String file = parameterClass.getSimpleName() + "_" + locale + ".properties";
    InputStream inStream = parameterClass.getResourceAsStream(file);
    Properties prop = new Properties();

    ParameterDocumentation documentation = new ParameterDocumentation();
    if (inStream != null) {
      try {
        prop.load(inStream);
      }
      catch (IOException e) {
        e.printStackTrace();
      }

      String title = prop.getProperty("title");
      documentation.setTitle(title);

      String description = prop.getProperty("description");
      documentation.setDescription(description);

      for (Object attributeObj : structure.getAttributes()) {
        IElementAttribute<?> attribute = (IElementAttribute<?>) attributeObj;

        String name = attribute.getName();
        String attrDoc = prop.getProperty(name);

        documentation.addAttribute(name, attrDoc);
      }
    }

    return documentation;
  }
}
