/*
 * $Author$ $Date$ $Revision$
 */
package csbase.logic.algorithms.parameters.triggers;

import java.io.Serializable;

import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.conditions.Condition;

/**
 * <p>
 * Gatilho
 * </p>
 * 
 * <p>
 * O configurador de algoritmos ({@link SimpleAlgorithmConfigurator} para
 * modificar o estado de um parmetro especfico quando uma condio (
 * {@link Condition}) especfica  satisfeita. O gatilho  a forma de modificar
 * atributos de um parmetro, quando o usurio modificar o valor de outro
 * parmetro.
 * </p>
 * 
 * @param <P> Tipo do parmetro aceito pelo gatilho.
 * @author lmoreira
 */
public abstract class Trigger<P extends Parameter<?>> implements Serializable {
  /**
   * A condio.
   */
  private Condition condition;

  /**
   * O parmetro.
   */
  private P parameter;

  /**
   * Cria um gatilho.
   * 
   * @param parameter O parmetro (No aceita {@code null}).
   * @param condition A condio (No aceita {@code null}).
   */
  protected Trigger(P parameter, Condition condition) {
    setParameter(parameter);
    setCondition(condition);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!getClass().equals(obj.getClass())) {
      return false;
    }
    Trigger<?> trigger = (Trigger<?>) obj;
    return getParameter().equals(trigger.getParameter());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return getParameter().hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    String toString = "Classe = " + getClass().getName() + "\n";
    toString += "Parametro = " + getParameter().getName() + "\n";
    toString += "Condio = " + getCondition() + "\n";
    return toString;
  }

  /**
   * Obtm a condio associada a este gatilho.
   * 
   * @return A condio.
   */
  public final Condition getCondition() {
    return this.condition;
  }

  /**
   * Obtm o parmetro associado a este gatilho.
   * 
   * @return O parmetro.
   */
  public P getParameter() {
    return this.parameter;
  }

  /**
   * <p>
   * Atualiza os atributos do parmetro obtido pelo mtodo
   * {@link #getParameter()} se a condio obtida pelo mtodo
   * {@link #getCondition()} for verdadeira (veja
   * {@link Condition#evaluate(SimpleAlgorithmConfigurator)}. Atualizar o estado
   * do parmetro  executar o mtodo {@link #doAction()}.
   * </p>
   * 
   * <p>
   * Se a condio for falsa, ele desfaz a alterao no estado do parmetro se
   * for possvel. Desfazer a alterao  executar o mtodo
   * {@link #undoAction()}.
   * </p>
   * 
   * @param configurator O configurador (No aceita {@code null}).
   * 
   * @return {@code true} se a alterao de estado foi feita ou {@code false} se
   *         a alterao de estado for desfeita.
   */
  public final boolean update(SimpleAlgorithmConfigurator configurator) {
    if (!this.condition.evaluate(configurator)) {
      undoAction();
      return false;
    }
    doAction();
    return true;
  }

  /**
   * Atualiza o estado do parmetro associado a este gatilho.
   */
  protected abstract void doAction();

  /**
   * Desfaz as alterao feitas pelo mtodo {@link #doAction()}. Se no for
   * possvel desfazer estas alteraes, este mtodo no deve fazer nada.
   */
  protected abstract void undoAction();

  /**
   * Atribui uma condio a este gatilho.
   * 
   * @param condition A condio (No aceita {@code null}).
   */
  private void setCondition(Condition condition) {
    if (condition == null) {
      throw new IllegalArgumentException("O parmetro condition est nulo.");
    }
    this.condition = condition;
  }

  /**
   * Atribui o parmetro a este gatilho.
   * 
   * @param parameter O parmetro (No aceita {@code null}).
   */
  private void setParameter(P parameter) {
    if (parameter == null) {
      throw new IllegalArgumentException("O parmetro parameter est nulo.");
    }
    this.parameter = parameter;
  }
}
