package csbase.logic.algorithms.parsers.triggers;

import java.io.Serializable;
import java.util.Set;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.triggers.Trigger;
import csbase.logic.algorithms.parsers.XmlParser;

/**
 * <p>
 * Fbrica de {@link Trigger Gatilhos}
 * </p>
 * 
 * <p>
 * Ela  responsvel por analisar o elemento de XML que define um
 * {@link Trigger gatilho} para {@link Parameter parmetros}.
 * </p>
 * 
 * <p>
 * Exemplo: o exemplo ilustra a criao de uma {@link Trigger gatilho} do tipo
 * {@link ShowParameterTriggerFactory}. Ela ir ser ativada quando o parmetro
 * "Parametro1" tiver valor verdadeiro. Ela ir exibir o parmetro "Parametro2".
 * </p>
 * <p>
 * {@code 
 * ...
 * <booleano nome="Parametro1" .../>
 * <texto nome="Parametro2" ... />
 * ...
 * <exibir parametro="Parametro2">
 *   <condicao parametro="Parametro1" valor="sim"/>
 * </exibir>
 * ...
 * }
 * </p>
 * 
 * <p>
 * Um {@link Trigger gatilho} opera em apenas 1 {@link Parameter parmetro},
 * porm para algumas fbricas de gatilho suportam criar um elemento para criar
 * gatilhos para vrios parmetros. Isto  suportado, quando no h atributos ou
 * elementos adicionais para criar a trigger.
 * </p>
 * <p>
 * Exemplo: o exemplo ilustra a criao de 2 {@link Trigger gatilhos} do tipo
 * {@link ShowParameterTriggerFactory}. Ambos sero ativados quando o parmetro
 * "Parametro1" tiver valor verdadeiro. Uma instncia ir exibir o parmetro
 * "Parametro2" e a outra, o parmetro "Parametro3".
 * </p>
 * <p>
 * {@code 
 * ...
 * <booleano nome="Parametro1" .../>
 * <texto nome="Parametro2" ... />
 * <texto nome="Parametro3" ... />
 * ...
 * <exibir parametro="Parametro2">
 *   <condicao parametro="Parametro1" valor="sim"/>
 *   <parametro nome="Parametro3"/>
 * </exibir>
 * ...
 * }
 * </p>
 * <p>
 * Exemplo: o exemplo ilustra a criao de 1 {@link Trigger gatilho} do tipo
 * {@link ChangeDefaultValueTriggerFactory}. Ele ser ativado quando o parmetro
 * "Parametro1" tiver valor verdadeiro. O {@link Trigger gatilho} ir modificar
 * o valor-padro do parmetro "Parametro2" para "Valor2". Por precisar do
 * valor-padro, ela no suporta a criao de mltiplas instncias.
 * </p>
 * <p>
 * {@code 
 * ...
 * <booleano nome="Parametro1" .../>
 * <texto nome="Parametro2" padrao="Valor1" ... />
 * ...
 * <trocar_valor_padrao parametro="Parametro2" novo_valor_padrao="Valor2">
 *   <condicao parametro="Parametro1" valor="sim"/>
 * </exibir>
 * ...
 * }
 * </p>
 */
public interface TriggerFactory extends Serializable {

  /**
   * Obtm o nome do elemento no XML.
   * 
   * @return O nome do elemento no XML.
   */
  String getElementName();

  /**
   * Cria uma instncia do gatilho a partir das informaes do parser do
   * configurador de algoritmos.
   * 
   * @param parser Parser do configurador.
   * @param configurator O configurador de algoritmos.
   * 
   * @return Os gatilhos criados.
   * @throws ParseException Caso no seja possvel criar o gatilho com as
   *         informaes do parser.
   */
  Set<Trigger<?>> createTriggers(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException;
}
