/*
 * SGAServiceInterface.java
 * 
 * $Author: tatimf $ $Revision: 175500 $ - $Date: 2008-06-10 13:11:41 -0300
 * (Tue, 10 Jun 2008) $
 */
package csbase.remote;

import java.rmi.RemoteException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import csbase.logic.ClientSGAFile;
import csbase.logic.CommandFinalizationType;
import csbase.logic.CommandInfo;
import csbase.logic.SGAInfo;
import csbase.logic.SGASet;
import csbase.logic.ServerGroupInfo;

/**
 * Interface remota para acesso ao Servio de Gerncia de SGAs
 * 
 * @author Ane Lcia de Moura, Andr Clinio e Cristina Ururahy
 */
public interface SGAServiceInterface extends ServiceInterface,
  RemoteObservable {

  /** Nome do servio para o <code>ServiceManager</code> */
  public static final String SERVICE_NAME = "SGAService";

  /**
   * Retorno de todos os nomes de servidores cadastrados no SSI (SGA-Service)
   * 
   * @return um vetor de strings com os nomes dos servidores.
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public Vector<String> getAllSGANames() throws RemoteException;

  /**
   * Obtm um SGA.
   * 
   * @param sgaName nome do SGA
   * 
   * @return as informaes do SGA.
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public SGASet getSGASet(final String sgaName) throws RemoteException;

  /**
   * Obtm as informaes de todos os ns de um SGA.
   * 
   * @param sgaName nome do SGA
   * 
   * @return array contendo as informaes dos ns do SGA.
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public SGAInfo[] getAllInfo(final String sgaName) throws RemoteException;

  /**
   * Obtm as informaes de um determinado n de um SGA.
   * 
   * @param sgaName nome do SGA
   * @param index ndice do n
   * 
   * @return as informaes do n especificado
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public SGAInfo getInfo(final String sgaName, final int index)
    throws RemoteException;

  /**
   * Retorna lista com os arquivos filhos dado um SGA e um path.
   * 
   * @param sgaName nome do SGA.
   * @param path path pai.
   * @return lista com arquivos filhos.
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public List<ClientSGAFile> getChildren(String sgaName, String path)
    throws RemoteException;

  /**
   * Retorna {@link ClientSGAFile} equivalente ao dado path.
   * 
   * @param sgaName nome do SGA.
   * @param path path do arquivo ou diretrio.
   * @return {@link ClientSGAFile} equivalente ao dado path.
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public ClientSGAFile getFile(String sgaName, String path)
    throws RemoteException;

  /**
   * Retorna o intervalo de atualizao das informaes dos SGAs
   * 
   * @return intervalo de atualizao em segundos
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public int getUpdateInterval() throws RemoteException;

  /**
   * Retorna o intervalo de atualizao das informaes dos comandos.
   * 
   * @return intervalo de atualizao em segundos
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public int getCommandsUpdateInterval() throws RemoteException;

  /**
   * Obtm os grupos de servidores definidos por um usurio.
   * 
   * @return um array com as informaes dos grupos ou null em caso de erro
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public ServerGroupInfo[] getServerGroups() throws RemoteException;

  /**
   * Atualiza os grupos de servidores definidos por um usurio.
   * 
   * @param groups array com as informaes dos grupos
   * 
   * @return true se a operao foi bem sucedida, false em caso de erro
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public boolean setServerGroups(final ServerGroupInfo[] groups)
    throws RemoteException;

  /**
   * Atualiza a descrio do comando.
   * 
   * @param commandId identificador do comando a ser atualizado.
   * @param description nova descrio do comando.
   * @return <code>true</code> se o comando foi atualizado com sucesso
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public boolean updateCommandDescription(String commandId, String description)
    throws RemoteException;

  /**
   * Solicita o shutdown de um SGA especfico.
   * 
   * @param sgaName o nome do SGA.
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   * 
   * @return true se o SGA tiver sido desativado com sucesso.
   */
  public boolean shutdownSGA(final String sgaName) throws RemoteException;

  /**
   * Solicita o relanamento de um SGA especfico
   * 
   * @param sgaName o nome do SGA.
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   * 
   * @return true se o SGA for reiniciado com sucesso.
   */
  public boolean restartSGA(final String sgaName) throws RemoteException;

  /**
   * Solicita o shutdown de todos os SGAs.
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   * 
   * @return coleo de SGAs que no puderam ser reiniciados. Se tudo correr
   *         bem, a lista estar vazia.
   */
  public Collection<String> shutdownAllSGAs() throws RemoteException;

  /**
   * Solicita o relanamento de todos os SGAs.
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   * 
   * @return coleo de SGAs que no puderam ser reiniciados. Se tudo correr
   *         bem, a lista estar vazia.
   */
  public Collection<String> restartAllSGAs() throws RemoteException;

  /**
   * Retorna as informaes da monitorao de um comando em execuo.
   * 
   * @param sgaName O nome do sga.
   * @param cmdId O identificador do comando.
   * 
   * @return as informaes da monitorao de um comando em execuo.
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public CommandInfo getSGACommand(final String sgaName, final String cmdId)
    throws RemoteException;

  /**
   * Retorna um Vector com informaes da monitorao de todos os comandos em
   * execuo nos diversos SGAs.
   * 
   * @return informaes da monitorao de todos os comandos em execuo.
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public Vector<CommandInfo> getAllSGACommands() throws RemoteException;

  /**
   * Retorna um conjunto com informaes da monitorao de todos os comandos em
   * execuo em um SGA.
   * 
   * @param sgaName O nome do SGA.
   * 
   * @return O conjunto de CommandInfo ou {@code null} caso no seja possvel
   *         acessar o SGA.
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public Set<CommandInfo> getSGACommands(String sgaName) throws RemoteException;

  /**
   * Envio de notificao para cancelar a execuo de um comando especfico
   * 
   * @param cmdId identificador do comando
   * 
   * @return true se conseguiu cancelar o comando ou false se houver falha na
   *         comunicao.
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public boolean killCommand(final String cmdId) throws RemoteException;

  /**
   * Envio de notificao para cancelar a execuo de um comando especfico
   * 
   * @param sgaName nome do SGA
   * @param cmdId identificador do comando
   * 
   * @return true se conseguiu cancelar o comando ou false se houver falha na
   *         comunicao.
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public boolean killCommand(final String sgaName, final String cmdId)
    throws RemoteException;

  /**
   * Finalizao de comando pelo administrador (kill). Remove o objeto do
   * comando mesmo que tenha ocorrido erro. Deve ser usado no caso em que no h
   * mais possibilidade do comando estar executando. Por exemplo, ocorreu
   * problema de hardware na mquina.
   * 
   * @param sgaName nome do SGA.
   * @param cmdId identificador do comando.
   * 
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public void killCommandAnyway(final String sgaName, final String cmdId)
    throws RemoteException;

  /**
   * Obtm um mapa com as estatsticas de tipos de trmino de execuo.
   * 
   * @param flowResults <code>true</code> para obter estatsticas de fluxos,
   *        <code>false</code> para obter das execues simples
   * 
   * @return mapa (no-modificvel) com as estatsticas de tipos de trmino de
   *         execuo
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public Map<CommandFinalizationType, Integer> getExeResultsStats(
    boolean flowResults) throws RemoteException;

  /**
   * Obtm um mapa com as estatsticas de execues por mquina (ns SGA).
   * 
   * @return mapa (no-modificvel) com as estatsticas de execues por mquina
   * @throws RemoteException se houver falha na chamada remota ao servidor.
   */
  public Map<String, Integer> getSGAsStats() throws RemoteException;

}
