/**
 * $Id: SharedObjectServiceInterface.java 181432 2018-04-10 13:21:32Z clinio $
 */

package csbase.remote;

import java.rmi.RemoteException;

import csbase.exception.ServiceFailureException;
import csbase.logic.SharedObject;

/**
 * Servio de armazenamento e compartilhamento de objetos genricos. Os dados
 * so guardados separados por categoria e por usurio. Cada objeto 
 * identificado univocamente pela sua categoria, seu usurio e seu nome. Cada
 * objeto pode ser compartilhado com outros usurios quaisquer. Todo objeto
 * possui uma data de criao e uma de ltima modificao. O contedo real do
 * objeto  irrelevante, desde que seja um Object serializvel.
 */
public interface SharedObjectServiceInterface extends ServiceInterface {

  /** Nome do servio para o ServiceManager */
  String SERVICE_NAME = "SharedObjectService";

  /**
   * Salva um shared object no servidor, e o registra na respectiva tabela em
   * memria. Caso o objeto ainda no exista na tabela, um novo objeto  criado.
   * Caso j exista, apenas o timestamp de ltima modificao  alterado.
   * <p>
   * Apenas o usurio dono do objeto pode atualiz-lo. Os campos
   * <code>ownerUserId</code>, <code>created</code> e <code>lastModified</code>
   * so especificados pelo servio: os valores recebidos no so utilizados.
   * 
   * @param object shared object (atributos + contedo) a ser armazenado
   * @return o shared object com as data atualizadas, <b>sem o contedo</b>
   * @throws RemoteException se houve algum erro
   */
  SharedObject saveSharedObject(SharedObject object) throws RemoteException;

  /**
   * Remove um objeto. Note que o usurio dono do objeto no  fornecido, pois
   * ele  determinado automaticamente pelo servio. Apenas o usurio dono do
   * objeto pode remov-lo.
   * 
   * @param category A categoria do objeto.
   * @param name O nome do objeto.
   * 
   * @throws RemoteException em caso de falha na comunicao com o servidor
   * @throws ServiceFailureException em caso de falha na operao
   */
  void removeSharedObject(String category, String name) throws RemoteException;

  /**
   * Retorna todos os objetos de uma dada categoria que o usurio tem acesso.
   * Todas as instncias de SharedObject retornadas por este mtodo possuem o
   * campo content igual a <code>null</code>. Ou seja, este mtodo serve para
   * recuperar os atributos dos objetos acessveis. Para obter o objeto real,
   * utilize o mtodo {@link #getSharedObject(String, Object, String)}.
   * 
   * @param category A categoria na qual os objetos sero pesquisados.
   * 
   * @return Um array com todos os objetos acessveis pelo usurio.
   * 
   * @throws RemoteException em caso de falha na comunicao com o servidor
   * @throws ServiceFailureException em caso de falha na operao
   */
  SharedObject[] getSharedObjectAttributes(String category)
    throws RemoteException;

  /**
   * Retorna um objeto especfico, dada a categoria, o usurio dono do objeto e
   * o nome deste objeto. Todos os campos com os atributos so retornados
   * preenchidos e o campo content possui referncia para o objeto real.
   * 
   * @param category A categoria do objeto.
   * @param userId O identificador do usurio dono do objeto.
   * @param name O nome do objeto.
   * 
   * @return objeto completo (atributos e o contedo), ou <code>null</code> em
   *         caso de erro
   * @throws RemoteException em caso de falha na comunicao com o servidor
   * @throws ServiceFailureException em caso de falha na operao
   */
  SharedObject getSharedObject(String category, Object userId, String name)
    throws RemoteException;

  /**
   * Destroi todos os dados de um usurio.
   * @param userId
   */
  boolean removeAllFromUser(Object userId) throws RemoteException;
}
