/*
 * $Id: ClientProjectFile.java 129343 2012-05-25 20:08:21Z oikawa $
 */

package csbase.logic;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.Arrays;

import tecgraf.ftc.common.exception.FailureException;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.io.FileUtils;
import csbase.exception.project.FileLockedException;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.remote.ClientRemoteLocator;

/**
 * Modela a viso que um cliente tem de um arquivo ou diretrio de um algoritmo.
 *
 * @author Tecgraf/PUC-Rio
 */
public class ClientAlgorithmFile implements ClientFile, Serializable,
Comparable<ClientAlgorithmFile> {

  /**
   * Canal, no  enviado quando o CPF  serializado.
   */
  private transient SyncRemoteFileChannel channel;

  /**
   * Indicativo de diretrio.
   */
  private boolean isDirectory;

  /**
   * Nome do arquivo.
   */
  private String name;

  /**
   * Caminho para o arquivo, a partir da raiz.
   */
  private String[] path;

  /**
   * Nmero de pedidos de abertura de canais.
   */
  private int numberOfChannelOpens;

  /** Pai (diretrio) */
  private ClientAlgorithmFile parent;

  /** Informaes sobre o algoritmo que possui o arquivo */
  private final AlgorithmInfo algorithmInfo;

  /** Identificador da verso a qual o arquivo pertence */
  private final AlgorithmVersionId versionId;

  /** Nome da plataforma */
  private final String platformName;

  //  /** Lista de arquivos-filho */
  //  private List<ClientAlgorithmFile> children = null;

  /** Indica se o arquivo est atualizado em relao ao servidor. */
  private boolean updated;

  /** Tamanho */
  private long size;

  /** Estado */
  private boolean opened;

  /** Tipo do arquivo na rvores de algoritmos */
  private AlgorithmFileType treeFileType;

  /** Tipo do arquivo na rvore de algoritmos */
  public static enum AlgorithmFileType {
    /** Arquivo executvel (ou binrio) */
    EXECUTABLE,
    /** Arquivo de configurao */
    CONFIGURATION,
    /** Arquivo de documentao */
    DOCUMENTATION
  }

  /**
   * Construtor para arquivos de configurao e documentao
   *
   * @param algorithmInfo
   * @param versionId
   *
   * @param name nome do arquivo
   * @param path caminho para o arquivo, a partir da raiz
   * @param treeFileType Tipo do arquivo na rvore de algoritmos
   * @param isDirectory
   */
  public ClientAlgorithmFile(AlgorithmInfo algorithmInfo,
    AlgorithmVersionId versionId, String name, String[] path,
    AlgorithmFileType treeFileType, boolean isDirectory) {
    this(algorithmInfo, versionId, "", name, path, 0, null, treeFileType,
      isDirectory);
  }

  /**
   * Construtor para arquivos executveis
   *
   * @param algorithmInfo
   * @param versionId
   * @param platformName Nome da plataforma
   *
   * @param name nome do arquivo
   * @param path caminho para o arquivo, a partir da raiz
   * @param isDirectory
   */
  public ClientAlgorithmFile(AlgorithmInfo algorithmInfo,
    AlgorithmVersionId versionId, String platformName, String name,
    String[] path, boolean isDirectory) {
    this(algorithmInfo, versionId, platformName, name, path, 0, null,
      AlgorithmFileType.EXECUTABLE, isDirectory);
  }

  /**
   * Construtor
   *
   * @param algorithmInfo
   * @param versionId
   * @param platformName nome da plataforma, quando arquivo executvel.
   *
   * @param name nome do arquivo
   * @param path caminho para o arquivo, a partir da raiz
   * @param size tamanho do arquivo.
   * @param parent diretrio-pai. Se estiver sendo construido no servidor, este
   *        valor deve ser null.
   * @param treeFileType Tipo do arquivo na rvore de algoritmos
   * @param isDirectory
   */
  public ClientAlgorithmFile(AlgorithmInfo algorithmInfo,
    AlgorithmVersionId versionId, String platformName, String name,
    String[] path, long size, ClientAlgorithmFile parent,
    AlgorithmFileType treeFileType, boolean isDirectory) {
    this.algorithmInfo = algorithmInfo;
    this.versionId = versionId;
    this.platformName = platformName;
    this.name = name;
    this.path = path;
    this.parent = parent;
    this.treeFileType = treeFileType;
    this.isDirectory = isDirectory;
    this.size = size;
    this.channel = null;
    this.numberOfChannelOpens = 0;
    this.opened = false;
    this.updated = true;
  }

  /**
   * Retorna o nome do arquivo.
   *
   * @return Nome do arquivo.
   */
  @Override
  public String getName() {
    return this.name;
  }

  /**
   * Muda o nome do arquivo.
   *
   * @param name novo nome do arquivo.
   */
  public void setName(String name) {
    this.name = name;
    path[path.length - 1] = name;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getStringPath() {
    String[] pathArray = getPath();
    if ((pathArray == null) || (pathArray.length <= 0)) {
      return "";
    }
    StringBuilder path = new StringBuilder();
    for (int i = 0; i < (pathArray.length - 1); i++) {
      path.append(pathArray[i]);
      path.append('/');
    }
    return path.append(pathArray[pathArray.length - 1]).toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] getPath() {
    return path.clone();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    final String fileExtension = FileUtils.getFileExtension(getName());
    final ProjectFileType pft =
      ProjectFileType.getProjectFileTypeFromExtension(fileExtension, false);
    if (pft == null) {
      return ProjectFileType.UNKNOWN;
    }
    return pft.getCode();
  }

  /**
   * {@inheritDoc}
   *
   * Este mtodo no deve ser utilizado no servidor pois o valor da propriedade
   * a que ele referencia s ser atribuido no cliente.
   */
  @Override
  public ClientAlgorithmFile getParent() {
    if (parent != null) {
      return parent;
    }
    int parentIdx = path.length - 2;
    if (parentIdx < 0 || treeFileType != AlgorithmFileType.EXECUTABLE) {
      return null;
    }
    String parentName = path[parentIdx];
    String[] parentPath = Arrays.copyOfRange(path, 0, parentIdx);
    parent =
      new ClientAlgorithmFile(algorithmInfo, versionId, platformName,
        parentName, parentPath, true);
    return this.parent;
  }

  /**
   * Redefine o pai do arquivo.
   *
   * @param parent novo pai
   */
  public void setParent(ClientAlgorithmFile parent) {
    this.parent = parent;
  }

  //  /**
  //   * Reseta a lista de filhos.
  //   */
  //  public void resetChildren() {
  //    children = null;
  //  }

  @Override
  public ClientFile[] getChildren() throws Exception {
    // TODO Ainda no necessrio. Ser obtido quando tivermos o StorageService.
    return null;
  }

  /**
   * Atualiza as informaes desse arquivo buscando os dados reais no servidor.
   *
   * @throws RemoteException Falha de rmi
   */
  public void updateInfo() throws RemoteException {
    // TODO: Precisamos disso?
    //    ClientProjectFileInfo info =
    //      ClientRemoteLocator.projectService.getUpdatedFileInfo(this.projectId,
    //        getPath());
    //    this.type = info.getType();
    //    this.size = info.getSize();
  }

  /**
   * Indica se este arquivo  um diretrio.
   *
   * @return Verdadeiro caso seja um diretrio e falso caso contrrio.
   */
  @Override
  public boolean isDirectory() {
    return this.isDirectory;
  }

  /**
   * Informa o tamanho corrente deste arquivo. Reflete o nmero corrente de
   * bytes do arquivo.
   *
   * @return Tamanho corrente do arquivo.
   */
  @Override
  public long size() {
    return this.size;
  }

  /**
   * Informa a data da ltima modificao deste arquivo (formato numrico).
   *
   * @return data da ltima modificao deste arquivo (formato numrico)
   */
  @Override
  public long getModificationDate() {
    return 0L;
  }

  /**
   * Indica se o arquivo est atualizado em relao ao servidor.
   *
   * @return <code>true</code> se est atualizado
   */
  public boolean isUpdated() {
    return updated;
  }

  /**
   * Informa se o arquivo est atualizado em relao ao servidor.
   *
   * @param updated <code>true</code> se est atualizado.
   */
  public void setUpdated(boolean updated) {
    this.updated = updated;
  }

  /**
   * Verifica se o arquivo existe no servidor.
   *
   * @return true se o arquivo existe no servidor.
   * @throws RemoteException se houver falha de RMI
   */
  @Override
  public boolean exists() throws RemoteException {
    return existsFile(algorithmInfo, versionId, getPath());
  }

  /**
   * Verifica se o arquivo existe no servidor.
   *
   * @param info dados de um algoritmo
   * @param algoVersion identificador de verso do algoritmo
   *
   * @param path o path do arquivo.
   *
   * @return true se o arquivo existe no servidor.
   * @throws RemoteException se houver falha de RMI
   */
  private boolean existsFile(final AlgorithmInfo info,
    final AlgorithmVersionId algoVersion, final String[] path)
      throws RemoteException {
    if (isExecutableFile()) {
      return ClientRemoteLocator.algorithmService.execFileExists(info.getId(),
        algoVersion, platformName, Arrays.toString(path));
    }
    if (isConfigurationFile()) {
      return ClientRemoteLocator.algorithmService.configFileExists(
        info.getId(), algoVersion, Arrays.toString(path));
    }
    return ClientRemoteLocator.algorithmService.docFileExists(info.getId(),
      algoVersion, Arrays.toString(path));
  }

  /**
   * Mtodo interno para decomposio de um caminho de arquivo (<i>path</i>) em
   * um array de strings; onde cada elemento  montado no separador
   * <code>"/"</code>.
   *
   * @param path o caminho lgico do arquivo.
   *
   * @return o array.
   */
  static public String[] splitPath(final String path) {
    String[] pathArray = path.split("/");
    int i = 0;
    while ((i < pathArray.length) && pathArray[i].equals("")) {
      i++;
    }
    if (i > 0) {
      final String[] newPath = new String[pathArray.length - i];
      System.arraycopy(pathArray, i, newPath, 0, newPath.length);
      pathArray = newPath;
    }
    return pathArray;
  }

  /**
   * Ajusta o tamanho corrente deste arquivo.
   *
   * @param newSize o novo tamanho a ser ajustado.
   *
   * @throws IOException Em caso de erro.
   */
  public void truncate(long newSize) throws IOException {
    try {
      this.channel.setSize(newSize);
    }
    catch (Exception e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * Cria um canal de acesso ao contedo deste arquivo.
   *
   * @param readOnly Indica se o acesso ser de leitura ou leitura e escrita.
   * @return O canal construdo e j aberto.
   * @throws RemoteException Em caso de falha na operao.
   */
  private SyncRemoteFileChannel openChannel(boolean readOnly)
    throws RemoteException {
    RemoteFileChannelInfo info = getChannel(readOnly);
    if (info == null) {
      return null;
    }
    SyncRemoteFileChannel srfc =
      new SyncRemoteFileChannel(info.getIdentifier(), info.isWritable(), info
        .getHost(), info.getPort(), info.getKey());
    try {
      srfc.open(readOnly);
      return srfc;
    }
    catch (Exception e) {
      this.channel = null;
      throw new RemoteException(e.getMessage(), e);
    }
  }

  /**
   * Obtm o canal de acesso ao contedo do arquivo com base no tipo do arquivo
   * da rvore de algoritmos
   *
   * @param readOnly Indica se o acesso ser de leitura ou escrita.
   * @return O canal .
   * @throws RemoteException Em caso de falha na operao.
   */
  private RemoteFileChannelInfo getChannel(boolean readOnly)
    throws RemoteException {
    if (isExecutableFile()) {
      return ClientRemoteLocator.algorithmService.openExecFileChannel(
        algorithmInfo.getId(), versionId, platformName, getStringPath(),
        readOnly);
    }
    if (isConfigurationFile()) {
      return ClientRemoteLocator.algorithmService.openConfFileChannel(
        algorithmInfo.getId(), versionId, getStringPath(), readOnly);
    }
    if (isDocumentationFile()) {
      return ClientRemoteLocator.algorithmService.openDocFileChannel(
        algorithmInfo.getId(), versionId, getStringPath(), readOnly);
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void open(final boolean readOnly) throws RemoteException {
    updateInfo(); // atualiza info do arquivo antes de abrir
    if (!this.opened) {
      this.channel = openChannel(readOnly);
      this.numberOfChannelOpens = 1;
      this.opened = true;
    }
    else {
      this.numberOfChannelOpens++;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long position() {
    return this.channel.getPosition();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void position(long newPosition) throws IOException {
    try {
      this.channel.setPosition(newPosition);
    }
    catch (FailureException e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * L uma sequncia de bytes do arquivo. Esse mtodo tenta fazer leituras
   * consecutivas at conseguir a quantidade de bytes solicitada, ou atingir o
   * final do arquivo. Caso no consiga recuperar a quantidade solicitada sem
   * ter atingido o final do arquivo (isto , por problemas de comunicao ou
   * outros), o mtodo lana uma exceo.
   *
   * @param dst O array a ser preenchido.
   * @param position A posio do arquivo a partir da qual a leitura deve ser
   *        iniciada.
   * @return A quantidade de bytes lidos. Se for menor que o nmero de bytes
   *         solicitados, o final do arquivo foi atingido.
   *
   * @throws IOException Caso ocorra alguma falha no procedimento.
   */
  @Override
  public int read(byte[] dst, long position) throws IOException {
    try {
      // S  necessrio limitar em avail por causa do erro [FTC-13].
      int length = (int) Math.min(dst.length, channel.getSize() - position);
      return channel.syncRead(dst, 0, length, position);
    }
    catch (FailureException e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * L uma sequncia de bytes do arquivo. Esse mtodo tenta fazer leituras
   * consecutivas at conseguir a quantidade de bytes solicitada, ou atingir o
   * final do arquivo. Caso no consiga recuperar a quantidade solicitada sem
   * ter atingido o final do arquivo (isto , por problemas de comunicao ou
   * outros), o mtodo lana uma exceo.
   *
   * @param target O array a ser preenchido.
   * @param offset O deslocamento no array a partir do qual os bytes sero
   *        armazenados.
   * @param length A quantidade de bytes a serem lidos do arquivo.
   * @param position A posio do arquivo a partir da qual a leitura deve ser
   *        iniciada.
   * @return A quantidade de bytes lidos. Se for menor que o nmero de bytes
   *         solicitados, o final do arquivo foi atingido.
   *
   * @throws IOException Caso ocorra alguma falha no procedimento.
   */
  @Override
  public int read(byte[] target, int offset, int length, long position)
    throws IOException {
    try {
      // S  necessrio limitar em avail por causa do erro [FTC-13].
      length = (int) Math.min(length, channel.getSize() - position);
      return channel.read(target, offset, length, position);
    }
    catch (FailureException e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] src, long position) throws IOException,
  FileLockedException {
    try {
      channel.syncWrite(src, 0, src.length, position);
    }
    catch (tecgraf.ftc.common.exception.FileLockedException e) {
      throw new FileLockedException(getName(), isDirectory());
    }
    catch (Exception e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] src, int off, int len, long position)
    throws IOException, FileLockedException {
    try {
      channel.syncWrite(src, off, len, position);
    }
    catch (tecgraf.ftc.common.exception.FileLockedException e) {
      throw new FileLockedException(getName(), isDirectory());
    }
    catch (Exception e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close(boolean force) throws RemoteException, IOException {
    if (this.opened) {
      if (--this.numberOfChannelOpens == 0) {
        try {
          this.channel.close();
          this.channel = null;
          this.opened = false;
        }
        catch (FailureException e) {
          if (force) {
            this.channel = null;
            this.opened = false;
          }
          throw new RemoteException(e.getMessage(), e);
        }
      }
    }
  }

  /**
   * Retorna um texto descritivo do arquivo.
   *
   * @return .
   */
  @Override
  public String toString() {
    return this.name;
  }

  /**
   * Testa a igualdade de dois arquivos. Dois <code>ClientProjectFile</code> so
   * considerados iguais se seus paths absolutos forem os mesmos.
   *
   * @param o Objeto a ser comparado com o atual
   *
   * @return true se ambos os objetos forem considerados iguais
   */
  @Override
  public boolean equals(Object o) {
    if (o == null) {
      return false;
    }
    if (!this.getClass().equals(o.getClass())) {
      return false;
    }
    ClientAlgorithmFile other = (ClientAlgorithmFile) o;

    return this.algorithmInfo.equals(other.getInfo())
      && versionId.equals(other.getAlgorithmVersion())
      && Arrays.equals(path, other.path);
  }

  /**
   * Obtm os dados de um algoritmo.
   *
   * @return os dados de um algoritmo.
   */
  public AlgorithmInfo getInfo() {
    return algorithmInfo;
  }

  /**
   * Obtm a verso do algoritmo que este arquivo est associado.
   *
   * @return a verso do algoritmo que este arquivo est associado.
   */
  public AlgorithmVersionId getAlgorithmVersion() {
    return versionId;
  }

  /**
   * Obtm a plataforma associada ao arquivo.
   *
   * @return platformName
   */
  public String getPlatformName() {
    if (this.isExecutableFile()) {
      return platformName;
    }
    return null;
  }

  /**
   * Redefine a chave pela qual o arquivo deve ser buscado/inserido em
   * hashtables.
   *
   * @return chave para insero/busca em hashtables.
   */
  @Override
  public int hashCode() {
    int hash = algorithmInfo.hashCode() + versionId.hashCode();
    for (String part : path) {
      hash += part.hashCode();
    }
    return hash;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int compareTo(ClientAlgorithmFile other) {
    return this.name.compareTo(other.name);
  }

  /**
   * Verifica se algum arquivo de projeto  diretrio.
   *
   * @param projectFiles Os arquivos que se deseja verificar.
   *
   * @return true, caso algum dos arquivos de projeto seja diretrio, ou false,
   *         caso contrrio.
   */
  public static boolean hasDirectories(ClientAlgorithmFile[] projectFiles) {
    for (int i = 0; i < projectFiles.length; i++) {
      if (projectFiles[i].isDirectory()) {
        return true;
      }
    }
    return false;
  }

  /**
   * Indica se este arquivo  o root da arvore.
   *
   * @return Retorna boolean indicando se este arquivo  o root da arvore.
   */
  public boolean isRoot() {
    return parent == null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public InputStream getInputStream() throws IOException {
    SyncRemoteFileChannel srfc = openChannel(true);
    String id = getStringPath();
    return new RemoteFileInputStream(id.getBytes(), srfc);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public OutputStream getOutputStream() throws IOException {
    SyncRemoteFileChannel srfc = openChannel(false);
    String id = getStringPath();
    return new RemoteFileOutputStream(id.getBytes(), srfc);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientFileType getClientFileType() {
    return ClientFileType.REMOTE;
  }

  /**
   * Obtm o tipo do arquivo na rvore de algoritmos. isExecutableFile
   *
   * @return treeFileType o tipo do arquivo na rvore de algoritmos.
   */
  public AlgorithmFileType getTreeFileType() {
    return treeFileType;
  }

  /**
   * Retorna true se o arquivo  arquivo executvel (binrio) ou se o diretrio
   * est no diretrio que contm os executveis para as plataformas.
   *
   * @return true se o arquivo  arquivo executvel (binrio) ou se o diretrio
   *         est no diretrio que contm os executveis para as plataformas,
   *         false caso contrrio.
   */
  private boolean isExecutableFile() {
    return treeFileType == AlgorithmFileType.EXECUTABLE;
  }

  /**
   * Retorna true se o arquivo  arquivo de configurao.
   *
   * @return true se o arquivo  arquivo de configurao. false caso contrrio.
   */
  private boolean isConfigurationFile() {
    return treeFileType == AlgorithmFileType.CONFIGURATION;
  }

  /**
   * Retorna true se o arquivo  arquivo de documentao.
   *
   * @return true se o arquivo  arquivo de documentao. false caso contrrio.
   */
  private boolean isDocumentationFile() {
    return treeFileType == AlgorithmFileType.DOCUMENTATION;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canExecute() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canRead() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canWrite() {
    return true;
  }
}
