/*
 * $Id: SimpleAlgorithmParser.java 174083 2016-06-01 21:03:25Z isabella $
 */
package csbase.logic.algorithms.parsers;

import java.io.Reader;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import csbase.exception.OperationFailureException;
import csbase.exception.ParseException;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.ExecutionLocation;
import csbase.logic.algorithms.ExecutionLocation.ExecutionLocationConverter;
import csbase.logic.algorithms.ExecutionType;
import csbase.logic.algorithms.parameters.EnumerationItem;
import csbase.logic.algorithms.parameters.EnumerationListParameter;
import csbase.logic.algorithms.parameters.HiddenParameter;
import csbase.logic.algorithms.parameters.InputFileParameter;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.ParameterLoader;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.ValidationExpression;
import csbase.logic.algorithms.parsers.triggers.TriggerParser;

/**
 * <p>
 * Analisador de arquivos que descrevem {@link SimpleAlgorithmConfigurator
 * configurador de algoritmos simples}.
 * </p>
 * 
 * <p>
 * Existem 2 arquivos que descrevem um configurador de algoritmos simples: o
 * {@code config.xml} que contm as informaes necessrias para a criao do
 * configurador de algoritmos e o arquivo de patch que  gerado pelo
 * {@link ParameterLoader} para realizar modificaes em um configurador de
 * algoritmos pr-existente. O mtodo
 * {@link #load(Reader, Map, Map, AlgorithmVersionInfo, ExecutionLocation, String)}
 *  utilizado para gerar configuradores de algoritmos e o mtodo
 * {@link #loadModifications(SimpleAlgorithmConfigurator, Reader)}  utilizado
 * para aplicar modificaes a um configurador de algoritmos pr-existente.
 * </p>
 */
public final class SimpleAlgorithmParser extends ParameterParser {
  /**
   * <p>
   * O elemento {@link #ALGORITHM_ELEMENT}: descreve propriedades do
   * configurador de algoritmo.
   * </p>
   * <p>
   *  o elemento-raz.
   * </p>
   */
  static final String ALGORITHM_ELEMENT = "algoritmo";

  /**
   * O atributo {@value #ALGORITHM_ELEMENT_ABBREVIATION_ATTRIBUTE} do elemento
   * {@link #ALGORITHM_ELEMENT}: define a abreviatura do configurador de
   * algoritmos,  opcional e  do tipo string.
   */
  static final String ALGORITHM_ELEMENT_ABBREVIATION_ATTRIBUTE = "abreviatura";

  /**
   * O atributo {@value #ALGORITHM_ELEMENT_COMMAND_ATTRIBUTE} do elemento
   * {@link #ALGORITHM_ELEMENT}: define o nome do executvel do configurador de
   * algoritmos,  obrigatrio e  do tipo string.
   */
  static final String ALGORITHM_ELEMENT_COMMAND_ATTRIBUTE = "comando";

  /**
   * O atributo {@value #ALGORITHM_ELEMENT_DEFAULT_INPUT_FILE_ATTRIBUTE} do
   * elemento {@link #ALGORITHM_ELEMENT}: define o nome do arquivo de entrada
   * padro do configurador de algoritmos,  opcional e  do tipo string.
   */
  static final String ALGORITHM_ELEMENT_DEFAULT_INPUT_FILE_ATTRIBUTE =
    "arquivo_de_entrada_padrao";

  /**
   * O atributo {@value #ALGORITHM_ELEMENT_EXECUTION_TYPE_ATTRIBUTE} do elemento
   * {@link #ALGORITHM_ELEMENT}: define o {@link ExecutionType tipo de execuo
   * * } do configurador de algoritmos,  opcional, o valor-padro 
   * {@link #ALGORITHM_ELEMENT_EXECUTION_TYPE_VALUE_SIMPLE} e  do tipo string.
   * Os valores possveis para ele so:
   * <ul>
   * <li>{@link #ALGORITHM_ELEMENT_EXECUTION_TYPE_VALUES_MULTIPLE}</li>
   * <li>{@link #ALGORITHM_ELEMENT_EXECUTION_TYPE_VALUE_SIMPLE}</li>
   * </ul>
   */
  static final String ALGORITHM_ELEMENT_EXECUTION_TYPE_ATTRIBUTE =
    "tipo_de_execucao";

  /**
   * O atributo {@value #ALGORITHM_ELEMENT_EXECUTION_LOCATION_ATTRIBUTE} do
   * elemento {@link #ALGORITHM_ELEMENT}: define o {@link ExecutionLocation
   * local de execuo} do configurador de algoritmos,  opcional}.
   */
  static final String ALGORITHM_ELEMENT_EXECUTION_LOCATION_ATTRIBUTE =
    "local_de_execucao";

  /**
   * O valor do atributo {@link #ALGORITHM_ELEMENT_EXECUTION_TYPE_ATTRIBUTE} do
   * elemento {@link #ALGORITHM_ELEMENT} que indicam que o {@link ExecutionType
   * tipo de execuo}  {@link ExecutionType#SIMPLE simples}.
   */
  static final String ALGORITHM_ELEMENT_EXECUTION_TYPE_VALUE_SIMPLE = "simples";

  /**
   * Os valores do atributo {@link #ALGORITHM_ELEMENT_EXECUTION_TYPE_ATTRIBUTE}
   * do elemento {@link #ALGORITHM_ELEMENT} que indicam que o
   * {@link ExecutionType tipo de execuo}  {@link ExecutionType#MULTIPLE
   * mltipla}.
   */
  static final String[] ALGORITHM_ELEMENT_EXECUTION_TYPE_VALUES_MULTIPLE =
    new String[] { "multipla", "mltipla" };

  /**
   * O atributo {@value #ALGORITHM_ELEMENT_PROVIDE_ID_ATTRIBUTE} do elemento
   * {@link #ALGORITHM_ELEMENT}: define se o configurador de algoritmos ir
   * exibir o identificador do comando na linha de comando,  opcional, o
   * valor-padro  {@link #ALGORITHM_ELEMENT_PROVIDE_ID_DEFAULT_VALUE} e  do
   * tipo booleano.
   */
  static final String ALGORITHM_ELEMENT_PROVIDE_ID_ATTRIBUTE =
    "fornecer_identificador";

  /**
   * </p>
   * O valor-padro do atributo {@link #ALGORITHM_ELEMENT_PROVIDE_ID_ATTRIBUTE}
   * do elemento {@link #ALGORITHM_ELEMENT}.
   * </p>
   * <p>
   * O seu valor  {@value #ALGORITHM_ELEMENT_PROVIDE_ID_DEFAULT_VALUE}.
   * </p>
   */
  static final boolean ALGORITHM_ELEMENT_PROVIDE_ID_DEFAULT_VALUE = false;

  /**
   * O atributo {@value #ALGORITHM_ELEMENT_SHOW_OUTPUT_ATTRIBUTE} do elemento
   * {@link #ALGORITHM_ELEMENT}: define se o configurador de algoritmos ir
   * exibir a sada padro e de erro.  opcional, o valor-padro 
   * {@link #ALGORITHM_ELEMENT_SHOW_OUTPUT_DEFAULT_VALUE} e  do tipo booleano.
   */
  static final String ALGORITHM_ELEMENT_SHOW_OUTPUT_ATTRIBUTE = "exibir_saida";

  /**
   * <p>
   * O valor-padro do atributo {@link #ALGORITHM_ELEMENT_SHOW_OUTPUT_ATTRIBUTE}
   * do elemento {@link #ALGORITHM_ELEMENT}.
   * </p>
   * <p>
   * O seu valor  {@value #ALGORITHM_ELEMENT_SHOW_OUTPUT_DEFAULT_VALUE}.
   * </p>
   */
  static final boolean ALGORITHM_ELEMENT_SHOW_OUTPUT_DEFAULT_VALUE = false;

  /**
   * O atributo {@value #ALGORITHM_ELEMENT_SHELL_ATTRIBUTE} do elemento
   * {@link #ALGORITHM_ELEMENT}: define o shell que ser utilizado para
   * interpretar o comando,  opcional e  do tipo string.
   */
  static final String ALGORITHM_ELEMENT_SHELL_ATTRIBUTE = "shell";

  /**
   * O atributo {@value #ALGORITHM_ELEMENT_CAPTURE_EXIT_CODE_ATTRIBUTE} do
   * elemento {@link #ALGORITHM_ELEMENT}: define se o algoritmo prov um cdigo
   * de sada no final de sua execuo,  opcional e  do tipo booleano.
   */
  static final String ALGORITHM_ELEMENT_CAPTURE_EXIT_CODE_ATTRIBUTE =
    "capturar_codigo_de_saida";

  /**
   * <p>
   * O valor-padro do atributo
   * {@link #ALGORITHM_ELEMENT_CAPTURE_EXIT_CODE_ATTRIBUTE} do elemento
   * {@link #ALGORITHM_ELEMENT}.
   * </p>
   * <p>
   * O seu valor  {@value #ALGORITHM_ELEMENT_CAPTURE_EXIT_CODE_DEFAULT_VALUE}.
   * </p>
   */
  static final boolean ALGORITHM_ELEMENT_CAPTURE_EXIT_CODE_DEFAULT_VALUE =
    false;

  /**
   * <p>
   * O elemento {@value #DESCRIPTION_ELEMENT}: define a descrio do
   * configurador de algoritmos.
   * </p>
   * <p>
   *  elemento-filho do elemento {@link #ALGORITHM_ELEMENT}.
   * </p>
   * <p>
   * O seu valor  opcional e  do tipo string.
   * </p>
   */
  static final String DESCRIPTION_ELEMENT = "descricao";
  /**
   * <p>
   * O elemento {@value #GROUP_ELEMENT}: descreve as propriedades de um
   * {@link ParameterGroup parmetro do tipo grupo}.
   * </p>
   * <p>
   *  filho do elemento {@link #ALGORITHM_ELEMENT}.
   * </p>
   */
  static final String GROUP_ELEMENT = "grupo";

  /**
   * <p>
   * O atributo {@value #GROUP_ELEMENT_LABEL_ATTRIBUTE} do elemento
   * {@link #GROUP_ELEMENT}. Indica o rtulo do grupo.  obrigatrio e o seu
   * tipo  string.
   * </p>
   */
  static final String GROUP_ELEMENT_LABEL_ATTRIBUTE = "rotulo";

  /**
   * <p>
   * O atributo {@value #GROUP_ELEMENT_ID_ATTRIBUTE} do elemento
   * {@link #GROUP_ELEMENT}. Indica o nome do grupo.  opcional e o seu tipo 
   * string.
   * </p>
   */
  static final String GROUP_ELEMENT_ID_ATTRIBUTE = "id";

  /**
   * <p>
   * O atributo {@value #GROUP_ELEMENT_ID_ATTRIBUTE} do elemento
   * {@link #GROUP_ELEMENT}. Indica se o grupo  minimizvel.  opcional e  do
   * tipo booleano.
   * </p>
   */
  static final String GROUP_ELEMENT_COLLAPSIBLE_ATTRIBUTE = "minimizavel";

  /**
   * <p>
   * O valor-padro do atributo {@link #GROUP_ELEMENT_COLLAPSIBLE_ATTRIBUTE} do
   * elemento {@link #GROUP_ELEMENT}.
   * </p>
   * <p>
   * O seu valor  {@value #GROUP_ELEMENT_COLLAPSIBLE_ATTRIBUTE_DEFAULT_VALUE}.
   * </p>
   */
  static final boolean GROUP_ELEMENT_COLLAPSIBLE_ATTRIBUTE_DEFAULT_VALUE =
    false;

  /**
   * <p>
   * O elemento {@value #PARAMETER_ELEMENT}: indica que parmetros sero
   * utilizados pelo {@link ParameterLoader carregador de parmetros}.
   * </p>
   * <p>
   *  filho do elemento {@link #ALGORITHM_ELEMENT}.
   * </p>
   */
  static final String PARAMETER_ELEMENT = "parametro";

  /**
   * O formato da linha de comando atual.
   */
  private String currentCommandLinePattern;

  /**
   * <p>
   * Cria um {@link SimpleAlgorithmConfigurator configurador de algoritmos
   * simples} para uma verso de algoritmo.
   * </p>
   * <p>
   * O texto obtido do parmetro {@code configXmlReader}  combinado ao mapa de
   * substituio. O mapa de substituio  utilizado para criar apelidos para
   * trechos de cdigo XML, permitindo a criao de grupos e parmetros
   * personalizados. Utiliza-se as propriedades dos parmetros para obter as
   * configuraes do CSBase e especficas da instncia de CSBase. Tambm
   * utiliza-se configuraes especficas da verso do algoritmo. A combinao
   * delas : primeiro utiliza-se a configuraes do CSBase, em seguida da
   * instncia e, por ltimo, da verso do algoritmo. Pode-se criar elementos
   * novos, atribuir valores a atributos ou modificar os valores dos atributos.
   * </p>
   * 
   * @param configXmlReader O leitor que aponta para os metadados do
   *        configurador de algoritmos (No aceita {@code null}).
   * @param paramConfigProperties As configuraes especficas dos parmetros da
   *        verso do algoritmo (No aceita {@code null}).
   * @param paramProperties A lista de parmetros especficos da verso do
   *        algoritmo.
   * @param algorithmVersion A verso do algoritmo.
   * @param defaultExecutionLocation O local padro de execuo dos comandos.
   * @param defaultCommandLinePattern pattern default de coamndos (vindo do
   *        servio).
   * 
   * @return O configurador de algoritmos.
   * 
   * @throws ParseException Erro no formato do contedo do leitor.
   */
  public SimpleAlgorithmConfigurator load(Reader configXmlReader,
    Map<String, String> paramConfigProperties,
    Map<String, String> paramProperties, AlgorithmVersionInfo algorithmVersion,
    ExecutionLocation defaultExecutionLocation,
    String defaultCommandLinePattern) throws ParseException {
    if (configXmlReader == null) {
      throw new IllegalArgumentException(
        "O parmetro configXmlReader est nulo.");
    }

    if (algorithmVersion == null) {
      throw new IllegalArgumentException(
        "O parmetro algorithmVersion est nulo.");
    }

    if (paramConfigProperties == null) {
      throw new IllegalArgumentException(
        "O parmetro replacementMap est nulo.");
    }

    if (paramProperties == null) {
      throw new IllegalArgumentException(
        "O parmetro paramProperties est nulo.");
    }

    XmlParser parser = new XmlParser(paramConfigProperties);
    parser.parseDocument(configXmlReader);
    parser.goToRoot();
    parser.ensureElementName(ALGORITHM_ELEMENT);
    String command = parser.extractAttributeValue(
      ALGORITHM_ELEMENT_COMMAND_ATTRIBUTE);
    String shell = parser.extractAttributeValue(
      ALGORITHM_ELEMENT_SHELL_ATTRIBUTE, null);
    boolean captureExitCode = parser.extractAttributeValueAsBoolean(
      ALGORITHM_ELEMENT_CAPTURE_EXIT_CODE_ATTRIBUTE,
      ALGORITHM_ELEMENT_CAPTURE_EXIT_CODE_DEFAULT_VALUE);

    String defaultInputFileParameterName = parser.extractAttributeValue(
      ALGORITHM_ELEMENT_DEFAULT_INPUT_FILE_ATTRIBUTE, null);
    boolean provideId = parser.extractAttributeValueAsBoolean(
      ALGORITHM_ELEMENT_PROVIDE_ID_ATTRIBUTE,
      ALGORITHM_ELEMENT_PROVIDE_ID_DEFAULT_VALUE);
    boolean showOutput = parser.extractAttributeValueAsBoolean(
      ALGORITHM_ELEMENT_SHOW_OUTPUT_ATTRIBUTE,
      ALGORITHM_ELEMENT_SHOW_OUTPUT_DEFAULT_VALUE);
    String description = loadDescription(parser);
    ExecutionType executionType = loadExecutionType(parser);

    ExecutionLocation executionLocation = parser
      .extractAttributeValueAsEnumeration(
        ALGORITHM_ELEMENT_EXECUTION_LOCATION_ATTRIBUTE,
        defaultExecutionLocation, new ExecutionLocationConverter());

    String abbreviation = parser.extractAttributeValue(
      ALGORITHM_ELEMENT_ABBREVIATION_ATTRIBUTE, null);
    currentCommandLinePattern = loadCommandLinePattern(parser,
      defaultCommandLinePattern);

    SimpleAlgorithmConfigurator configurator = new SimpleAlgorithmConfigurator(
      algorithmVersion, description, executionType, executionLocation,
      abbreviation, provideId, command, shell, defaultInputFileParameterName,
      showOutput, currentCommandLinePattern);
    try {
      configurator.setHasExitCode(captureExitCode);
      loadGroups(parser, configurator, paramProperties);
      loadExpressions(parser, configurator);
      loadTriggers(parser, configurator);
      loadHiddenParameters(parser, configurator);
      checkDefaultInputFileParameter(defaultInputFileParameterName,
        configurator);
      parser.checkAttributes();
      parser.goToRoot();
    }
    finally {
      currentCommandLinePattern = defaultCommandLinePattern;
    }
    return configurator;
  }

  /**
   * <p>
   * Carrega os gatilhos do configurador de algoritmos.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @throws ParseException Se houver um erro no XML.
   */
  private void loadTriggers(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    TriggerParser triggerParser = new TriggerParser();
    triggerParser.loadTriggers(parser, configurator);
  }

  /**
   * <p>
   * Modifica o configurador de algoritmos de acordo com as instrues do leitor
   * {@code configXmlReader}.
   * </p>
   * 
   * <p>
   * Por enquanto s aceita o parmetro enumerao com seleo mltipla.
   * </p>
   * 
   * @param configurator O configurador de algoritmos que ser modificado. (No
   *        aceita {@code null}).
   * @param configXmlReader O leitor (No aceita {@code null}).
   * 
   * @throws ParseException Se o contedo do leitor estiver em um formato
   *         errado.
   */
  public void loadModifications(SimpleAlgorithmConfigurator configurator,
    Reader configXmlReader) throws ParseException {
    XmlParser parser = new XmlParser(new HashMap<String, String>());
    parser.parseDocument(configXmlReader);
    parser.goToRoot();
    parser.ensureElementName(ALGORITHM_ELEMENT);
    parser.checkAttributes();
    boolean hasElements = parser.goToFirstChild();
    while (hasElements) {
      parser.ensureElementName(PARAMETER_ELEMENT);
      String name = parser.extractAttributeValue(
        SimpleParameterParser.PARAMETER_ELEMENT_NAME_ATTRIBUTE);
      parser.checkAttributes();
      SimpleParameter<?> parameter = configurator.getSimpleParameter(name);
      if (parameter == null) {
        throw new ParseException("O parmetro {0} no foi encontrado.", name);
      }
      if (!(parameter instanceof EnumerationListParameter)) {
        throw new ParseException("O parmetro {0} no  de um tipo suportado. "
          + "O nico tipo de parmetro suportado  o {1} com seleco mltipla.",
          name, EnumerationParameterFactory.ENUMERATION_PARAMETER_ELEMENT);
      }
      EnumerationListParameter enumParameter =
        (EnumerationListParameter) parameter;
      List<EnumerationItem> items = new EnumerationItemFactory()
        .loadEnumerationItems(parser, name);
      enumParameter.setItems(items);
      hasElements = parser.goToNextSibling();
    }
    parser.goToRoot();
  }

  /**
   * Verifica se o parmetro indicado pelo atributo
   * {@link #ALGORITHM_ELEMENT_DEFAULT_INPUT_FILE_ATTRIBUTE} existe.
   * 
   * @param defaultInputFileParameterName O nome do parmetro (No aceita
   *        {@code null}).
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @throws ParseException Se ele no existir.
   */
  private void checkDefaultInputFileParameter(
    String defaultInputFileParameterName,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    if (defaultInputFileParameterName != null) {
      List<?> inputFileParameters = configurator.getInputFileParameters();
      Iterator<?> inputFileParameterIterator = inputFileParameters.iterator();
      while (inputFileParameterIterator.hasNext()) {
        InputFileParameter inputFileParameter =
          (InputFileParameter) inputFileParameterIterator.next();
        if (inputFileParameter.getName().equals(
          defaultInputFileParameterName)) {
          return;
        }
      }
      throw new ParseException(
        "O parmetro {0} utilizado no atributo {1} do elemento {2} no foi "
          + "definido neste algoritmo ({3} - {4}).",
        defaultInputFileParameterName,
        ALGORITHM_ELEMENT_DEFAULT_INPUT_FILE_ATTRIBUTE, ALGORITHM_ELEMENT,
        configurator.getAlgorithmVersion().getInfo(), configurator
          .getAlgorithmVersion().getId());
    }
  }

  /**
   * Verifica se os parmetros do carregador de parmetros foram definidos
   * dentro do seu grupo.
   * 
   * @param group o grupo do carregador.
   * @throws ParseException se um dos parmetros de entrada no estiver definido
   *         dentro do grupo.
   */
  private void checkParameterLoader(ParameterGroup group)
    throws ParseException {
    ParameterLoader parameterLoader = group.getParameterLoader();
    if (parameterLoader != null) {
      for (String inputParameterName : parameterLoader
        .getInputParameterNames()) {
        if (group.getSimpleParameter(inputParameterName) == null) {
          throw new ParseException(
            "O parmetro {0} referenciado pelo elemento {1} no existe.",
            inputParameterName, ParameterLoaderParser.INPUT_PARAMETER_ELEMENT);
        }
      }
    }
  }

  /**
   * Carrega a descrio do configurador de algoritmos.
   * 
   * @param parser O analisador(No aceita {@code null}).
   * 
   * @return A descrio ou {@code null} se ela no existir.
   * 
   * @throws ParseException Erro no XML.
   */
  private String loadDescription(XmlParser parser) throws ParseException {
    if (!parser.goToFirstChild(DESCRIPTION_ELEMENT)) {
      return null;
    }
    String description = parser.getElementValue(null);
    parser.checkAttributes();
    parser.checkChildElements();
    parser.goToParent();
    return description;
  }

  /**
   * <p>
   * Carrega um {@link ExecutionType}.
   * </p>
   * 
   * @param parser O analisador(No aceita {@code null}).
   * 
   * @return O tipo de execuo.
   * 
   * @throws ParseException Em caso de erro no XML.
   */
  private ExecutionType loadExecutionType(XmlParser parser)
    throws ParseException {
    String executionTypeName = parser.extractAttributeValue(
      ALGORITHM_ELEMENT_EXECUTION_TYPE_ATTRIBUTE,
      ALGORITHM_ELEMENT_EXECUTION_TYPE_VALUE_SIMPLE);
    if (executionTypeName.equals(
      ALGORITHM_ELEMENT_EXECUTION_TYPE_VALUE_SIMPLE)) {
      return ExecutionType.SIMPLE;
    }
    for (String value : ALGORITHM_ELEMENT_EXECUTION_TYPE_VALUES_MULTIPLE) {
      if (executionTypeName.equals(value)) {
        return ExecutionType.MULTIPLE;
      }
    }
    throw new ParseException(
      "O valor {0} do atributo {1} do elemento {2} no  "
        + "um tipo de execuo vlido.", new Object[] { executionTypeName,
            ALGORITHM_ELEMENT_EXECUTION_TYPE_ATTRIBUTE, ALGORITHM_ELEMENT });
  }

  /**
   * <p>
   * Carrega as {@link ValidationExpression}.
   * </p>
   * 
   * 
   * @param parser O analisador(No aceita {@code null}).
   * @param configurator O configurador de algoritmo (No aceita {@code null}).
   * 
   * @throws ParseException Em caso de erro no XML.
   */
  private void loadExpressions(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    if (parser.goToFirstChild(ValidationExpressionParser.EXPRESSION_ELEMENT)) {
      ValidationExpressionParser expressionParser =
        new ValidationExpressionParser();
      do {
        expressionParser.loadExpression(parser, configurator);
      } while (parser.goToNextSibling(
        ValidationExpressionParser.EXPRESSION_ELEMENT));
      parser.goToParent();
    }
  }

  /**
   * <p>
   * Carrega um {@link ParameterGroup}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #GROUP_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * @param params Mapa com parmetros no-padro do sistema.
   * 
   * @return O grupo.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private ParameterGroup loadGroup(XmlParser parser,
    SimpleAlgorithmConfigurator configurator, Map<String, String> params)
      throws ParseException {
    String groupLabel = parser.extractAttributeValue(
      GROUP_ELEMENT_LABEL_ATTRIBUTE);
    // Se no for informado um identificador, o rtulo  usado.
    String groupId = parser.extractAttributeValue(GROUP_ELEMENT_ID_ATTRIBUTE,
      groupLabel);

    boolean isCollapsible = parser.extractAttributeValueAsBoolean(
      GROUP_ELEMENT_COLLAPSIBLE_ATTRIBUTE,
      GROUP_ELEMENT_COLLAPSIBLE_ATTRIBUTE_DEFAULT_VALUE);

    ParameterGroup group = new ParameterGroup(groupId, groupLabel,
      isCollapsible);
    if (parser.goToFirstChild()) {
      do {
        Parameter<?> parameter = null;
        String elementName = parser.getElementName();
        if (elementName.equals(GROUP_ELEMENT)) {
          parameter = loadGroup(parser, configurator, params);
        }
        else if (elementName.equals(
          ParameterLoaderParser.PARAMETER_LOADER_ELEMENT)) {
          ParameterLoaderParser parameterLoaderParser =
            new ParameterLoaderParser();
          parameterLoaderParser.loadParameterLoader(parser, group);
        }
        else {
          parameter = loadSimpleParameter(parser, params, group, configurator);
        }
        if (parameter != null) {
          if (!group.addParameter(parameter)) {
            throw new ParseException(
              "J existe um parmetro {0} com este no grupo {1}.",
              new Object[] { parameter, group });
          }
        }
      } while (parser.goToNextSibling());
      parser.goToParent();
    }
    parser.checkAttributes();
    checkParameterLoader(group);

    return group;
  }

  /**
   * Carrega um parmetro que no  padro do sistema.
   * 
   * @param parser Parser xml do configurador.
   * @param customParamsMap Mapa com parmetros no-padro do sistema e suas
   *        fbricas prprias.
   * @param group grupo em criao.
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * @return Uma instncia do parmetro no-padro.
   * @throws ParseException Caso no seja possvel criar o parmetro a partir da
   *         fbrica registrada.
   */
  private Parameter<?> loadSimpleParameter(XmlParser parser,
    Map<String, String> customParamsMap, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    String elementName = parser.getElementName();
    if (customParamsMap.containsKey(elementName)) {
      String factoryClassName = customParamsMap.get(elementName);
      try {
        ParameterFactory factory = createParameterFactory(factoryClassName);
        return factory.createParameter(parser, currentCommandLinePattern, group,
          configurator);
      }
      catch (OperationFailureException e) {
        throw new ParseException(
          "No foi possvel criar o elemento {0} a partir da fbrica {1}. <br> {2}",
          elementName, factoryClassName, e.getMessage());
      }
    }
    else {
      throw new ParseException(
        "No foi possvel criar o elemento {0}, pois este no pertence ao conjunto de parmetros aceitos pelo configurador.",
        elementName);
    }
  }

  /**
   * Cria uma instncia, por reflexo, da fbrica de um tipo de parmetro, a
   * partir do nome completo da classe que a implementa.
   * 
   * @param factoryClassName O nome completo da classe que implementa a fbrica.
   * @return Uma instncia da fbrica.
   * @throws OperationFailureException Caso no seja possvel criar a fbrica.
   */
  private ParameterFactory createParameterFactory(String factoryClassName)
    throws OperationFailureException {
    Class<?> factoryClass;
    try {
      factoryClass = Class.forName(factoryClassName);
    }
    catch (ClassNotFoundException e) {
      throw new OperationFailureException("A classe {0} no foi encontrada.",
        factoryClassName);
    }
    if (!ParameterFactory.class.isAssignableFrom(factoryClass)) {
      throw new OperationFailureException(
        "A classe {0} no implementa a interface necessria {1}.",
        factoryClassName, ParameterFactory.class.getName());
    }
    Constructor<?> factoryConstructor;
    try {
      factoryConstructor = factoryClass.getConstructor();
    }
    catch (NoSuchMethodException e) {
      throw new OperationFailureException(
        "No existe um construtor vazio em {0}.", factoryClassName);
    }
    ParameterFactory factory;
    try {
      factory = (ParameterFactory) factoryConstructor.newInstance();
    }
    catch (InstantiationException e) {
      throw new OperationFailureException(MessageFormat.format(
        "No foi possvel construir a fbrica {0}.", factoryClassName), e);
    }
    catch (IllegalAccessException e) {
      throw new OperationFailureException(MessageFormat.format(
        "Construtor da fbrica {0} no aceita os parmetros usados.",
        factoryClassName), e);
    }
    catch (InvocationTargetException e) {
      throw new OperationFailureException(MessageFormat.format(
        "Erro ao construir a fbrica de parmetros {0}.", factoryClassName), e
          .getTargetException());
    }
    return factory;
  }

  /**
   * <p>
   * Carrega os {@link ParameterGroup grupos}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * @param customParamsMap Tipos de parmetros disponveis.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private void loadGroups(XmlParser parser,
    SimpleAlgorithmConfigurator configurator,
    Map<String, String> customParamsMap) throws ParseException {
    if (parser.goToFirstChild(GROUP_ELEMENT)) {
      do {
        ParameterGroup group = loadGroup(parser, configurator, customParamsMap);
        if (!configurator.addGroup(group)) {
          throw new ParseException(
            "O grupo {0} ou algum de seus parmetros j existe com mesmo " +
              "nome no algoritmo {1}.", group, configurator);
        }
      } while (parser.goToNextSibling(GROUP_ELEMENT));
      parser.goToParent();

    }
  }

  /**
   * Carrega os {@link HiddenParameter}.
   * 
   * @param parser O analisador (No aceita {@code null}).
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private void loadHiddenParameters(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    if (parser.goToFirstChild(HiddenParameterParser.HIDDEN_PARAMETER_ELEMENT)) {
      HiddenParameterParser hiddenParameterParser = new HiddenParameterParser();
      do {
        HiddenParameter hiddenParameter = hiddenParameterParser.createParameter(
          parser, currentCommandLinePattern, null, configurator);
        if (!configurator.addHiddenParameter(hiddenParameter)) {
          throw new ParseException(
            "J existe um grupo {0} cadastrado no algoritmo {1}.",
            new Object[] { hiddenParameter, configurator });
        }
      } while (parser.goToNextSibling(
        HiddenParameterParser.HIDDEN_PARAMETER_ELEMENT));
      parser.goToParent();
    }
  }
}
