/**
 * $Id: SharedObject.java 179359 2017-03-13 19:58:16Z cviana $
 */

package csbase.logic;

import java.io.Serializable;
import java.util.Date;

/**
 * Define um objeto genrico que pode ser armazenado e compartilhado atravs do
 * servio SharedObjectService.
 * TODO explicar melhor o que  um shared-object
 */
public class SharedObject implements Serializable {
  /** Identificador do objeto, composto pelo userId + separador + nome */
  private Object id;
  /**
   * Categoria do shared object.
   */
  private String category;
  /**
   * Dono do shared object.
   */
  private Object ownerUserId;
  /**
   * Nome do shared object.
   */
  private String name;
  /**
   * Usurios que tm acesso de leitura ao shared object.
   */
  private Object[] usersRO;
  /**
   * Usurios que tm acesso de leitura e escrita ao shared object.
   */
  private Object[] usersRW;
  /**
   * Data de criao.
   */
  private Date created;
  /**
   * Data da ltima modificao.
   */
  private Date lastModified;
  /**
   * Contedo.
   */
  private Object contents;
  /**
   * Define o compartilhamento para todos os usurios. Ignora usersRO.
   */
  private boolean global;

  /**
   * Construtor.
   * 
   * @param category categoria do shared object
   * @param ownerUserId ID do dono do shared object
   * @param name nome do shared object
   * @param global {@code true} se o shared object  global
   * @param contents contedo do shared object
   */
  public SharedObject(String category, Object ownerUserId, String name,
    boolean global, Object contents) {
    this(category, ownerUserId, name, global, new Object[0], new Object[0],
      contents);
  }

  /**
   * Construtor.
   * 
   * @param category categoria do shared object
   * @param ownerUserId ID do dono do shared object
   * @param name nome do shared object
   * @param usersRO array com os IDs dos usurios que tm acesso ao shared
   *        object
   * @param contents contedo do shared object
   */
  public SharedObject(String category, Object ownerUserId, String name,
    Object[] usersRO, Object contents) {
    this(category, ownerUserId, name, false, usersRO, new Object[0], contents);
  }

  /**
   * Construtor.
   * 
   * @param category categoria do shared object
   * @param ownerUserId ID do dono do shared object
   * @param name nome do shared object
   * @param global {@code true} se o shared object  global
   * @param usersRO array com os IDs dos usurios que tm acesso de leitura ao
   *        shared object
   * @param usersRW array com os IDs dos usurios que tm acesso de escrita ao
   *        shared object
   * @param contents contedo do shared object
   */
  public SharedObject(String category, Object ownerUserId, String name,
    boolean global, Object[] usersRO, Object[] usersRW, Object contents) {
    this.category = category;
    this.ownerUserId = ownerUserId;
    this.name = name;
    this.global = global;
    this.usersRO = usersRO;
    this.usersRW = usersRW;
    this.contents = contents;
    this.id = getId(ownerUserId, name);
  }

  /**
   * @return identificador do shared object
   */
  public Object getId() {
    return id;
  }

  /**
   * @return array com os identificadores dos usurios que tm acesso de leitura
   *         apenas ao shared object
   */
  public Object[] getUsersRO() {
    return usersRO;
  }

  /**
   * Define os usurios que tm acesso de leitura apenas ao shared object
   * 
   * @param usersRO array com os identificadores dos usurios que tm acesso de
   *        leitura apenas ao shared object
   */
  public void setUsersRO(Object[] usersRO) {
    this.usersRO = usersRO;
  }

  /**
   * @return array com os identificadores dos usurios que tm acesso de leitura
   *         e escrita ao shared object
   */
  public Object[] getUsersRW() {
    return usersRW;
  }

  /**
   * Define os usurios que tm acesso de leitura e escrita ao shared object
   * 
   * @param usersRW array com os identificadores dos usurios que tm acesso de
   *        leitura e escrita ao shared object
   */
  public void setUsersRW(Object[] usersRW) {
    this.usersRW = usersRW;
  }

  /**
   * @return categoria  qual pertence o shared object
   */
  public String getCategory() {
    return category;
  }

  /**
   * @param category define a categoria do shared object
   */
  public void setCategory(String category) {
    this.category = category;
  }

  /**
   * @return contedo do shared object
   */
  public Object getContents() {
    return contents;
  }

  /**
   * Define o contedo do shared object.
   * 
   * @param contents contedo
   */
  public void setContents(Object contents) {
    this.contents = contents;
  }

  /**
   * @return data de criao do shared object
   */
  public Date getCreated() {
    return created;
  }

  /**
   * Define a data de criao do shared object.
   * 
   * @param created a data de criao.
   */
  public void setCreated(Date created) {
    this.created = created;
  }

  /**
   * @return {@code true} se o shared object  global
   */
  public boolean isGlobal() {
    return global;
  }

  /**
   * @return {@code true} se o shared object  global ou compartilhado
   */
  public boolean isShared() {
    if (isGlobal() || (usersRO != null && usersRO.length > 0)
      || (usersRW != null && usersRW.length > 0)) {
      return true;
    }
    return false;
  }

  /**
   * Define se o shared object  global.
   * 
   * @param global {@code true} se o shared object  global
   */
  public void setGlobal(boolean global) {
    this.global = global;
  }

  /**
   * @return data de ltima modificao
   */
  public Date getLastModified() {
    return lastModified;
  }

  /**
   * Define a data da ltima modificao.
   * 
   * @param lastModified data da ltima modificao
   */
  public void setLastModified(Date lastModified) {
    this.lastModified = lastModified;
  }

  /**
   * @return nome do shared object
   */
  public String getName() {
    return name;
  }

  /**
   * Define o nome do shared object.
   * 
   * @param name nome
   */
  public void setName(String name) {
    this.name = name;
  }

  /**
   * @return identificador do dono do shared object
   */
  public Object getOwnerUserId() {
    return ownerUserId;
  }

  /**
   * Define o identificador do dono do shared object.
   * 
   * @param ownerUserId identificador do dono
   */
  public void setOwnerUserId(Object ownerUserId) {
    this.ownerUserId = ownerUserId;
  }

  /**
   * Devolve true se e somente se usurio  o dono da tabela, ou  ADMIN ou se
   * est na lista de usurios com permisso de escrita
   * 
   * @param userId - id do usurio que se quer testar a permisso de escrita
   * @return true se e somente se usurio  o dono da tabela ou se est na lista
   *         de usurios com permisso de escrita
   */
  public boolean userHasAccessRW(Object userId) {
    boolean userCanWrite = false;
    if (getOwnerUserId().equals(userId) || userId.equals(User.getAdminId())) {
      userCanWrite = true;
    }
    else {
      Object[] usersRW = getUsersRW();
      for (Object currUserId : usersRW) {
        if (currUserId.equals(userId)) {
          userCanWrite = true;
          break;
        }
      }
    }
    return userCanWrite;
  }

  /**
   * Devolve todos os ids de usurios que podem ler este shared object,
   * incluindo os de leitura apenas e os de leitura e escrita
   * 
   * @return todos os ids de usurios que podem ler este shared object,
   *         incluindo os de leitura apenas e os de leitura e escrita
   */
  public Object[] getAllUsers() {
    Object[] usersId = new Object[usersRO.length + usersRW.length];
    System.arraycopy(usersRO, 0, usersId, 0, usersRO.length);
    System.arraycopy(usersRW, 0, usersId, usersRO.length, usersRW.length);
    return usersId;
  }

  /**
   * Retorna o identificador de um shared object associado a um determinado
   * usurio.
   * 
   * @param userId identificador do usurio
   * @param name nome do shared object
   * @return identificador
   */
  public static Object getId(Object userId, String name) {
    return userId + "/" + name;
  }

  /**
   * Retorna o identificador do dono de um shared object com um identificador
   * especfico.
   * 
   * @param id identificador do shared object
   * @return identificador do usurio dono
   */
  public static Object getUserId(Object id) {
    return ((String) id).split("/")[0];
  }

  /**
   * Retorna o nome de um shared object com um identificador especfico.
   * 
   * @param id identificador do shared object
   * @return nome do shared object
   */
  public static String getName(Object id) {
    return ((String) id).split("/")[1];
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object other) {
    if (other == this) {
      return true;
    }
    if (other == null) {
      return false;
    }
    if (getClass() != other.getClass()) {
      return false;
    }
    SharedObject shared = (SharedObject) other;
    return ((category == shared.category || (category != null && category
      .equals(shared.category)))
      && (ownerUserId == shared.ownerUserId || (ownerUserId != null && ownerUserId
        .equals(shared.ownerUserId))) && (name == shared.name || (name != null && name
      .equals(shared.name))));
  }

  /**
   * Retorna o nome do objeto, que permite, por exemplo, que uma combo-box seja
   * populada com uma lista de SharedObjects.
   */
  @Override
  public String toString() {
    return name;
  }
}
