/*
 * $Id: XmlParser.java 179359 2017-03-13 19:58:16Z cviana $
 */

package csbase.logic.algorithms.parsers;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parsers.elements.IElementStructure;
import csbase.logic.algorithms.parsers.elements.ParsedElement;
import csbase.logic.algorithms.parsers.elements.attributes.IElementAttribute;
import csbase.logic.algorithms.parsers.elements.attributes.StringToValueConverter;

/**
 * Parser de XML para criao de configuradores de algortimos.
 *
 * @author Tecgraf/PUC-Rio
 */
public final class XmlParser {

  /**
   * Mensagem de erro:
   */
  private final String error_check_attributes =
    "Alguns atributos do elemento {0} no foram processados.\nAtributos:\n{1}";

  /**
   * Mensagem de erro:
   */
  private final String error_check_child_elements =
    "Alguns elementos filhos do elemento {0} no foram processados" + "" +
      ".\nElementos-filhos:\n{1}";

  /**
   * Mensagem de erro:
   */
  private final String error_configuration =
    "Erro nas configuraes do parser.";

  /**
   * Mensagem de erro:
   */
  private final String error_syntax =
    "Erro de sintaxe no arquivo.\nDetalhes: {0}.";

  /**
   * Mensagem de erro:
   */
  private final String error_io = "Erro de IO ao ler o arquivo.";

  /**
   * Mensagem de erro:
   */
  private final String error_wrong_element =
    "Foi encontrado o elemento {0}, porm era esperado o elemento {1}.";

  /**
   * Mensagem de erro:
   */
  private final String error_attribute_not_found =
    "O atributo {1} do elemento {0} no foi encontrado.";

  /**
   * Mensagem de erro:
   */
  private final String error_attribute_not_boolean =
    "O valor {1} do atributo {0} no  do tipo booleano.";

  /**
   * Mensagem de erro:
   */
  private final String error_attribute_not_double =
    "O valor do atributo {1} do elemento {0} deveria ser um nmero real" + "" +
      ".\nValor encontrado: ({2}).";

  /**
   * Mensagem de erro:
   */
  private static final String error_attribute_not_enum =
    "O valor {1} do atributo {0} no  do tipo enumerao.";

  /**
   * Mensagem de erro:
   */
  private final String error_above_maximum =
    "O valor do atributo {1} do elemento {0} est acima do valor mximo " +
      "permitido.\nValor encontrado: ({2}).\nValor mximo: {3}.\n";

  /**
   * Mensagem de erro:
   */
  private final String error_below_minimum =
    "O valor do atributo {1} do elemento {0} est abaixo do valor mnimo " +
      "permitido.\nValor encontrado: ({2}).\nValor mnimo: {3}.\n";

  /**
   * Mensagem de erro:
   */
  private final String error_attribute_not_integer =
    "O valor do atributo {1} do elemento {0} deveria ser um nmero inteiro" +
      ".\nValor encontrado: ({2}).";

  /**
   * Strings aceitas para booleanos falsos
   */
  public static final String[] FALSE_VALUES =
    new String[] { "falso", "false", "F", "nao", "no", "no", "N" };

  /**
   * Strings aceitas para booleanos verdadeiros
   */
  public static final String[] TRUE_VALUES =
    new String[] { "verdadeiro", "true", "V", "T", "sim", "yes", "S", "Y" };

  /**
   * Elemento corrente
   */
  private Element currentElement;
  /**
   * Documento
   */
  private Document document;
  /**
   * Fbrica
   */
  private final DocumentBuilderFactory factory;

  /**
   * Mapas de replacements
   */
  private final Map<String, XmlParser> replacements;

  /**
   * Checagem de atributos restantes,.
   *
   * @throws ParseException em caso de falha de parser
   */
  public void checkAttributes() throws ParseException {
    if (this.currentElement == null) {
      final String err = "O atributo currentElement est nulo.";
      throw new IllegalStateException(err);
    }
    if (this.currentElement.hasAttributes()) {
      final SortedSet<String> attributeValues = new TreeSet<String>();
      final NamedNodeMap nameNodeMap = this.currentElement.getAttributes();
      for (int i = 0; i < nameNodeMap.getLength(); i++) {
        final Attr attribute = (Attr) nameNodeMap.item(i);
        attributeValues.add(attribute.getNodeName());
      }
      final String nodeName = this.currentElement.getNodeName();
      throw new ParseException(error_check_attributes, nodeName,
        attributeValues);
    }
  }

  /**
   * Checagem de elementos filhos.
   *
   * @throws ParseException em caso de falha
   */
  public void checkChildElements() throws ParseException {
    if (this.currentElement == null) {
      final String err = "O atributo currentElement est nulo.";
      throw new IllegalStateException(err);
    }
    final SortedSet<String> elementNames = new TreeSet<String>();
    for (Node childNode =
         this.currentElement.getFirstChild(); childNode != null;
         childNode = childNode.getNextSibling()) {
      if (childNode.getNodeType() == Node.ELEMENT_NODE) {
        elementNames.add(childNode.getNodeName());
      }
    }
    if (!elementNames.isEmpty()) {
      final String nodeName = this.currentElement.getNodeName();
      throw new ParseException(error_check_child_elements, nodeName,
        elementNames);
    }
  }

  /**
   * Cpia de atributos.
   *
   * @param newElement novo elemento.
   * @param oldElement velho elemento
   */
  private void copyAttributes(final Element newElement,
    final Element oldElement) {
    final NamedNodeMap attributes = oldElement.getAttributes();
    for (int i = 0; i < attributes.getLength(); i++) {
      final Attr attribute = (Attr) attributes.item(i);
      final String attributeName = attribute.getNodeName();
      final String attributeValue = attribute.getNodeValue();
      newElement.setAttribute(attributeName, attributeValue);
    }
  }

  /**
   * Cpia de elementos filhos.
   *
   * @param newElement novo elemento.
   * @param oldElement velho elemento
   */
  private void copyChildElements(final Element newElement,
    final Element oldElement) {
    final NodeList childNodes = oldElement.getChildNodes();
    for (int i = 0; i < childNodes.getLength(); i++) {
      final Node oldChildNode = childNodes.item(i);
      if (oldChildNode.getNodeType() == Node.ELEMENT_NODE) {
        final Element oldChildElement = (Element) oldChildNode;
        final Element newChildElement = copyElement(oldChildElement);
        newElement.appendChild(newChildElement);
      }
    }
  }

  /**
   * Cpia de elemento
   *
   * @param element elemento
   * @return cpia.
   */
  private Element copyElement(final Element element) {
    final String nodeName = element.getNodeName();
    final Element newElement = createElement(nodeName);
    copyAttributes(newElement, element);
    copyChildElements(newElement, element);
    return newElement;
  }

  /**
   * Criao de elemento.
   *
   * @param elementName nome
   * @return elemento
   */
  private Element createElement(final String elementName) {
    return this.document.createElement(elementName);
  }

  /**
   * Criao de mapa de replacements.
   *
   * @param replacementMap mapa
   * @throws ParseException em caso de erro.
   */
  private void createReplacements(
    final Map<String, String> replacementMap) throws ParseException {
    final Iterator<String> elementNameIterator =
      replacementMap.keySet().iterator();
    while (elementNameIterator.hasNext()) {
      final String elementName = elementNameIterator.next();
      final String elementText = replacementMap.get(elementName);
      this.replacements.put(elementName, createXmlParser(elementText));
    }
  }

  /**
   * Criaa de parser.
   *
   * @param elementText texto
   * @return parser
   * @throws ParseException em caso de falha.
   */
  private XmlParser createXmlParser(
    final String elementText) throws ParseException {
    final XmlParser parser = new XmlParser();
    parser.parseDocument(new StringReader(elementText));
    return parser;
  }

  /**
   * Checagem de nome de elemento.
   *
   * @param expectedElementName nome esperado
   * @throws ParseException em caso de falha.
   */
  public void ensureElementName(
    final String expectedElementName) throws ParseException {
    if (expectedElementName == null) {
      final String err = "O parmetro expectedElementName est nulo.";
      throw new IllegalArgumentException(err);
    }
    if (this.currentElement == null) {
      final String err = "O atributo currentElement est nulo.";
      throw new IllegalStateException(err);
    }
    final String currentElementName = this.currentElement.getNodeName();
    if (!currentElementName.equals(expectedElementName)) {
      throw new ParseException(error_wrong_element, currentElementName,
        expectedElementName);
    }
  }

  /**
   * Extrao de valor
   *
   * @param attributeName nome do atributo
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public String extractAttributeValue(
    final String attributeName) throws ParseException {
    final String attributeValue = getAttributeValue(attributeName);
    removeAttribute(attributeName);
    return attributeValue;
  }

  /**
   * Extrao de valor
   *
   * @param attributeName nome do atributo
   * @param defaultValue valor default
   * @return valor
   */
  public String extractAttributeValue(final String attributeName,
    final String defaultValue) {
    final String attributeValue =
      getAttributeValue(attributeName, defaultValue);
    removeAttribute(attributeName);
    return attributeValue;
  }

  /**
   * Extrao de valor em forma de lista
   *
   * @param attributeName nome do atributo
   * @param defaultValue valor default
   * @return valor
   */
  public String[] extractAttributeValueAsArray(final String attributeName,
    final String[] defaultValue) {
    final String[] attributeValue =
      getAttributeValueAsArray(attributeName, defaultValue);
    removeAttribute(attributeName);
    return attributeValue;
  }

  /**
   * Extrao de valor
   *
   * @param attributeName nome do atributo
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public boolean extractAttributeValueAsBoolean(
    final String attributeName) throws ParseException {
    final boolean value = getAttributeValueAsBoolean(attributeName);
    removeAttribute(attributeName);
    return value;
  }

  /**
   * Extrao de valor
   *
   * @param attributeName nome do atributo
   * @param defaultValue valor default
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public boolean extractAttributeValueAsBoolean(final String attributeName,
    final boolean defaultValue) throws ParseException {
    final boolean attributeValue =
      getAttributeValueAsBoolean(attributeName, defaultValue);
    removeAttribute(attributeName);
    return attributeValue;
  }

  /**
   * Extrao de valor
   *
   * @param attributeName nome do atributo
   * @param defaultValue valor default
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public Double extractAttributeValueAsDouble(final String attributeName,
    final Double defaultValue) throws ParseException {
    final Double value = getAttributeValueAsDouble(attributeName, defaultValue);
    removeAttribute(attributeName);
    return value;
  }

  /**
   * Extrao de valor
   *
   * @param attributeName nome do atributo
   * @param defaultValue valor default
   * @param minimumValue valor mnimo
   * @param maximumValue valor mximo
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public Double extractAttributeValueAsDouble(final String attributeName,
    final Double defaultValue, final Double maximumValue,
    final Double minimumValue) throws ParseException {
    final Double value =
      getAttributeValueAsDouble(attributeName, defaultValue, maximumValue,
        minimumValue);
    removeAttribute(attributeName);
    return value;
  }

  /**
   * Extrao de valor
   *
   * @param attributeName nome do atributo
   * @param defaultValue valor default
   * @param converter conversor.
   * @param <E> enumerao.
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public <E extends Enum<E>> E extractAttributeValueAsEnumeration(
    final String attributeName, final E defaultValue,
    final StringToEnumConverter<E> converter) throws ParseException {
    final E attributeValue =
      getAttributeValueAsEnumeration(attributeName, defaultValue, converter);
    removeAttribute(attributeName);
    return attributeValue;
  }

  /**
   * Extrao de valor
   *
   * @param attributeName nome do atributo
   * @param converter conversor.
   * @param <E> enumerao.
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public <E extends Enum<E>> E extractAttributeValueAsEnumeration(
    final String attributeName,
    final StringToEnumConverter<E> converter) throws ParseException {
    final E attributeValue =
      getAttributeValueAsEnumeration(attributeName, converter);
    removeAttribute(attributeName);
    return attributeValue;
  }

  /**
   * Extrao de valor
   *
   * @param attributeName nome do atributo
   * @param defaultValue valor default
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public Integer extractAttributeValueAsInteger(final String attributeName,
    final Integer defaultValue) throws ParseException {
    final Integer value =
      getAttributeValueAsInteger(attributeName, defaultValue);
    removeAttribute(attributeName);
    return value;
  }

  /**
   * Extrao de valor
   *
   * @param attributeName nome do atributo
   * @param minimum valor mnimo
   * @param maximum valor mximo
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public Integer extractAttributeValueAsInteger(final String attributeName,
    final Integer maximum, final Integer minimum) throws ParseException {
    final Integer value =
      getAttributeValueAsInteger(attributeName, maximum, minimum);
    removeAttribute(attributeName);
    return value;
  }

  /**
   * Extrao de valor
   *
   * @param attributeName nome do atributo
   * @param defaultValue valor default
   * @param minimumValue valor mnimo
   * @param maximumValue valor mximo
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public Integer extractAttributeValueAsInteger(final String attributeName,
    final Integer defaultValue, final Integer maximumValue,
    final Integer minimumValue) throws ParseException {
    final Integer value =
      getAttributeValueAsInteger(attributeName, defaultValue, maximumValue,
        minimumValue);
    removeAttribute(attributeName);
    return value;
  }

  /**
   * Contagem do nmero de atributos.
   *
   * @return nmero de atributos.
   */
  public int getAttributeCount() {
    final NamedNodeMap attributes = this.currentElement.getAttributes();
    return attributes.getLength();
  }

  /**
   * Busca de valor
   *
   * @param attributeName nome do atributo
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public String getAttributeValue(
    final String attributeName) throws ParseException {
    if (attributeName == null) {
      final String err = "O parmetro attributeName est nulo.";
      throw new IllegalArgumentException(err);
    }
    if (this.currentElement == null) {
      final String err = "O atributo currentElement est nulo.";
      throw new IllegalStateException(err);
    }
    final String attributeValue = getAttributeValue(attributeName, null);
    if (attributeValue == null) {
      final String nodeName = this.currentElement.getNodeName();
      throw new ParseException(error_attribute_not_found, nodeName,
        attributeName);
    }
    return attributeValue;
  }

  /**
   * Busca de valor
   *
   * @param attributeName nome do atributo
   * @param defaultValue valor default
   * @return valor
   */
  public String getAttributeValue(final String attributeName,
    final String defaultValue) {
    if (this.currentElement == null) {
      final String err = "O atributo currentElement est nulo.";
      throw new IllegalStateException(err);
    }
    final Attr attribute = this.currentElement.getAttributeNode(attributeName);
    if (attribute == null) {
      return defaultValue;
    }
    return attribute.getNodeValue();
  }

  /**
   * Busca de valor como lista
   *
   * @param attributeName nome do atributo
   * @param defaultValue valor default
   * @return valor
   */
  public String[] getAttributeValueAsArray(final String attributeName,
    final String[] defaultValue) {
    if (this.currentElement == null) {
      final String err = "O atributo currentElement est nulo.";
      throw new IllegalStateException(err);
    }
    final Attr attribute = this.currentElement.getAttributeNode(attributeName);
    if (attribute == null) {
      return defaultValue;
    }
    String nodeValue = attribute.getNodeValue();
    String[] valueList = nodeValue.trim().split("\\s*,\\s*");
    return valueList;
  }

  /**
   * Busca de valor
   *
   * @param attributeName nome do atributo
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public boolean getAttributeValueAsBoolean(
    final String attributeName) throws ParseException {
    final String attributeValue = getAttributeValue(attributeName);
    for (final String trueValue : XmlParser.TRUE_VALUES) {
      if (trueValue.equalsIgnoreCase(attributeValue)) {
        return true;
      }
    }
    for (final String falseValue : XmlParser.FALSE_VALUES) {
      if (falseValue.equalsIgnoreCase(attributeValue)) {
        return false;
      }
    }
    throw new ParseException(error_attribute_not_boolean, attributeName,
      attributeValue);
  }

  /**
   * Busca de valor
   *
   * @param attributeName nome do atributo
   * @param defaultValue valor default
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public boolean getAttributeValueAsBoolean(final String attributeName,
    final boolean defaultValue) throws ParseException {
    final String attributeValue =
      getAttributeValue(attributeName, Boolean.toString(defaultValue));
    if (attributeValue == null) {
      return defaultValue;
    }
    for (final String trueValue : XmlParser.TRUE_VALUES) {
      if (trueValue.equalsIgnoreCase(attributeValue)) {
        return true;
      }
    }
    for (final String falseValue : XmlParser.FALSE_VALUES) {
      if (falseValue.equalsIgnoreCase(attributeValue)) {
        return false;
      }
    }
    throw new ParseException(error_attribute_not_boolean, attributeName,
      attributeValue);
  }

  /**
   * Busca de valor
   *
   * @param attributeName nome do atributo
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public Double getAttributeValueAsDouble(
    final String attributeName) throws ParseException {
    final Double doubleValue = getAttributeValueAsDouble(attributeName, null);
    if (doubleValue == null) {
      final String nodeName = this.currentElement.getNodeName();
      throw new ParseException(error_attribute_not_found, nodeName,
        attributeName);
    }
    return doubleValue;
  }

  /**
   * Busca de valor
   *
   * @param attributeName nome do atributo
   * @param defaultValue valor default
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public Double getAttributeValueAsDouble(final String attributeName,
    final Double defaultValue) throws ParseException {
    final String textValue =
      (defaultValue == null ? null : defaultValue.toString());
    final String attributeValue = getAttributeValue(attributeName, textValue);
    if (attributeValue == null) {
      return null;
    }
    Double doubleValue;
    try {
      doubleValue = new Double(attributeValue);
    }
    catch (final NumberFormatException e) {
      final String nodeName = this.currentElement.getNodeName();
      throw new ParseException(e, error_attribute_not_double, nodeName,
        attributeName, attributeValue);
    }
    return doubleValue;
  }

  /**
   * Busca de valor
   *
   * @param attributeName nome do atributo
   * @param minimumValue valor mnimo
   * @param maximumValue valor mximo
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public Double getAttributeValueAsDouble(final String attributeName,
    final Double maximumValue,
    final Double minimumValue) throws ParseException {
    final Double value =
      getAttributeValueAsDouble(attributeName, null, maximumValue,
        minimumValue);
    if (value == null) {
      final String nodeName = this.currentElement.getNodeName();
      throw new ParseException(error_attribute_not_found, nodeName,
        attributeName);
    }
    return value;
  }

  /**
   * Busca de valor
   *
   * @param attributeName nome do atributo
   * @param defaultValue valor default
   * @param minimumValue valor mnimo
   * @param maximumValue valor mximo
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public Double getAttributeValueAsDouble(final String attributeName,
    final Double defaultValue, final Double maximumValue,
    final Double minimumValue) throws ParseException {
    final Double doubleValue =
      getAttributeValueAsDouble(attributeName, defaultValue);
    if (doubleValue == null) {
      return null;
    }

    if (maximumValue != null) {
      if (doubleValue.doubleValue() > maximumValue.doubleValue()) {
        final String nodeName = this.currentElement.getNodeName();
        throw new ParseException(error_above_maximum, nodeName, attributeName,
          doubleValue, maximumValue);
      }
    }
    if (minimumValue != null) {
      if (doubleValue.doubleValue() < minimumValue.doubleValue()) {
        final String nodeName = this.currentElement.getNodeName();
        throw new ParseException(error_below_minimum, nodeName, attributeName,
          doubleValue, minimumValue);
      }
    }
    return doubleValue;
  }

  /**
   * Busca de valor
   *
   * @param attributeName nome do atributo
   * @param defaultValue valor default
   * @param converter conversor.
   * @param <E> enumerao.
   * @return valor
   * @throws ParseException em caso de erro.
   */
  public <E extends Enum<E>> E getAttributeValueAsEnumeration(
    final String attributeName, final E defaultValue,
    final StringToEnumConverter<E> converter) throws ParseException {
    final String textValue =
      (defaultValue == null ? null : defaultValue.toString());
    final String attributeValue = getAttributeValue(attributeName, textValue);
    if (attributeValue == null) {
      return defaultValue;
    }
    final E enumValue = converter.valueOf(attributeValue);
    if (enumValue != null) {
      return enumValue;
    }

    throw new ParseException(XmlParser.error_attribute_not_enum, attributeName,
      attributeValue);
  }

  /**
   * Busca de valor
   *
   * @param attributeName nome do atributo
   * @param converter conversor.
   * @param <E> enumerao.
   * @return valor
   * @throws ParseException em caso de erro.
   */
  public <E extends Enum<E>> E getAttributeValueAsEnumeration(
    final String attributeName,
    final StringToEnumConverter<E> converter) throws ParseException {
    final String attributeValue = getAttributeValue(attributeName);
    final E enumValue = converter.valueOf(attributeValue);
    if (enumValue != null) {
      return enumValue;
    }
    throw new ParseException(XmlParser.error_attribute_not_enum, attributeName,
      attributeValue);
  }

  /**
   * Busca de valor
   *
   * @param attributeName nome do atributo
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public Integer getAttributeValueAsInteger(
    final String attributeName) throws ParseException {
    final Integer integerValue =
      getAttributeValueAsInteger(attributeName, null);
    if (integerValue == null) {
      final String nodeName = this.currentElement.getNodeName();
      throw new ParseException(error_attribute_not_found, nodeName,
        attributeName);
    }
    return integerValue;
  }

  /**
   * Busca de valor
   *
   * @param attributeName nome do atributo
   * @param defaultValue valor default
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public Integer getAttributeValueAsInteger(final String attributeName,
    final Integer defaultValue) throws ParseException {
    final String textValue =
      (defaultValue == null ? null : defaultValue.toString());
    final String attributeValue = getAttributeValue(attributeName, textValue);
    if (attributeValue == null) {
      return null;
    }
    Integer integerValue;
    try {
      integerValue = new Integer(attributeValue);
    }
    catch (final NumberFormatException e) {
      final String nodeName = this.currentElement.getNodeName();
      throw new ParseException(e, error_attribute_not_integer, nodeName,
        attributeName, attributeValue);
    }
    return integerValue;
  }

  /**
   * Busca de valor
   *
   * @param attributeName nome do atributo
   * @param minimumValue valor mnimo
   * @param maximumValue valor mximo
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public Integer getAttributeValueAsInteger(final String attributeName,
    final Integer maximumValue,
    final Integer minimumValue) throws ParseException {
    final Integer value =
      getAttributeValueAsInteger(attributeName, null, maximumValue,
        minimumValue);
    if (value == null) {
      final String nodeName = this.currentElement.getNodeName();
      throw new ParseException(error_attribute_not_found, nodeName,
        attributeName);
    }
    return value;
  }

  /**
   * Busca de valor
   *
   * @param attributeName nome do atributo
   * @param defaultValue valor default
   * @param minimumValue valor mnimo
   * @param maximumValue valor mximo
   * @return valor
   * @throws ParseException em caso de falha de parser
   */
  public Integer getAttributeValueAsInteger(final String attributeName,
    final Integer defaultValue, final Integer maximumValue,
    final Integer minimumValue) throws ParseException {
    final Integer integerValue =
      getAttributeValueAsInteger(attributeName, defaultValue);
    if (integerValue == null) {
      return null;
    }

    if (maximumValue != null) {
      if (integerValue.intValue() > maximumValue.intValue()) {
        final String nodeName = this.currentElement.getNodeName();
        throw new ParseException(error_above_maximum, nodeName, attributeName,
          integerValue, maximumValue);
      }
    }
    if (minimumValue != null) {
      if (integerValue.intValue() < minimumValue.intValue()) {
        final String nodeName = this.currentElement.getNodeName();
        throw new ParseException(error_below_minimum, nodeName, attributeName,
          integerValue, minimumValue);
      }
    }
    return integerValue;
  }

  /**
   * Busca do nome o elemento corrente.
   *
   * @return o nome
   */
  public String getElementName() {
    if (this.currentElement == null) {
      final String err = "O atributo currentElement est nulo.";
      throw new IllegalStateException(err);
    }
    return this.currentElement.getNodeName();
  }

  /**
   * Busca do valor corrente.
   *
   * @param defaultValue valor default
   * @return o valor
   */
  public String getElementValue(final String defaultValue) {
    if (this.currentElement == null) {
      final String err = "O atributo currentElement est nulo.";
      throw new IllegalStateException(err);
    }
    String content = "";
    final NodeList childNodes = this.currentElement.getChildNodes();
    for (int i = 0; i < childNodes.getLength(); i++) {
      final Node child = childNodes.item(i);
      if (child.getNodeType() == Node.TEXT_NODE) {
        content += child.getNodeValue().trim();
      }
    }
    content = content.trim();
    if (content.length() == 0) {
      return defaultValue;
    }
    return content;
  }

  /**
   * Ida para primeiro filho de elemento.
   *
   * @return indicativo de sucesso.
   */
  public boolean goToFirstChild() {
    if (this.currentElement == null) {
      final String err = "O atributo currentElement est nulo.";
      throw new IllegalStateException(err);
    }
    final NodeList childNodes = this.currentElement.getChildNodes();
    for (int i = 0; i < childNodes.getLength(); i++) {
      final Node childNode = childNodes.item(i);
      if (childNode.getNodeType() == Node.ELEMENT_NODE) {
        this.currentElement = (Element) childNode;
        return true;
      }
    }
    return false;
  }

  /**
   * Ida para primeiro filho de elemento com nome
   *
   * @param elementName o nome.
   * @return indicativo de sucesso.
   */
  public boolean goToFirstChild(final String elementName) {
    if (elementName == null) {
      final String err = "O parmetro elementName est nulo.";
      throw new IllegalArgumentException(err);
    }
    if (this.currentElement == null) {
      final String err = "O atributo currentElement est nulo.";
      throw new IllegalStateException(err);
    }
    final NodeList childNodes = this.currentElement.getChildNodes();
    for (int i = 0; i < childNodes.getLength(); i++) {
      final Node childNode = childNodes.item(i);
      if ((childNode.getNodeType() == Node.ELEMENT_NODE) && elementName
        .equals(childNode.getNodeName())) {
        this.currentElement = (Element) childNode;
        return true;
      }
    }
    return false;
  }

  /**
   * Ajuste de corrente para prximo.
   *
   * @return indicativo de sucesso.
   */
  public boolean goToNextSibling() {
    if (this.currentElement == null) {
      final String err = "O atributo currentElement est nulo.";
      throw new IllegalStateException(err);
    }
    for (Node nextSibling =
         this.currentElement.getNextSibling(); nextSibling != null;
         nextSibling = nextSibling.getNextSibling()) {
      if (nextSibling.getNodeType() == Node.ELEMENT_NODE) {
        this.currentElement = (Element) nextSibling;
        return true;
      }
    }
    return false;
  }

  /**
   * Ajuste de corrente para prximo.
   *
   * @param elementName nome do elemento.
   * @return indicativo de sucesso.
   */
  public boolean goToNextSibling(final String elementName) {
    if (elementName == null) {
      final String err = "O parmetro elementName est nulo.";
      throw new IllegalArgumentException(err);
    }
    if (this.currentElement == null) {
      final String err = "O atributo currentElement est nulo.";
      throw new IllegalStateException(err);
    }
    for (Node siblingNode =
         this.currentElement.getNextSibling(); siblingNode != null;
         siblingNode = siblingNode.getNextSibling()) {
      if ((siblingNode.getNodeType() == Node.ELEMENT_NODE) && elementName
        .equals(siblingNode.getNodeName())) {
        this.currentElement = (Element) siblingNode;
        return true;
      }
    }
    return false;
  }

  /**
   * Ajuste de corrente para pai.
   *
   * @return indicativo de sucesso.
   */
  public boolean goToParent() {
    if (this.currentElement == null) {
      final String err = "O atributo currentElement est nulo.";
      throw new IllegalStateException(err);
    }
    final Node parentNode = this.currentElement.getParentNode();
    if (parentNode == null) {
      return false;
    }
    if (parentNode.getNodeType() != Node.ELEMENT_NODE) {
      return false;
    }
    this.currentElement = (Element) parentNode;
    return true;
  }

  /**
   * Ajuste de corrente para raiz.
   */
  public void goToRoot() {
    if (this.document == null) {
      final String err = "O atributo document est nulo.";
      throw new IllegalStateException(err);
    }
    this.currentElement = this.document.getDocumentElement();
  }

  /**
   * Indica a exitncia de um atributo.
   *
   * @param attributeName nome do atributo
   * @return indicativo
   */
  public boolean hasAttributeValue(final String attributeName) {
    final String value = getAttributeValue(attributeName, null);
    return value != null;
  }

  /**
   * Move de filhos de elemento.
   *
   * @param newElement novo
   * @param oldElement velho
   */
  private void moveChildElements(final Element newElement,
    final Element oldElement) {
    final NodeList children = oldElement.getChildNodes();
    while (children.getLength() != 0) {
      final Node childNode = children.item(0);
      oldElement.removeChild(childNode);
      newElement.appendChild(childNode);
    }
  }

  /**
   * Mtodo de parser.
   *
   * @param reader leitor
   * @throws ParseException em caso de erro.
   */
  public void parseDocument(final Reader reader) throws ParseException {
    if (reader == null) {
      final String err = "O parmetro reader est nulo.";
      throw new IllegalArgumentException(err);
    }
    try {
      final DocumentBuilder builder = this.factory.newDocumentBuilder();
      this.document = builder.parse(new InputSource(reader));
    }
    catch (final ParserConfigurationException e) {
      throw new ParseException(e, error_configuration);
    }
    catch (final SAXException e) {
      Exception ex = e.getException();
      if (ex == null) {
        ex = e;
      }
      throw new ParseException(e, error_syntax, ex.getLocalizedMessage());
    }
    catch (final IOException e) {
      throw new ParseException(e, error_io);
    }
    while (replaceElements()) {
    }
  }

  /**
   * Remoo de atributo
   *
   * @param attributeName nome do atributo.
   */
  private void removeAttribute(final String attributeName) {
    this.currentElement.removeAttribute(attributeName);
  }

  /**
   * Remoo de elemento corrente.
   *
   * @return indicativo
   */
  private boolean replaceCurrentElement() {
    boolean wasReplaced = false;
    final XmlParser parser = this.replacements.get(getElementName());
    if (parser != null) {
      parser.goToRoot();
      final Element newElement =
        createElement(parser.currentElement.getNodeName());
      copyAttributes(newElement, parser.currentElement);
      moveChildElements(newElement, this.currentElement);
      copyChildElements(newElement, parser.currentElement);
      copyAttributes(newElement, this.currentElement);
      replaceElement(newElement, this.currentElement);
      this.currentElement = newElement;
      wasReplaced = true;
    }
    if (goToFirstChild()) {
      if (replaceCurrentElement()) {
        wasReplaced = true;
      }
      goToParent();
    }
    if (goToNextSibling()) {
      if (replaceCurrentElement()) {
        wasReplaced = true;
      }
    }
    return wasReplaced;
  }

  /**
   * Troca de elemento
   *
   * @param newElement novo
   * @param oldElement antigo
   */
  private void replaceElement(final Element newElement,
    final Element oldElement) {
    final Node parentNode = oldElement.getParentNode();
    parentNode.replaceChild(newElement, oldElement);
  }

  /**
   * Troca de elementos.
   *
   * @return indicativo
   */
  private boolean replaceElements() {
    goToRoot();
    final boolean wasReplaced = replaceCurrentElement();
    this.currentElement = null;
    return wasReplaced;
  }

  /**
   * Extrai um elemento do elemento corrente.
   *
   * @param attribute o atributo.
   * @param <E> o tipo do valor do atributo.
   * @return o valor do atributo.
   * @throws ParseException em caso de erro de leitura.
   */
  public <E> E extractAttribute(
    IElementAttribute<E> attribute) throws ParseException {
    E value = getAttribute(attribute);
    removeAttribute(attribute.getName());
    return value;
  }

  /**
   * Extrai os atributos do elemento.
   *
   * @param structure a estrutura do parmetro.
   * @return o mapa de atributos do parmetro.
   * @throws ParseException em caso de erro na leitura.
   */
  public Map<IElementAttribute<?>, Object> extractAttributes(
    IElementStructure<?> structure) throws ParseException {
    Map<IElementAttribute<?>, Object> values = new HashMap<>();
    for (IElementAttribute<?> attribute : structure.getAttributes()) {
      values.put(attribute, extractAttribute(attribute));
    }
    checkAttributes();
    return values;
  }

  /**
   * Extrai os elementos filhos do elemento corrente.
   *
   * @param structures a lista de estruturas dos elementos filhos.
   * @return um mapa com as estruturas e os elementos correspondentes
   * encontrados.
   * @throws ParseException em caso de erro de leitura.
   */
  public Map<IElementStructure<?>, List<ParsedElement>>
  extractChildren(
    List<IElementStructure<?>> structures) throws ParseException {
    Map<IElementStructure<?>, List<ParsedElement>> children =
      new HashMap<>();
    for (IElementStructure<?> childElement : structures) {
      List<ParsedElement> childElements = new ArrayList<>();
      if (goToFirstChild(childElement.getName())) {
        do {
          String elementValue = getElementValue(null);
          Map<IElementAttribute<?>, Object> attributeValues =
            extractAttributes(childElement);
          ParsedElement ParsedElement =
            new ParsedElement(attributeValues, elementValue);
          childElements.add(ParsedElement);
          children.put(childElement, childElements);
          checkAttributes();
          checkChildElements();
        } while (goToNextSibling(childElement.getName()));
        goToParent();
      }
    }
    return children;
  }

  /**
   * Busca de valor de um atriburo.
   *
   * @param attribute o atributo;
   * @return valor do atributo.
   * @throws ParseException em caso de falha de leitura.
   */
  public <E> E getAttribute(
    IElementAttribute<E> attribute) throws ParseException {
    String value = getValue(attribute.getName());
    if (value == null) {
      if (attribute.isOptional()) {
        IElementAttribute<E> defaultValueParameter =
          attribute.getDefaultValueAttribute();
        if (defaultValueParameter == null) {
          return attribute.getDefaultValue();
        }
        value = getAttributeValue(defaultValueParameter.getName());
      }
      else {
        final String nodeName = this.currentElement.getNodeName();
        String error_attribute_not_found =
          "O atributo {1} do elemento {0} no foi encontrado.";
        throw new ParseException(error_attribute_not_found, nodeName,
          attribute.getName());
      }
    }
    StringToValueConverter<E> valueConverter = attribute.getValueConverter();
    return valueConverter.valueOf(value);
  }

  /**
   * Busca de valor
   *
   * @param attributeName nome do atributo
   * @return valor
   */
  private String getValue(final String attributeName) {
    if (this.currentElement == null) {
      final String err = "O atributo currentElement est nulo.";
      throw new IllegalStateException(err);
    }
    final Attr attribute = this.currentElement.getAttributeNode(attributeName);
    if (attribute == null) {
      return null;
    }
    return attribute.getNodeValue();
  }


  /**
   * Construtor
   */
  XmlParser() {
    this.factory = DocumentBuilderFactory.newInstance();
    this.factory.setCoalescing(true);
    this.factory.setIgnoringComments(true);
    this.factory.setIgnoringElementContentWhitespace(true);
    this.factory.setExpandEntityReferences(false);
    this.replacements = new HashMap<String, XmlParser>();
  }

  /**
   * Construtor
   *
   * @param replacements mapa
   * @throws ParseException em caso de falha de parser
   */
  public XmlParser(
    final Map<String, String> replacements) throws ParseException {
    this();
    createReplacements(replacements);
  }


}
