package csbase.logic.algorithms;

import java.util.List;

import csbase.logic.ProjectFileType;

/**
 * Representa o script de execuo de um comando.
 *
 * @author isabella
 */
public class CommandScript {

  /**
   * Tipo ({@link ProjectFileType}) do arquivo de script.
   */
  private static final String SCRIPT_FILE_TYPE = "SCRIPT";

  /**
   * Nome padro do arquivo de script.
   */
  private static final String DEFAULT_SCRIPT_FILE_NAME = "script";

  /**
   * Contedo do script.
   */
  private String scriptContent;

  /**
   * Nome do arquivo de script.
   */
  private String fileName;

  /**
   * Caminho para o diretrio de persisncia do comando.
   */
  private String persistencePath;

  /**
   * Construtor.
   *
   * @param context Contexto para construo da linha de comando do script.
   * @param scriptContent Contedo do script.
   */
  public CommandScript(CommandLineContext context, String scriptContent) {
    this(context, scriptContent, null);
  }

  /**
   * Construtor.
   *
   * @param context Contexto para construo da linha de comando do script.
   * @param scriptContent Contedo do script.
   * @param fileName Nome do arquivo de script.
   */
  public CommandScript(CommandLineContext context, String scriptContent,
    String fileName) {
    setScriptContent(context, scriptContent);
    if (fileName == null) {
      this.fileName = CommandScript.makeFileName(null);
    }
    else {
      this.fileName = fileName;
    }
    this.persistencePath =
      context.getPersistenceDirectory() + context.getFileSeparator();
  }

  /**
   * Atribui o contedo do script.
   *
   * @param context O contexto para gerao da linha de comando.
   *
   * @param scriptContent O contedo do script.
   */
  protected void setScriptContent(CommandLineContext context,
    String scriptContent) {
    CommandLineBuilder content = new CommandLineBuilder(true);
    content.appendScriptHeader(context);
    content.append(scriptContent);
    this.scriptContent = content.toString();
  }

  /**
   * Retorna o contedo do script.
   *
   * @return scriptContent O contedo do script.
   */
  public String getScriptContent() {
    return scriptContent;
  }

  /**
   * Retorna o nome do arquivo de script.
   *
   * @return fileName O nome do arquivo.
   */
  public String getFileName() {
    return fileName;
  }

  /**
   * Retorna o caminho para o diretrio de persisncia do comando.
   *
   * @return persistencePath O caminho.
   */
  protected String getPersistencePath() {
    return persistencePath;
  }

  /**
   * Monta a linha de comando para a execuo <b>do script</b>.
   *
   * @return commandLine A linha de comando.
   */
  public String makeCommandLine() {
    return CommandLineBuilder.SHELL + " " + this.persistencePath
      + this.fileName;
  }

  /**
   * Retorna o nome do arquivo de script, com o sufixo especificado.
   *
   * @param suffix O sufixo para o nome do arquivo.
   * @return O nome do arquivo de script.
   */
  public static String makeFileName(String suffix) {
    StringBuilder name = new StringBuilder();
    name.append(DEFAULT_SCRIPT_FILE_NAME);
    if (suffix != null) {
      name.append(suffix);
    }
    ProjectFileType fileType = ProjectFileType.getFileType(SCRIPT_FILE_TYPE);
    if (fileType != null) {
      List<String> scriptFileExtensions = fileType.getExtensions();
      if (!scriptFileExtensions.isEmpty()) {
        String extension = scriptFileExtensions.get(0).trim();
        if (!extension.isEmpty()) {
          name.append(".");
          name.append(extension);
        }
      }
    }
    return name.toString();
  }

}
