/*
 * CommandInfo.java $author$ $date$ $revision$
 */
package csbase.logic;

import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;
import csbase.logic.algorithms.ExecutionType;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.remote.ClientRemoteLocator;

/**
 * Representa as informas bsicas aos comandos.
 *
 * @author Bruno Oliveira Silvestre
 */
public class CommandInfo implements Serializable {

  /** Configurador do algoritmo. No  transiente para poder ir pro cliente. */
  private AlgorithmConfigurator configurator;

  /** Descrio para o comando. */
  private String description;
  /**
   * Posio global do comando na fila.  usado apenas para leitura, quando um
   * cliente pede todos os seus comandos. No  usado para controle.
   */
  private int globalPosition;

  /**
   * Tentativas de submisso do comando.
   */
  private int submissionAttempts = 0;

  /** Identificador de comando (nico) para efeito de log e auditagem. */
  private String id;
  /** Indica se um e-mail de aviso ser enviado ao trmino do comando. */
  private boolean mailAtEnd;
  /**
   * Array com endereos que devem receber um e-mail de aviso ao trmino do
   * comando.
   */
  private String[] emailList;

  /** Nmero de execues que sero disparadas na execuo mltipla. */
  private Integer executionCountForMultipleExecution;

  /**
   * O nmero de execues por sga que ser utilizado na execuo mltipla
   * manual.
   */
  private Integer executionCountPerSGAForMultipleExecution;

  /*
   * Comeo dos dados utilizados para a monitorao da execuo do comando.
   */
  /** O total de CPU utilizada por todos os processos do comando. */
  private Double cpuPerc;
  /** O total de tempo de CPU utilizado por todos os processos do comando. */
  private Double cpuTimeSec;
  /** O total de memria RAM utilizada por todos os processos do comando. */
  private Double ramMemoryMB;
  /** O percentual de memria RAM utilizada por todos os processos do comando. */
  private Double ramMemoryPerc;
  /** O total de memria Swap utilizada por todos os processos do comando. */
  private Double swapMemoryMB;
  /** O total de memria Swap utilizada por todos os processos do comando. */
  private Double swapMemoryPerc;
  /** As informaes especficas do comando. */
  private Map<String, String> specificData;
  /** O tempo total de execuo do processo principal. */
  private Integer wallTimeSec;
  /** Uma descrio dos ns que executam o comando. */
  private String executionNodeDescription;
  /** Tempo de execuo do processo no nvel de usurio */
  private Double userTimeSec;
  /** Tempo de execuo do processo no nvel do kernel */
  private Double systemTimeSec;
  /** Quantidade de memria virtual ocupada pelo processo */
  private Double virtualMemorySizeMB;
  /** Quantidade de bytes recebidos na interface de rede */
  private Double bytesInKB;
  /** Quantidade de bytes enviados pela interface de rede */
  private Double bytesOutKB;
  /** Quantidade de bytes lidos no disco */
  private Double diskBytesReadKB;
  /** Quantidade de bytes escritos no disco */
  private Double diskBytesWriteKB;

  /*
   * Fim dos dados utilizados para a monitorao da execuo do comando.
   */

  /** Prioridade do comando. */
  private Priority priority;
  /** Identificador do projeto. */
  private Object projectId;
  /** Nome do SGA que executa o comando. */
  private String sgaName;
  /** Servidores selecionados pelo usurio para a execuo do comando. */
  private List<String> selectedSGAsNames;
  /** Estado do comando. */
  private CommandStatus status;
  /**
   * Informaes sobre a finalizao do comando.
   */
  private CommandFinalizationInfo finalizationInfo;
  /** Indica a data e hora que o comando foi submetido para a fila de execuo */
  private Date submittedDate;
  /** Descrio do algoritmo. */
  private String tip;
  /** Identificador do usurio que executou o comando */
  private Object userId;
  /** Filtro de plataformas para a execuo do comando. */
  private String platformFilter;
  /** Indica se a escolha de servidores  automtica. */
  private boolean isAutomatic;
  /**
   * Define o tipo de execuo: {@link ExecutionType#SIMPLE} ou
   * {@link ExecutionType#MULTIPLE}.
   */
  private ExecutionType executionType;
  /** Indica se as informaes dinmicas do comando esto vlidas. */
  private boolean isValid;
  /**
   * Indica se o comando est esperando execuo na fila de um escalonador
   * externo ao CSBase (atualmente, esse atributo s  usado no caso do PBS).
   * Essa flag _no_ se refere  fila do Scheduler). Para saber se o comando
   * est enfileirado no Scheduler, verificar se o status do comando 
   * {@link CommandStatus#SCHEDULED} . Essa flag deve ser utilizada somente por
   * comandos com status {@link CommandStatus#EXECUTING} (que j foram
   * disparados pelo SGA), para que se diferencie os que realmente esto rodando
   * dos que esto parados na fila de um escalonador sobre o qual o CSBase no
   * tem controle.
   */
  private boolean isQueued;

  private String partialLog = "";

  /**
   * Informaes de progresso cacheadas para uso na tabela de monitorao.
   * {@link #getProgressData()}
   */
  private transient ProgressData cachedProgressData;

  /**
   * Caminho para o diretrio onde est persistido o comando.
   */
  private String[] persistencyPath;

  /**
   * Utilizado para verificar se o estado corrente do comando  antes de
   * alter-lo.
   */
  private Lock statusLock = new ReentrantLock();
  
  /**
   * Nome da mquina que solicitou a execuo do algoritmo.
   */
  private String clientHostName;

  /**
   * Mapa de propriedades extra
   */
  //TODO: avaliar se  necessrio fazer cpia do mapa
  private Map<String, String> extraPropertiesMap;

  /**
   * Cria um comando obtido a partir dos dados de persistncia.
   *
   * @param id Identificador de comando (nico) para efeito de log e auditagem.
   * @param globalPosition Posio global do comando na fila.
   * @param mailAtEnd Indica se um e-mail de aviso ser enviado ao trmino do
   *        comando.
   * @param projectId Identificador do projeto no qual o comando foi submetido.
   * @param submittedDate Data de submisso do comando.
   * @param userId Usurio que submeteu o comando.
   * @param priority Prioridade atual do comando.
   * @param status Estado atual do comando.
   * @param finalizationInfo Informaes sobre a finalizao do comando.
   * @param tip Dica sobre a descrio do algoritmo.
   */
  public CommandInfo(String id, int globalPosition, boolean mailAtEnd,
    Object projectId, Date submittedDate, Object userId, Priority priority,
    CommandStatus status, CommandFinalizationInfo finalizationInfo, String tip) {
    if (null == finalizationInfo) {
      try {
        throw new IllegalArgumentException("finalizationType");
      }
      catch (Exception e) {
        e.printStackTrace();
      }
    }
    this.id = id;
    this.projectId = projectId;
    this.globalPosition = globalPosition;
    this.mailAtEnd = mailAtEnd;
    this.emailList = null;
    this.submittedDate = submittedDate;
    this.userId = userId;
    this.priority = priority;
    setStatus(status, finalizationInfo);
    this.selectedSGAsNames = new ArrayList<String>();
    this.tip = tip;
    this.isValid = true;
    this.isQueued = false;
    configureSimpleExecution();
  }

  /**
   * Construtor de um ComandInfo.
   *
   * @param id Identificador de comando (nico).
   * @param configurator Configurador do algoritmo. (No pode ser {@code null}.)
   * @param submission Dados da submisso do comando.
   * @param userId Identificador do usurio que submeteu o comando.
   * @param extraPropertiesMap O mapa de propriedades extras
   */
  public CommandInfo(String id, AlgorithmConfigurator configurator,
    CommandSubmission submission, String userId,
    Map<String, String> extraPropertiesMap) {
	this.clientHostName = submission.getClientHostName();
    this.configurator = configurator;
    this.projectId = submission.getProjectId();
    this.id = id;
    this.description = submission.getDescription();
    this.priority = submission.getPriority();
    this.mailAtEnd = submission.isMailAtEnd();
    this.emailList = submission.getEmailList();
    this.submittedDate = submission.getSubmittedDate();
    this.executionCountForMultipleExecution =
      submission.getExecutionCountForMultipleExecution();
    this.executionCountPerSGAForMultipleExecution =
      submission.getExecutionCountPerSGAForMultipleExecution();
    this.isAutomatic = submission.isAutomatic();
    this.platformFilter = submission.getPlatform();
    this.selectedSGAsNames = new ArrayList<String>(submission.getSGANames());
    this.executionType = submission.getExecutionType();
    this.tip = configurator.toString();
    this.userId = userId;
    setStatus(CommandStatus.SCHEDULED, new SimpleCommandFinalizationInfo(
      CommandFinalizationType.NOT_FINISHED, false));
    this.isValid = true;
    this.isQueued = false;

    //TODO: avaliar se  necessrio fazer cpia do mapa
    this.extraPropertiesMap = extraPropertiesMap;
  }

  /**
   * Invalida os dados dinmicos do comando. Deve ser usado quando a conexo com
   * o servidor cai e os dados no so mais vlidos.
   */
  public void invalidateDynamicFields() {
    cpuPerc = null;
    cpuTimeSec = null;
    ramMemoryPerc = null;
    swapMemoryPerc = null;
    wallTimeSec = null;
    executionNodeDescription = null;
    specificData = null;
    userTimeSec = null;
    systemTimeSec = null;
    virtualMemorySizeMB = null;
    bytesInKB = null;
    bytesOutKB = null;
    diskBytesReadKB = null;
    diskBytesWriteKB = null;
    isValid = false;
    isQueued = false;
  }

  /**
   * Indica se o comando est preso na fila de um escalonador externo ao CSBase.
   *
   * @return verdadeiro se o comando est na fila do escalonador externo ou
   *         falso, caso contrrio.
   * @see #isQueued
   */
  public boolean isQueued() {
    return isQueued;
  }

  /**
   * Determina se o comando est preso na fila de um escalonador externo ao
   * CSBase.
   *
   * @param isQueued verdadeiro se o comando est na fila do escalonador externo
   *        ou falso, caso contrrio.
   * @see #isQueued
   */
  public void setQueued(boolean isQueued) {
    this.isQueued = isQueued;
  }

  /**
   * Vlida os dados dinmicos do comando.
   */
  public void validateDynamicFields() {
    isValid = true;
  }

  /*
   * Comeo dos mtodos utilizados para a monitorao da execuo do comando.
   */
  /**
   * Obtm o total de CPU utilizada por todos os processos do comando.
   *
   * @return o total de CPU utilizada por todos os processos do comando.
   */
  public Double getCpuPerc() {
    return cpuPerc;
  }

  /**
   * Define o total de CPU utilizada por todos os processos do comando.
   *
   * @param cpuPerc o total de CPU utilizada por todos os processos do comando.
   */
  public void setCpuPerc(Double cpuPerc) {
    this.cpuPerc = cpuPerc;
  }

  /**
   * Obtm o tempo total de CPU, em segundos, utilizado por todos os processos
   * do comando.
   *
   * @return o tempo total de CPU, em segundos, utilizadas por todos os
   *         processos do comando.
   */
  public Double getCpuTimeSec() {
    return cpuTimeSec;
  }

  /**
   * Define o tempo total de CPU, em segundos, utilizado por todos os processos
   * do comando.
   *
   * @param cpuTimeSec o tempo total de CPU, em segundos, utilizado por todos os
   *        processos do comando.
   */
  public void setCpuTimeSec(Double cpuTimeSec) {
    this.cpuTimeSec = cpuTimeSec;
  }

  /**
   * Obtm o total de memria RAM utilizada por todos os processos do comando.
   *
   * @return o total de memria RAM utilizada por todos os processos do comando.
   */
  public Double getRAMMemoryMB() {
    return ramMemoryMB;
  }

  /**
   * Define o total de memria RAM utilizada por todos os processos do comando.
   *
   * @param ramMemoryMB o total de memria RAM utilizada por todos os processos
   *        do comando.
   */
  public void setRAMMemoryMB(Double ramMemoryMB) {
    this.ramMemoryMB = ramMemoryMB;
  }

  /**
   * Obtm o percentual de memria RAM utilizada por todos os processos do
   * comando.
   *
   * @return o percentual de memria RAM utilizada por todos os processos do
   *         comando.
   */
  public Double getRAMMemoryPerc() {
    return ramMemoryPerc;
  }

  /**
   * Define o percentual de memria RAM utilizada por todos os processos do
   * comando.
   *
   * @param ramMemoryPerc o total de memria RAM utilizada por todos os
   *        processos do comando.
   */
  public void setRAMMemoryPerc(Double ramMemoryPerc) {
    this.ramMemoryPerc = ramMemoryPerc;
  }

  /**
   * Obtm o total de memria Swap utilizada por todos os processos do comando.
   *
   * @return o total de memria Swap utilizada por todos os processos do
   *         comando.
   */
  public Double getSwapMemoryMB() {
    return swapMemoryMB;
  }

  /**
   * Define o total de memria Swap utilizada por todos os processos do comando.
   *
   * @param swapMemoryMB o total de memria Swap utilizada por todos os
   *        processos do comando.
   */
  public void setSwapMemoryMB(Double swapMemoryMB) {
    this.swapMemoryMB = swapMemoryMB;
  }

  /**
   * Obtm o total de memria Swap utilizada por todos os processos do comando.
   *
   * @return o total de memria Swap utilizada por todos os processos do
   *         comando.
   */
  public Double getSwapMemoryPerc() {
    return swapMemoryPerc;
  }

  /**
   * Define o total de memria Swap utilizada por todos os processos do comando.
   *
   * @param swapMemoryPerc o total de memria Swap utilizada por todos os
   *        processos do comando.
   */
  public void setSwapMemoryPerc(Double swapMemoryPerc) {
    this.swapMemoryPerc = swapMemoryPerc;
  }

  /**
   * Obtm as informaes especficas do comando.
   *
   * @return as informaes especficas do comando.
   */
  public Map<String, String> getSpecificData() {
    return specificData;
  }

  /**
   * Define as informaes especficas do comando.
   *
   * @param specificData as informaes especficas do comando.
   */
  public void setSpecificData(Map<String, String> specificData) {
    this.specificData = specificData;
    this.cachedProgressData = null;
  }

  /**
   * Obtm o tempo total de execuo do processo principal.
   *
   * @return o tempo total de execuo do processo principal.
   */
  public Integer getWallTimeSec() {
    return wallTimeSec;
  }

  /**
   * Define o valor do tempo total de execuo do processo principal.
   *
   * @param wallTimeSec o valor do tempo total de execuo do processo
   *        principal.
   */
  public void setWallTimeSec(Integer wallTimeSec) {
    this.wallTimeSec = wallTimeSec;
  }

  /**
   * Otm o tempo total de execuo no nvel do kernel para todos os processos
   * do comando.
   *
   * @return o tempo de execuo no nvel do kernel para todos os processos do
   *         comando.
   */
  public Double getSystemTimeSec() {
    return systemTimeSec;
  }

  /**
   * Define o valor de tempo de execuo no nvel do kernel para todos os
   * processos do comando.
   *
   * @param systemTimeSec o valor do tempo de execuo no nvel do kernel para
   *        todos os processos do comando.
   */
  public void setSystemTimeSec(Double systemTimeSec) {
    this.systemTimeSec = systemTimeSec;
  }

  /**
   * Otm a quantidade de memria virtual usada por todos os processos do
   * comando.
   *
   * @return a quantidade de memria virtual usada por todos os processos do
   *         comando.
   */
  public Double getVirtualMemorySizeMB() {
    return virtualMemorySizeMB;
  }

  /**
   * Define o valor da quantidade de memria virtual usada por todos os
   * processos do comando.
   *
   * @param virtualMemorySizeMB o valor da quantidade de memria virtual usada
   *        por todos os processos do comando.
   */
  public void setVirtualMemorySizeMB(Double virtualMemorySizeMB) {
    this.virtualMemorySizeMB = virtualMemorySizeMB;
  }

  /**
   * Otm a quantidade de bytes recebidos na interface de rede.
   *
   * @return a quantidade de bytes recebidos na interface de rede.
   */
  public Double getBytesInKB() {
    return bytesInKB;
  }

  /**
   * Define a quantidade de bytes recebidos na interface de rede.
   *
   * @param bytesInKB a quantidade de bytes recebidos na interface de rede.
   */
  public void setBytesInKB(Double bytesInKB) {
    this.bytesInKB = bytesInKB;
  }

  /**
   * Otm a quantidade de bytes enviados pela interface de rede.
   *
   * @return a quantidade de bytes enviados pela interface de rede.
   */
  public Double getBytesOutKB() {
    return bytesOutKB;
  }

  /**
   * Define a quantidade de bytes enviados pela interface de rede.
   *
   * @param bytesOutKB a quantidade de bytes enviados pela interface de rede.
   */
  public void setBytesOutKB(Double bytesOutKB) {
    this.bytesOutKB = bytesOutKB;
  }

  /**
   * Otm a quantidade de bytes lidos do disco.
   *
   * @return a quantidade de bytes lidos do disco.
   */
  public Double getDiskBytesReadKB() {
    return diskBytesReadKB;
  }

  /**
   * Define a quantidade de bytes lidos do disco.
   *
   * @param diskBytesReadKB a quantidade de bytes lidos do disco.
   */
  public void setDiskBytesReadKB(Double diskBytesReadKB) {
    this.diskBytesReadKB = diskBytesReadKB;
  }

  /**
   * Otm a quantidade de bytes escritos em disco.
   *
   * @return a quantidade de bytes escritos em disco.
   */
  public Double getDiskBytesWriteKB() {
    return diskBytesWriteKB;
  }

  /**
   * Define a quantidade de bytes escritos em disco.
   *
   * @param diskBytesWriteKB a quantidade de bytes escritos em disco.
   */
  public void setDiskBytesWriteKB(Double diskBytesWriteKB) {
    this.diskBytesWriteKB = diskBytesWriteKB;
  }

  /**
   * Define o valor de tempo de execuo no nvel do usurio para o processo
   * principal.
   *
   * @param userTimeSec o valor de tempo de execuo no nvel do usurio para o
   *        processo principal.
   */
  public void setUserTimeSec(Double userTimeSec) {
    this.userTimeSec = userTimeSec;
  }

  /**
   * Obtm o valor de tempo de execuo no nvel do usurio para o processo
   * principal.
   *
   * @return o valor de tempo de execuo no nvel do usurio para o processo
   *         principal.
   */
  public Double getUserTimeSec() {
    return this.userTimeSec;
  }

  /**
   * Obtm uma descrio dos ns que executam o comando.
   *
   * @return uma descrio dos ns que executam o comando.
   */
  public String getExecutionNodeDescription() {
    return executionNodeDescription;
  }

  /**
   * Define uma descrio dos ns que executam o comando.
   *
   * @param executionNodeDescription uma descrio dos ns que executam o
   *        comando.
   */
  public void setExecutionNodeDescription(String executionNodeDescription) {
    this.executionNodeDescription = executionNodeDescription;
  }

  /*
   * Fim dos mtodos utilizados para a monitorao da execuo do comando.
   */

  /**
   * Atribui o identificador do comando.
   *
   * @param id O identificador do comando.
   */
  public void setId(String id) {
    this.id = id;
  }

  /**
   * Determina se um e-mail de aviso ser enviado ao trmino do comando.
   *
   * @param mailAtEnd Verdadeiro se um e-mail de aviso ser enviado ao trmino
   *        do comando ou falso, caso contrrio.
   */
  public void setMailAtEnd(boolean mailAtEnd) {
    this.mailAtEnd = mailAtEnd;
  }

  /**
   * Atribui o nmero de execues que sero disparadas na execuo mltipla.
   *
   * @param executionCountForMultipleExecution O nmero de execues.
   */
  public void setExecutionCountForMultipleExecution(
    Integer executionCountForMultipleExecution) {
    this.executionCountForMultipleExecution =
      executionCountForMultipleExecution;
  }

  /**
   * Atribui o nmero de execues que sero disparadas por SGA na execuo
   * mltipla.
   *
   * @param executionCountPerSGAForMultipleExecution O nmero de execues por
   *        SGA.
   */
  public void setExecutionCountPerSGAForMultipleExecution(
    Integer executionCountPerSGAForMultipleExecution) {
    this.executionCountPerSGAForMultipleExecution =
      executionCountPerSGAForMultipleExecution;
  }

  /**
   * Atribui o identificador do projeto.
   *
   * @param projectId O identificador do projeto.
   */
  public void setProjectId(Object projectId) {
    this.projectId = projectId;
  }

  /**
   * Atribui a data e hora que o comando foi submetido para a fila de execuo.
   *
   * @param submittedDate A data e hora que o comando foi submetido.
   */
  public void setSubmittedDate(Date submittedDate) {
    this.submittedDate = submittedDate;
  }

  /**
   * Atribui o identificador do usurio que executou o comando
   *
   * @param userId O identificador do usurio.
   */
  public void setUserId(Object userId) {
    this.userId = userId;
  }

  /**
   * Determina se a escolha de servidores  automtica.
   *
   * @param isAutomatic verdadeiro se a escolha de servidores  automtica ou
   *        falso, caso sontrrio.
   */
  public void setAutomatic(boolean isAutomatic) {
    this.isAutomatic = isAutomatic;
  }

  /**
   * Atribui o tipo de execuo: {@link ExecutionType#SIMPLE} ou
   * {@link ExecutionType#MULTIPLE}.
   *
   * @param executionType O tipo de execuo.
   */
  public void setExecutionType(ExecutionType executionType) {
    this.executionType = executionType;
  }

  /**
   * Configura os dados da execuo mltipla.
   *
   * @param executionCount Nmero de execues que sero disparadas.
   */
  public void configureMultipleExecution(int executionCount) {
    executionType = ExecutionType.MULTIPLE;
    isAutomatic = true;
    executionCountForMultipleExecution = executionCount;
  }

  /**
   * Configura os dados da execuo mltipla.
   *
   * @param sgaNames Nomes dos sgas selecionados pelo usurio.
   * @param executionCountPerSGA Nmero de execues por servidor.
   */
  public void configureMultipleExecution(List<String> sgaNames,
    int executionCountPerSGA) {
    executionType = ExecutionType.MULTIPLE;
    isAutomatic = false;
    setSelectedSGAsNames(sgaNames);
    executionCountPerSGAForMultipleExecution = executionCountPerSGA;
    executionCountForMultipleExecution =
      executionCountPerSGA * selectedSGAsNames.size();
  }

  /**
   * Configura os dados da execuo simples.
   */
  public void configureSimpleExecution() {
    executionType = ExecutionType.SIMPLE;
    isAutomatic = true;
  }

  /**
   * Configura os dados da execuo simples.
   *
   * @param sgaName Nome do sgas selecionado pelo usurio.
   */
  public void configureSimpleExecution(String sgaName) {
    executionType = ExecutionType.SIMPLE;
    isAutomatic = false;
    setSelectedSGAName(sgaName);
  }

  /**
   * Mtodo para comparar objetos da mesma classe.
   *
   * @param obj Objeto a ser comparado.
   * @return Verdadeiro se ambos forem iguais.
   */
  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof CommandInfo)) {
      return false;
    }

    if (id == null) {
      return false;
    }

    CommandInfo other = (CommandInfo) obj;
    return id.equals(other.id);
  }

  /**
   * Obtm o configurador do algoritmo.
   *
   * @return .
   * @throws RemoteException falha de rmi
   */
  public AlgorithmConfigurator getConfigurator() throws RemoteException {
    if (configurator == null) {
      // Obtm o configurador no servio de persistncia.
      configurator =
        ClientRemoteLocator.commandPersistenceService
        .readAlgorithmConfigurator(getProjectId(), getId());
    }
    return configurator;
  }

  /**
   * Obtm a descrio do comando.
   *
   * @return .
   */
  public String getDescription() {
    return description;
  }

  /**
   * Obtm o tipo da execuo do comando {@link ExecutionType}.
   *
   * @return .
   */
  public ExecutionType getExecutionType() {
    return executionType;
  }

  /**
   * Obtm a posio do comando na fila do escalonador.
   *
   * @return .
   */
  public int getGlobalPosition() {
    return globalPosition;
  }

  /**
   * Obtm o identificador do comando.
   *
   * @return .
   */
  public String getId() {
    return id;
  }

  /**
   * Obtm a prioridade do comando.
   *
   * @return .
   */
  public Priority getPriority() {
    return priority;
  }

  /**
   * Obtm o identificador do projeto no qual o comando foi submetido.
   *
   * @return .
   */
  public Object getProjectId() {
    return projectId;
  }

  /**
   * Retorna o nome do SGA que executa o comando. Pode ser {@code null} quando o
   * comando ainda est na fila do escalonador e  automtico.
   *
   * @return .
   */
  public String getSGAName() {
    return sgaName;
  }

  /**
   * Retorna os nomes dos SGAs que foram selecionados pelo usurio.
   *
   * @return sgas Os nomes dos SGAs que foram selecionados pelo usurio.
   */
  public List<String> getSelectedSGAsNames() {
    return Collections.unmodifiableList(selectedSGAsNames);
  }

  /**
   * Define quais os SGAs que sero usados neste comando.
   *
   * @param selectedSGAsNames lista de SGAs a serem utilizados.
   */
  public void setSelectedSGAsNames(List<String> selectedSGAsNames) {
    this.selectedSGAsNames.clear();
    this.selectedSGAsNames.addAll(selectedSGAsNames);
  }

  /**
   * Obtm o estado do comando.
   *
   * @return .
   */
  public CommandStatus getStatus() {
    return status;
  }

  /**
   * Obtm o que fez com que esse comando fosse para o estado
   * {@link CommandStatus#FINISHED} ou
   * {@link CommandFinalizationType#NOT_FINISHED} caso este comando ainda no
   * tenha terminado.
   *
   * @return o que fez com que esse comando fosse para o estado
   *         {@link CommandStatus#FINISHED} ou
   *         {@link CommandFinalizationType#NOT_FINISHED} caso este comando
   *         ainda no tenha terminado.
   */
  public CommandFinalizationType getFinalizationType() {
    if (null == finalizationInfo
      || null == finalizationInfo.getFinalizationType()) {
      return CommandStatus.FINISHED == this.status ? CommandFinalizationType.UNKNOWN
        : CommandFinalizationType.NOT_FINISHED;
    }
    return finalizationInfo.getFinalizationType();
  }

  /**
   * Indica se o comando gerou log de alertas durante sua execuo.
   *
   * @return verdadeiro caso o comando tenha gerado alertas ou falso, caso
   * contrrio.
   */
  public boolean hasWarnings() {
    if (null == finalizationInfo || null == finalizationInfo
      .getFinalizationType() || finalizationInfo
      .getFinalizationType() == CommandFinalizationType.NOT_FINISHED) {
      return false;
    }
    return finalizationInfo.hasWarnings();
  }

  /**
   * Obtm a data em que o comando foi submetido.
   *
   * @return .
   */
  public Date getSubmittedDate() {
    return submittedDate;
  }

  /**
   * Obtm o nmero de execues que sero disparadas na execuo mltipla.
   *
   * @return nmero de execues
   */
  public Integer getExecutionCountForMultipleExecution() {
    return executionCountForMultipleExecution;
  }

  /**
   * Obtm o nmero de execues por sga que ser utilizado na execuo mltipla
   * manual.
   *
   * @return .
   */
  public Integer getExecutionCountPerSGAForMultipleExecution() {
    return executionCountPerSGAForMultipleExecution;
  }

  /**
   * Obtm o identificador do usurio.
   *
   * @return .
   */
  public Object getUserId() {
    return userId;
  }

  /**
   * Calcula o cdigo hash do objeto.
   *
   * @return Inteiro que representa o cdigo hash do objeto.
   */
  @Override
  public int hashCode() {
    if (id == null) {
      return "CommandInfoWhitoutId".hashCode();
    }
    return id.hashCode();
  }

  /**
   * Indica se o sistema deve enviar um email para o usurio que submeteu o
   * comando quando este terminar.
   *
   * @return true se deve enviar o email e false caso contrrio.
   */
  public boolean isMailAtEnd() {
    return mailAtEnd;
  }

  /**
   * Obtm a lista de endereos que devem receber email de aviso de trmino de
   * execuo do comando. Os emails somente so enviados se estiver configurado
   * para envio de emails.
   *
   * @see #setMailAtEnd(boolean)
   * @return o array com endereos de email
   */
  public String[] getEmailList() {
    return emailList;
  }

  /**
   * Mtodo criado para auxiliar nos testes.
   *
   * @return texto Texto com informaes do comando.
   */
  public String print() {
    StringBuffer buffer = new StringBuffer();
    buffer.append("id: " + id + "\n");
    buffer.append("tip: " + getTip() + "\n");
    buffer.append("description: " + description + "\n");
    buffer.append("sgaName: " + sgaName + "\n");
    buffer.append("userId: " + userId + "\n");
    return buffer.toString();
  }

  /**
   * Altera a posio do comando na fila do escalonador.
   *
   * @param globalPosition a posio do comando na fila do escalonador.
   */
  public void setGlobalPosition(int globalPosition) {
    this.globalPosition = globalPosition;
  }

  /**
   * Altera a prioridade do comando.
   *
   * @param priority a prioridade do comando.
   */
  public void setPriority(Priority priority) {
    this.priority = priority;
  }

  /**
   * Altera o nome do SGA para qual o comando ser submetido.
   *
   * @param sgaName Nome do SGA.
   */
  public void setSGAName(String sgaName) {
    this.sgaName = sgaName;
  }

  /**
   * Altera o estado do comando.<br>
   * Se este mtodo for utilizado para atribuir o estado
   * {@link CommandStatus#FINISHED terminado} ao comando, ele ir alterar o seu
   * {@link CommandFinalizationType} para
   * {@link CommandFinalizationType#UNKNOWN}. Para esta finalidade,  prefervel
   * que se utilize o mtodo {@link #setFinished(CommandFinalizationInfo)}.
   *
   * @param status O estado do comando {@link CommandStatus}.
   * @return <tt>False</tt> indicando que a alterao no foi feita por que o
   *         comando est no estado terminal - {@link CommandStatus#FINISHED} -.
   * @see #setFinished(CommandFinalizationInfo)
   */
  public boolean setStatus(CommandStatus status) {
    if (CommandStatus.FINISHED == status) {
      return setStatus(status, new SimpleCommandFinalizationInfo(
        CommandFinalizationType.UNKNOWN, false));
    }
    return setStatus(status, new SimpleCommandFinalizationInfo(
      CommandFinalizationType.NOT_FINISHED, false));
  }

  /**
   * Altera o estado do comando para {@link CommandStatus#FINISHED terminado} e
   * atribui um valor ao {@link CommandFinalizationType modo em que este comando
   * foi finalizado}.
   *
   * @param info Informaes sobre a finalizao do comando.
   */
  public void setFinished(CommandFinalizationInfo info) {
    setStatus(CommandStatus.FINISHED, info);
  }

  /**
   * Indica se as informaes dinmicas do comando esto vlidas.
   *
   * @return verdadeiro se as informaes dinmicas do comando esto vlidas e
   *         falso caso contrrio.
   */
  public boolean isValid() {
    return isValid;
  }

  /**
   * Converte para uma string.
   *
   * @return o nome da mquina e o nome do comando.
   */
  @Override
  public String toString() {
    return id + " ( " + getTip() + " )";
  }

  /**
   * Altera o estado e a indicao de se esse comando foi terminado e por que.<br>
   * O {@code finalizationType} dever ser
   * {@link CommandFinalizationType#NOT_FINISHED} exclusivamente no caso do
   * {@code status} ser {@link CommandStatus#FINISHED}.
   *
   * @param status O estado do comando.
   * @param info Informaes sobre a finalizao do comando.
   * @return <tt>False</tt> indicando que a alterao no foi feita por que o
   *         comando est no estado terminal - {@link CommandStatus#FINISHED} -.
   */
  private boolean setStatus(CommandStatus status, CommandFinalizationInfo info) {
    if (null == status) {
      throw new IllegalArgumentException(String.format(
        "Atribuindo estado nulo ao comando %s.", id));
    }
    if (null == info) {
      throw new IllegalArgumentException(String.format(
        "Atribuindo estado de trmino nulo ao comando %s.", id));
    }
    CommandFinalizationType finalizationType = info.getFinalizationType();
    if ((CommandStatus.FINISHED == status && CommandFinalizationType.NOT_FINISHED == finalizationType)
      || (CommandStatus.FINISHED != status
      && CommandStatus.SYSTEM_FAILURE != status && CommandFinalizationType.NOT_FINISHED != finalizationType)) {
      throw new IllegalArgumentException(
        String
        .format(
          "Atribuindo estado inconsistente ao comando %s. CommandStatus: %s e CommandFinalisationType: %s",
          id, status, finalizationType));
    }

    statusLock.lock();
    try {
      // O estado FINISHED  um estado terminal. Isso quer dizer que uma vez
      // nele, o comando no pode mais mudar de estado.
      if (this.status == CommandStatus.FINISHED) {
        return false;
      }
      this.status = status;
      this.finalizationInfo = info;
      return true;
    }
    finally {
      statusLock.unlock();
    }
  }

  /**
   * Define qual o SGA que foi selecionado pelo usurio.
   *
   * @param selectedSGAName O sga selecionado.
   */
  private void setSelectedSGAName(String selectedSGAName) {
    setSelectedSGAsNames(Arrays.asList(selectedSGAName));
  }

  /**
   * Obtm a plataforma na qual o comando dever ser executado.
   *
   * @return plataforma .
   */
  public String getPlatformFilter() {
    return platformFilter;
  }

  /**
   * Obtm o mapa de propriedades extras
   *
   * @return o mapa de propriedades extras
   */
  public Map<String, String> getExtraPropertiesMap() {
    return extraPropertiesMap;
  }

  /**
   * Indica se a seleo de servidores de algoritmos dever ser feita pelo
   * escalonador.
   *
   * @return true se a seleo dever ser feita pelo escalonador e false caso
   *         contrrio.
   */
  public boolean isAutomatic() {
    return isAutomatic;
  }

  /**
   * Obtm uma dica para auxiliar a identificao do algoritmo do comando.
   *
   * @return a dica do comando. (Pode ser {@code null}.)
   */
  public String getTip() {
    return tip;
  }

  /**
   * Define a descrio do comando.
   *
   * @param description a descrio do comando.
   */
  public void setDescription(String description) {
    this.description = description;
  }

  /**
   * Define a plataforma na qual o comando dever ser executado.
   *
   * @param platformFilter a plataforma na qual o comando deve ser executado.
   */
  public void setPlatformFilter(String platformFilter) {
    this.platformFilter = platformFilter;
  }

  /**
   * Obtm o progresso geral do comando. Em caso de comandos simples, retorna as
   * informaes indicadas diretamente pelo comando. Caso o comando seja um
   * fluxo, todos os ns envolvidos precisam informar informaes de progresso
   * seno o progresso geral no  computado e o mtodo retorna nulo.
   *
   * O progresso, quando disponvel,  provido em dois formatos: valor texto
   * original e tambm o valor numrico correspondente, entre 0 e 100%.
   *
   * @return O progresso do comando em dois formatos: valor texto original e
   *         tambm o valor numrico correspondente, entre 0 e 100%. <br>
   *         Ex: { "1/10", 10 } <br>
   *         <b>Se no houver informaes sobre progresso, esse mtodo retorna
   *         null.</b>
   */
  public ProgressData getProgressData() {
    switch (status) {
      case EXECUTING:
        return getExecutingCommandProgress();
      case FINISHED:
      case SYSTEM_FAILURE:
      case DOWNLOADING:
        /*
         * O valor  passado como string para representar o estado terminal e
         * no o valor numrico 100/100. Poderia ser usada qualquer outra
         * string, mas dessa forma fica independente da internacionalizao.
         */
        return new ProgressData("100%");
      case INIT:
      case SCHEDULED:
      case UPLOADING:
        /*
         * O valor  passado como string para representar o estado inicial e no
         * o valor numrico 0/100. Poderia ser usada qualquer outra string, mas
         * dessa forma fica independente da internacionalizao.
         */
        return new ProgressData("0%");
      default:
        return null;
    }
  }

  /**
   * Obtm o valor do progresso do comando em execuo.
   *
   * @return o valor do progresso.
   */
  public ProgressData getExecutingCommandProgress() {
    /*
     * Se os dados obtidos na ltima atualizao do comando ainda estiver
     * vlido, no precisa computar novamente.
     */
    if (cachedProgressData != null) {
      return cachedProgressData;
    }

    /*
     * Se ainda no foram lidos os dados do comando  porque a execuo acabou
     * de comear, podemos assumir o progresso de 0%.
     */
    if (specificData == null) {
      return new ProgressData(0d);
    }

    /*
     * Se os dados do comando foram lidos, mas ainda no h registro de
     * progresso,  porque o comando no registra essa informao. Consideramos
     * que o progresso  de 0% (e deve permanecer dessa forma enquanto o comando
     * no termina).
     */
    ProgressDataParser progressDataParser = new ProgressDataParser();
    if (!progressDataParser.hasProgressData(specificData)) {
      return new ProgressData(0d);
    }

    /*
     * No caso de comandos do tipo fluxo, so esperados mltiplos valores de
     * progresso (um para cada n). Na execuo simples teremos um nico valor.
     */
    int expectedValues;
    ConfiguratorType configuratorType = configurator.getConfiguratorType();
    if (configuratorType.equals(ConfiguratorType.FLOW)) {
      FlowAlgorithmConfigurator flowConfigurator =
        (FlowAlgorithmConfigurator) configurator;
      expectedValues = flowConfigurator.getNodes().size();
    }
    else {
      expectedValues = 1;
    }

    ProgressData data =
      progressDataParser.extractOverallProgressData(specificData,
        expectedValues);
    if (data != null) {
      cachedProgressData = data;
    }

    return data;
  }

  /**
   * Retorna um mapa de dados de progresso por chave numrica. Utilizado para
   * obter os progressos individuais de ns em um fluxo. Caso seja um comando
   * simples, o mapa  retornado vazio.
   *
   * @return Os dados de progresso mapeados pela chave numrica que os
   *         identificam.
   */
  public Map<Integer, ProgressData> getProgressDataMap() {
    ProgressDataParser progressDataParser = new ProgressDataParser();
    return progressDataParser.extractProgressDataMap(specificData);
  }

  /**
   * Obtm o caminho para o diretrio onde o comando foi persistido.
   *
   * @return O caminho para o diretrio de persistncia do comando.
   */
  public String[] getPersistencyPath() {
    return persistencyPath;
  }

  /**
   * Atribui o caminho para o diretrio onde o comando ser persistido.
   *
   * @param persistencyPath O caminho para o diretrio de persistncia do
   *        comando.
   */
  public void setPersistencyPath(String[] persistencyPath) {
    this.persistencyPath = persistencyPath;
  }

  /**
   * Retorna
   *
   * @return finalizationInfo
   */
  public CommandFinalizationInfo getFinalizationInfo() {
    return finalizationInfo;
  }

  /**
   * Obtm o nmero de tentativas de submisso do comando.
   *
   * @return o nmero de tentativas de submisso
   */
  public synchronized Integer getSubmissionAttempts() {
    return submissionAttempts;
  }

  /**
   * Incrementa o nmero de tentativas de submisso.
   */
  public synchronized void incSubmissionAttempts() {
    this.submissionAttempts = submissionAttempts + 1;
  }

  /**
   * A deserializao tentar usar uma instncia que j esteja na cache. WA para
   * problema #4601. Somente deve ser usado no servidor.
   *
   * @return a instncia
   */
  public Object readResolve() {
    return CommandInfoCache.getInstance().getCommandInfo(this);
  }
  
  /**
   * @return Nome da mquina que solicitou a submisso do algoritmo.
   */
  public String getClientHostName() {
    return clientHostName;
  }

  public String getPartialLog() { return partialLog; }

  public void setPartialLog(String partialLog) { this.partialLog = partialLog; }
}
